package org.gcube.common.homelibrary.jcr.workspace.trash;

import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;

import javax.jcr.Node;
import javax.jcr.RepositoryException;
import javax.jcr.Session;
import javax.jcr.query.QueryManager;
import javax.jcr.query.QueryResult;

import org.gcube.common.homelibrary.home.exceptions.InternalErrorException;
import org.gcube.common.homelibrary.home.workspace.WorkspaceItem;
import org.gcube.common.homelibrary.home.workspace.WorkspaceItemType;
import org.gcube.common.homelibrary.home.workspace.exceptions.InsufficientPrivilegesException;
import org.gcube.common.homelibrary.home.workspace.exceptions.ItemAlreadyExistException;
import org.gcube.common.homelibrary.home.workspace.exceptions.ItemNotFoundException;
import org.gcube.common.homelibrary.home.workspace.exceptions.WorkspaceFolderNotFoundException;
import org.gcube.common.homelibrary.home.workspace.exceptions.WrongDestinationException;
import org.gcube.common.homelibrary.home.workspace.trash.WorkspaceTrashItem;
import org.gcube.common.homelibrary.jcr.repository.JCRRepository;
import org.gcube.common.homelibrary.jcr.repository.external.GCUBEStorage;
import org.gcube.common.homelibrary.jcr.workspace.JCRWorkspace;
import org.gcube.common.homelibrary.jcr.workspace.JCRWorkspaceFolder;
import org.gcube.contentmanagement.blobstorage.transport.backend.RemoteBackendException;

public class JCRWorkspaceTrashItem extends JCRWorkspaceFolder implements WorkspaceTrashItem {

	protected static final String CONTENT				= "jcr:content";
	protected static final String DELETE_DATE			= "hl:deletedTime";
	protected static final String DELETE_USER			= "hl:deletedBy";
	protected static final String ORIGINAL_PARENT_ID	= "hl:originalParentId";
	protected static final String ORIGINAL_PATH			= "hl:deletedFrom";
	protected static final String NAME					= "hl:name";
	protected static final String MIME_TYPE				= "hl:mimeType";
	protected static final String LENGTH				= "hl:length";
	protected static final String IS_FOLDER				= "hl:isFolder";


	private final JCRWorkspace workspace;

	private String identifier;
	private Calendar deletedTime;
	//	private String name;
	private String originalParentId;
	private String deletedBy;
	private String mimeType;
	private long length;
	private boolean isFolder;
	private String deletedFrom;
	//	private String trashId;




	public JCRWorkspaceTrashItem(JCRWorkspace workspace, Node node) throws RepositoryException {
		super(workspace, node);

		this.identifier = node.getIdentifier();
		this.workspace = workspace;
		//		this.trashId = workspace.trashPath;
		try{
			this.deletedTime = node.getProperty(DELETE_DATE).getDate();
			this.deletedBy = node.getProperty(DELETE_USER).getString();
			this.deletedFrom = node.getProperty(ORIGINAL_PATH).getString();
		}catch (Exception e) {
			// TODO: handle exception
		}

		this.originalParentId = node.getProperty(ORIGINAL_PARENT_ID).getString();
		this.isFolder = node.getProperty(IS_FOLDER).getBoolean();

		try{
			this.mimeType = node.getProperty(MIME_TYPE).getString();
		}catch (Exception e) {}
		try{
			this.length = node.getProperty(LENGTH).getLong();
		}catch (Exception e) {}

	}



	public JCRWorkspaceTrashItem(JCRWorkspace workspace, Node node,
			String name, String description, Calendar date, String portalLogin,
			String originalParentId, String mimeType, long length, boolean isFolder, String originalPath) throws RepositoryException {
		super(workspace, node, name, description);
		this.identifier = node.getIdentifier();
		this.workspace = workspace;
		//		this.trashId = workspace.getTrashId();
		this.deletedBy = portalLogin;
		this.deletedTime = date;
		this.originalParentId = originalParentId;
		this.deletedFrom = originalPath;

		this.mimeType = mimeType;
		this.length = length;
		this.isFolder = isFolder;

		node.setProperty(NAME, name);
		node.setProperty(DELETE_DATE,Calendar.getInstance());
		node.setProperty(DELETE_USER, portalLogin);
		node.setProperty(ORIGINAL_PARENT_ID, originalParentId);
		node.setProperty(ORIGINAL_PATH, originalPath);

		node.setProperty(MIME_TYPE, mimeType);
		node.setProperty(LENGTH,length);
		node.setProperty(IS_FOLDER,isFolder);


	}

	@Override
	public void restore() throws InternalErrorException {
		Session session = JCRRepository.getSession();		
		Node node = null;
		try {
			//			System.out.println("getName() "+ getName());
			node = session.getNodeByIdentifier(identifier).getNode(getName());
			//			node = session.getNodeByIdentifier(identifier);
			//			System.out.println("node " + node.getPath());
			Node originalParentNode = null;
			try{
				originalParentNode = session.getNodeByIdentifier(originalParentId);

			}catch (Exception e) {
				logger.warn("The original Parent Node does not exist anymore, the item will be restored to the root");
			}


			Node parentNode = null;

			if (originalParentNode != null) {
				boolean inTrash = workspace.isInTrash(originalParentNode);
				logger.info(originalParentNode.getPath() + " is In Trash? " + inTrash);

				if (inTrash){
					parentNode = session.getNodeByIdentifier(workspace.getRoot().getId());
					logger.info("The file " + getName()  + " will be restored in the root");
				}
				else{
					parentNode = originalParentNode;
					logger.info("The file " + getName()  + " will be restored to the original parent");
				}
			}

			//if a file with the same name already exists to the destination path, 
			//the item will be restored using time stamp + original name
			//es. "2012_04_05_11400029_MyAwesomeFile.txt"
			if	(workspace.exists(getName(), parentNode)){

				logger.info(getName() + " already exists in " + parentNode.getPath());
				try{
					String time = timeStampRename();
					setName(node, time + getName());
				}catch (Exception e1) {
					throw new InternalErrorException(e1);
				}
			}

			try {
				workspace.moveItem(node.getIdentifier(), parentNode.getIdentifier());
			} catch (ItemNotFoundException | WrongDestinationException
					| InsufficientPrivilegesException
					| ItemAlreadyExistException
					| WorkspaceFolderNotFoundException e) {
				throw new InternalErrorException(e);
			}


			//remove trash item (just the folder)
			session.getNodeByIdentifier(identifier).remove();
			session.save();

		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} catch (RemoteBackendException e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}

	}


	/**
	 * Get Time Stamp to rename a file
	 * @return
	 */
	private static String timeStampRename() {
		//		"2012_04_05_11400029_MyAwesomeFile.txt"
		Date date = new Date();
		SimpleDateFormat sdf = new SimpleDateFormat("yyyy_MM_dd_HHmmssss_");
		String formattedDate = sdf.format(date);
		//		System.out.println(formattedDate);
		return formattedDate;

	}



	@Override
	public void deletePermanently() throws InternalErrorException {

		Session session = null;
		Node node = null;
		try {
			session = JCRRepository.getSession();
			node = session.getNodeByIdentifier(identifier);
			WorkspaceItem item = workspace.getWorkspaceItem(node);
			logger.debug("deletePermanently node: " + item.getName());

			//remove the content of the trash item
			try{
				GCUBEStorage.removeRemoteFolder(node.getPath(), workspace.getOwner().getPortalLogin());
				//				removeRemoteContent(node.getNode(item.getName()), node.getNode(item.getName()).getType());
			}catch (Exception e) {
				logger.warn("Error removing " + node.getPath() + " from storage", e);
			}

			//remove the trash item
			node.remove();
			session.save();
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}
	}


	@Override
	public WorkspaceItemType getType() {
		return WorkspaceItemType.TRASH_ITEM;
	}



	public long getLength() throws InternalErrorException {
		return length;
	}

	@Override
	public String getMimeType() throws InternalErrorException {
		return mimeType;
	}


	@Override
	public String getDeletedFrom() {
		return deletedFrom;
	}


	//getter methods
	@Override
	public String getOriginalParentId() {
		return originalParentId;
	}

	@Override
	public Calendar getDeletedTime() {
		return deletedTime;
	}

	@Override
	public String getDeletedBy() {
		return deletedBy;
	}

	@Override
	public boolean isFolder() {
		return isFolder;
	}


	@Override
	public String getName() throws InternalErrorException {	

		String name = null;
		Session session = null;
		Node node = null;
		try {
			session = JCRRepository.getSession();
			node = session.getNodeByIdentifier(identifier);
			name =	node.getProperty(NAME).getString();
		} catch (RepositoryException e) {
			//			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}

		return name;
	}


	public void setName(Node node, String name) throws RepositoryException {

		try {
			String newPath = node.getParent().getPath() 
					+ workspace.getPathSeparator() + name;
			//			System.out.println("set title " + name);
			node.setProperty(TITLE, name);
			node.getSession().save();
			node.getSession().getWorkspace().move(node.getPath(), newPath);
			node.getSession().save();
		} catch (Exception e) {
			logger.error("Error setting a new name to WorkspaceTrashItem: " + e);
		} 

	}



	@Override
	public void removeChild(WorkspaceItem child) throws InternalErrorException,
	InsufficientPrivilegesException {}




}
