package org.gcube.common.homelibrary.jcr.workspace;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.jcr.ItemExistsException;
import javax.jcr.RepositoryException;

import org.apache.commons.httpclient.HttpException;
import org.apache.jackrabbit.util.Text;
import org.gcube.common.homelibary.model.items.ItemDelegate;
import org.gcube.common.homelibary.model.items.type.NodeProperty;
import org.gcube.common.homelibary.model.items.type.WorkspaceItemType;
import org.gcube.common.homelibary.model.util.WorkspaceItemAction;
import org.gcube.common.homelibrary.home.Home;
import org.gcube.common.homelibrary.home.HomeLibrary;
import org.gcube.common.homelibrary.home.HomeManager;
import org.gcube.common.homelibrary.home.exceptions.InternalErrorException;
import org.gcube.common.homelibrary.home.workspace.WorkspaceFolder;
import org.gcube.common.homelibrary.home.workspace.WorkspaceSharedFolder;
import org.gcube.common.homelibrary.home.workspace.accessmanager.ACLType;
import org.gcube.common.homelibrary.home.workspace.exceptions.InsufficientPrivilegesException;
import org.gcube.common.homelibrary.home.workspace.exceptions.ItemAlreadyExistException;
import org.gcube.common.homelibrary.home.workspace.exceptions.ItemNotFoundException;
import org.gcube.common.homelibrary.home.workspace.exceptions.WorkspaceFolderNotFoundException;
import org.gcube.common.homelibrary.home.workspace.exceptions.WrongDestinationException;
import org.gcube.common.homelibrary.home.workspace.exceptions.WrongItemTypeException;
import org.gcube.common.homelibrary.home.workspace.usermanager.GCubeGroup;
import org.gcube.common.homelibrary.home.workspace.usermanager.UserManager;
import org.gcube.common.homelibrary.jcr.repository.JCRRepository;
import org.gcube.common.homelibrary.jcr.workspace.accessmanager.JCRAccessManager;
import org.gcube.common.homelibrary.jcr.workspace.accessmanager.JCRPrivilegesInfo;
import org.gcube.common.homelibrary.jcr.workspace.servlet.JCRServlets;
import org.gcube.common.homelibrary.jcr.workspace.servlet.wrapper.DelegateManager;
import org.gcube.common.homelibrary.jcr.workspace.usermanager.JCRUserManager;
import org.gcube.common.homelibrary.util.WorkspaceUtil;
import org.gcube.contentmanagement.blobstorage.transport.backend.RemoteBackendException;

import com.thoughtworks.xstream.XStream;


public class JCRWorkspaceSharedFolder extends JCRAbstractWorkspaceFolder implements WorkspaceSharedFolder {

	private String applicationName;
	private String destinationFolderId;
	private String itemName;
	private List<String> users;

	public JCRWorkspaceSharedFolder(JCRWorkspace workspace, ItemDelegate delegate) throws RepositoryException, InternalErrorException {
		super(workspace,delegate);	
	}

	public JCRWorkspaceSharedFolder(JCRWorkspace workspace, ItemDelegate delegate,
			String name, String description, String originalDestinationFolderId, List<String> users, String applicationName, String itemName) throws RepositoryException, InternalErrorException {		

		super(workspace,delegate,name,description);

		this.destinationFolderId = originalDestinationFolderId;
		this.applicationName = applicationName;
		this.itemName = itemName;
		this.users = users;



	}



	private void shareWithUses(List<String> users) throws RepositoryException {
		try {
			// The save method creates a clone of the sharable node
			//for the owner and all users below user roots.
			addUser(workspace.getOwner().getPortalLogin(), destinationFolderId);

			logger.trace("Share with " + users.toString());
			for (String user : users) {

				HomeManager homeManager = workspace.getHome().getHomeManager();
				Home home = homeManager.getHome(user);

				if (applicationName==null){
					if (isVreFolder())
						addUser(user, home.getWorkspace().getMySpecialFolders().getId());
					else
						addUser(user, home.getWorkspace().getRoot().getId());
				}
				else
					addUser(user, home.getDataArea().getApplicationRoot(applicationName).getId());
			}
		} catch (Exception e) {
			throw new RepositoryException(e);
		}

	}

	public JCRWorkspaceSharedFolder(JCRWorkspace workspace, ItemDelegate delegate,
			String name, String description, String originalDestinationFolderId, List<String> users, String applicationName, String itemName, String displayName, boolean isVreFolder) throws RepositoryException, InternalErrorException {		

		this(workspace,delegate,name,description, originalDestinationFolderId, users, applicationName, itemName);

		delegate.getProperties().put(NodeProperty.IS_VRE_FOLDER, new XStream().toXML(isVreFolder));
		delegate.getProperties().put(NodeProperty.DISPLAY_NAME, displayName);

		//		DelegateManager wrap = new DelegateManager(delegate, workspace.getOwner().getPortalLogin());
		//		try {
		//			ItemDelegate saved = wrap.save();
		//			System.out.println("saved " + saved.toString());
		//		} catch (Exception e) {
		//			// TODO Auto-generated catch block
		//			e.printStackTrace();
		//		}


	}


	/**
	 * Resolve groupIds and add user/group to member Node
	 * @param usersList
	 * @return a list of users (no group ids)
	 * @throws InternalErrorException
	 */
	@SuppressWarnings("unchecked")
	private List<String> listUsers(List<String> usersList) throws InternalErrorException {

		List<String> groups = new ArrayList<String>();
		List<String> users = new ArrayList<String>();

		//get a list of groups
		JCRUserManager userManager = new JCRUserManager();
		List<GCubeGroup> groupsList = userManager.getGroups();		
		for (GCubeGroup group : groupsList){
			groups.add(group.getName());
		}


		List<String> memberIds = null;

		try {	
			for (String user: usersList){	

				//add users to members node
				//				 ItemDelegate sharedNode = servlets.getItemById(getId());

				memberIds = (List<String>) new XStream().fromXML(delegate.getProperties().get(NodeProperty.MEMBERS));
				if (memberIds.size()<1){
					memberIds = new ArrayList<String>();			
				}

				if (!memberIds.contains(user)){				
					if(!user.endsWith("-Manager")){
						memberIds.add(user);
						logger.warn(user + " add to membersList");
					}
				}

				//resolve groups
				if (groups.contains(user)) {
					logger.info("User " + user + " is a Group, resolve group id");
					//if a user is a group, resolve group
					List<String> userList = workspace.resolveGroupId(user);
					users.addAll(userList);
				}else
					users.add(user);
			}

			//check here
			delegate.getProperties().put(NodeProperty.MEMBERS, new XStream().toXML(memberIds));

			DelegateManager manager = new DelegateManager(delegate, workspace.getOwner().getPortalLogin());
			manager.save();

		} catch (Exception e) {
			throw new InternalErrorException(e);
		} 
		return users;
	}


	@Override
	public ItemDelegate save() throws RepositoryException {

		return super.save();

	}


	public ItemDelegate getUserNode(ItemDelegate delegate, String user) throws RepositoryException,
	InternalErrorException, ItemNotFoundException {

		@SuppressWarnings("unchecked")
		Map<String, String> usersNode = (Map<String, String>) new XStream().fromXML(delegate.getProperties().get(NodeProperty.USERS));
		//		logger.trace("Looking for user: " + user + " in node: " + usersNode.getPath());
		String value = usersNode.get(user);
		//		logger.info("value "+  value + " for delegate " + delegate.getPath());

		String[] values = value.split(workspace.getPathSeparator());
		if (values.length < 2)
			throw new InternalErrorException("Path node corrupt");

		String parentId = values[0];
		String nodeName = values[1];
		ItemDelegate parentNode = JCRRepository.getServlets().getItemById(parentId);

		return JCRRepository.getServlets().getItemByPath(parentNode.getPath() + 
				workspace.getPathSeparator() + Text.escapeIllegalJcrChars((nodeName)), workspace.getOwner().getPortalLogin());

	}



	private ItemDelegate getUserNode(ItemDelegate delegate) throws RepositoryException,
	InternalErrorException, ItemNotFoundException {
		return getUserNode(delegate, workspace.getOwner().getPortalLogin());
	}

	private String getNodeName(ItemDelegate node) throws RepositoryException,
	InternalErrorException {

		String[] names = node.getPath().split(
				workspace.getPathSeparator());

		return names[names.length - 1];
	}

	@Override
	public String getName() throws InternalErrorException {

		//		try {
		//			ItemDelegate userNode = getUserNode(delegate);
		//			return getNodeName(userNode);
		//		} catch (RepositoryException | ItemNotFoundException e) {
		//			throw new InternalErrorException(e);
		//		} 		
		return delegate.getTitle();
	}

	@Override
	public void internalRename(String newName, String remotePath) throws ItemAlreadyExistException, InternalErrorException {


		String nodeNewName = Text.escapeIllegalJcrChars(newName);
		try {

			ItemDelegate userNode = getUserNode(delegate);

			if (workspace.exists(nodeNewName, userNode.getParentId())) {
				logger.error("Item with name " + nodeNewName + " exists");
				throw new ItemAlreadyExistException("Item " + nodeNewName + " already exists");
			}
			ItemDelegate parent = JCRRepository.getServlets().getItemById(userNode.getParentId());
			String newPath = parent.getPath() 
					+ workspace.getPathSeparator() + nodeNewName;

			delegate.setLastModificationTime(Calendar.getInstance());
			delegate.setLastModifiedBy(workspace.getOwner().getPortalLogin());
			delegate.setLastAction(WorkspaceItemAction.RENAMED);
			try{
				delegate.getContent().put(NodeProperty.REMOTE_STORAGE_PATH, remotePath);
			}catch (Exception e) {
				logger.info("RemotePath not in " + delegate.getPath());
			}

			//there was a node.save(); here
			//			ItemDelegate newDelegate = null;
			String path = userNode.getPath();
			try {
				delegate = JCRRepository.getServlets().move(path, newPath);

			} catch (HttpException e1) {
				throw new InternalErrorException(e1);
			} catch (IOException e1) {
				throw new InternalErrorException(e1);
			}

			Map<NodeProperty, String> properties = delegate.getProperties();
			@SuppressWarnings("unchecked")
			Map<String, String> usersNode = (Map<String, String>) new XStream().fromXML(properties.get(NodeProperty.USERS));
			String value = userNode.getParentId() + workspace.getPathSeparator() + newName;

			usersNode.put(workspace.getOwner().getPortalLogin(), value);

			properties.put(NodeProperty.USERS, new XStream().toXML(usersNode));

			DelegateManager wrap = null;
			try {
				wrap = new DelegateManager(delegate, workspace.getOwner().getPortalLogin());
				wrap.save();
			} catch (Exception e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
			}


		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} catch (ItemNotFoundException e) {
			throw new InternalErrorException(e);
		} catch (WrongItemTypeException e) {
			throw new InternalErrorException(e);
		} finally {

		}

	}

	@Override
	public void internalMove(ItemDelegate destinationFolderNode, String remotePath) throws ItemAlreadyExistException,
	InternalErrorException, RepositoryException {

		try {

			logger.debug("Start internal move item with id " 
					+ getId() + " to destination item with id " + destinationFolderNode.getId());


			ItemDelegate itemDelegate = JCRRepository.getServlets().getItemById(getId());

			if (workspace.exists(itemDelegate.getTitle(), destinationFolderNode.getId())) {
				logger.error("Item with name " + getName() + " exists");
				throw new ItemAlreadyExistException("Item " + itemDelegate.getTitle() + " already exists");
			}

			itemDelegate.setLastModificationTime(Calendar.getInstance());
			itemDelegate.setLastModifiedBy(workspace.getOwner().getPortalLogin());
			itemDelegate.setLastAction(WorkspaceItemAction.MOVED);

			//	System.out.println("internalMove:getUserNode ");
			ItemDelegate userNode = getUserNode(itemDelegate);
			String userNodeName = getNodeName(userNode);

			String newPath = destinationFolderNode.getPath() 
					+ workspace.getPathSeparator() + userNodeName;

			String value = destinationFolderNode.getId() +
					workspace.getPathSeparator() + userNodeName;

			try {
				JCRRepository.getServlets().clone(itemDelegate.getPath(), newPath, false);
			} catch (HttpException e1) {
				e1.printStackTrace();
			} catch (IOException e1) {
				e1.printStackTrace();
			}


			@SuppressWarnings("unchecked")
			Map<String, String> usersNode = (Map<String, String>) new XStream().fromXML(itemDelegate.getContent().get(NodeProperty.USERS));
			usersNode.put(workspace.getOwner().getPortalLogin(), value);


			JCRRepository.getServlets().removeItem(userNode.getPath());

			DelegateManager wrap = new DelegateManager(itemDelegate, workspace.getOwner().getPortalLogin());
			wrap.save();



		} catch (IOException e1) {
			e1.printStackTrace();	
		} catch (ItemExistsException e) {
			throw new ItemAlreadyExistException(e.getMessage());
		} catch (RepositoryException e) {
			logger.error("Repository exception thrown by move operation",e);
			throw new InternalErrorException(e);
		} catch (WrongItemTypeException e) {
			logger.error("Unhandled Exception ");
			throw new InternalErrorException(e);
		} catch (ItemNotFoundException e) {
			logger.error("Unhandled Exception ");
			throw new InternalErrorException(e);
		} catch (Exception e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
	}


	public ItemDelegate unShareNode(ItemDelegate sharedNode, String destinationNodeId, boolean localCopy) throws Exception {

		//		System.out.println("localCopy? " + localCopy);
		logger.debug("unShare Node: "+ sharedNode.getPath() + " -  by user: " + workspace.getOwner().getPortalLogin());

		ItemDelegate userNode = getUserNode(sharedNode);

		String folderName = getNodeName(userNode);
		String description = getDescription();

		// shareNode parent it's the same of destinationNode
		if (destinationNodeId.equals(userNode.getParentId())) {
			removeUserSharedFolder(sharedNode);
		}

		if (localCopy){
			String unSharedFolderId = workspace.createFolder(folderName, description, destinationNodeId).getId();
			ItemDelegate nodeFolder = JCRRepository.getServlets().getItemById(unSharedFolderId);

			DelegateManager wrap = new DelegateManager(sharedNode, workspace.getOwner().getPortalLogin());
			List<ItemDelegate> children = wrap.getNodes();
			//			for (NodeIterator iterator = sharedNode.getNodes(); iterator.hasNext();) {
			for (ItemDelegate child: children){
				//				Node child = (Node) iterator.next();

				if (!child.getName().startsWith(JCRRepository.HL_NAMESPACE) 
						&& !child.getName().startsWith(JCRRepository.JCR_NAMESPACE)
						&& !child.getName().startsWith(JCRRepository.REP_NAMESPACE)) {			
					try {
						JCRRepository.getServlets().copy(child.getPath(), nodeFolder.getPath() 
								+ workspace.getPathSeparator() + child.getName());
					} catch (HttpException e) {
						// TODO Auto-generated catch block
						e.printStackTrace();
					} catch (IOException e) {
						// TODO Auto-generated catch block
						e.printStackTrace();
					}
				}
			}

			//		session.save();
			//TODO temporarily 

			ItemDelegate destinationNode = JCRRepository.getServlets().getItemById(destinationNodeId);
			logger.debug("copyremotecontent from "+  nodeFolder.getPath() + " to parent id " + destinationNode.getPath());
			workspace.copyRemoteContent(nodeFolder, destinationNode);

			//TODO 
			//			session.save();		 WHERE????

			JCRWorkspaceItem itemUnshared = (JCRWorkspaceItem) workspace.getItem(unSharedFolderId);
			//add UNSHARE operation in History
			itemUnshared.setUnshareHistory(workspace.getOwner().getPortalLogin());
			//change owner
			itemUnshared.setOwnerToCurrentUser(itemUnshared);
			// Check if it's possible remove shared node completely
			//			checkRemoveSharedFolder(sharedNode);

			return nodeFolder;
		}
		return null;
	}



	@Override
	public ItemDelegate internalCopy(ItemDelegate delegateFolder, String newName) throws InternalErrorException,
	ItemAlreadyExistException, WrongDestinationException, RepositoryException{

		JCRServlets servlets = JCRRepository.getServlets();

		String pathNewNode = delegateFolder.getPath()
				+ workspace.getPathSeparator() + Text.escapeIllegalJcrChars(newName);

		try {

			if(servlets.getItemByPath(pathNewNode, workspace.getOwner().getPortalLogin()) != null)
				throw new ItemAlreadyExistException(newName + " already exist");

		} catch(Exception e) {}


		String description = getDescription();
		try {

			String unSharedFolderId = workspace.createFolder(Text.escapeIllegalJcrChars(newName), description, delegateFolder.getId()).getId();
			ItemDelegate newNodeFolder = servlets.getItemById(unSharedFolderId);
			DelegateManager wrap = new DelegateManager(newNodeFolder, workspace.getOwner().getPortalLogin());
			List<ItemDelegate> children = wrap.getNodes();

			for (ItemDelegate child: children){
				if (!child.getName().startsWith(JCRRepository.HL_NAMESPACE) 
						&& !child.getName().startsWith(JCRRepository.JCR_NAMESPACE)
						&& !child.getName().startsWith(JCRRepository.REP_NAMESPACE)) {				

					servlets.copy(child.getPath(), newNodeFolder.getPath() 
							+ workspace.getPathSeparator() + child.getName());
				}
			}

			return newNodeFolder;
		} catch (Exception e) {
			throw new InternalErrorException(e);
		} 
	}

	@Override
	public JCRAbstractWorkspaceFolder getParent() throws InternalErrorException {

		try {
			return workspace.getParent(getUserNode(delegate));
		} catch (RepositoryException | ItemNotFoundException e) {
			throw new InternalErrorException(e);
		} 
	}

	//	@Override
	//	public String getPath() throws InternalErrorException {
	////		String path = null;
	////		ItemDelegate userNode = null;
	////		try {
	////			userNode = getUserNode(delegate);
	//////			path = ((JCRWorkspaceFolder)getParent(userNode)).getPath()
	//////					+ workspace.getPathSeparator() + getNodeName(userNode);
	////		} catch (ItemNotFoundException | RepositoryException e) {
	////			throw new InternalErrorException(e);
	////		}
	////				return ((JCRWorkspaceFolder)getParent(userNode)).getPath(JCRRepository.getServlets().getItemById(userNode.getParentId()))
	////						+ workspace.getPathSeparator() + getNodeName(userNode);		
	////		System.out.println("SHARED NODE path: " + path); 
	////	return path;	
	//		
	//		String delPath = null;
	//		if (delegate.getParentPath().startsWith("/Home/" + workspace.getOwner().getPortalLogin())){
	//			delPath = delegate.getParentPath().replace("/Home/" + workspace.getOwner().getPortalLogin(), "");
	//		}
	//		return delPath;
	//		System.out.println("SHARED NODE: PARENTPATH " + delegate.getParentPath());
	//		return delegate.getParentPath();
	//	}


	@Override
	public void remove() throws InternalErrorException, InsufficientPrivilegesException {

		logger.debug("Remove shared item");
		logger.debug("portalLogin: " + workspace.getOwner().getPortalLogin() + " - owner: "+ getOwner());

		try {


			if (!JCRPrivilegesInfo.canDelete(getOwner().getPortalLogin(), workspace.getOwner().getPortalLogin(), getAbsolutePath(), true)) 
				throw new InsufficientPrivilegesException("Insufficient Privileges to remove the node");
			if (isVreFolder())
				throw new InternalErrorException("A VRE folder cannot be removed");
			if (delegate.getPath().equals(workspace.mySpecialFoldersPath))
				throw new InternalErrorException("This folder cannot be removed");

			try {

				//	System.out.println("trash id: " + trashFolderId);
				WorkspaceFolder unsharedFolder = unShare();

				logger.trace("unsharedFolder: " + unsharedFolder.getPath());
				ItemDelegate usharedNode = JCRRepository.getServlets().getItemById(unsharedFolder.getId());

				workspace.moveToTrash(usharedNode);

				//				logger.trace("remove clones");
				//				removeClones(sharedNode);

				//				logger.trace("remove sharedNode: " + sharedNode.getPath());
				//				sharedNode.remove();
				//				session.save();

			} catch (WrongDestinationException | ItemAlreadyExistException
					| WorkspaceFolderNotFoundException | ItemNotFoundException e) {
				throw new InternalErrorException(e);
			}


			//			removeUserSharedFolder(sharedNode);
			//			checkRemoveSharedFolder(sharedNode);
			//			System.out.println(getSharePath() + " is shared and can be deleted by " + JCRRepository.portalLogin);
			//			System.out.println("remove");

		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} catch (RemoteBackendException e) {
			throw new InternalErrorException(e);
		} 
	}
	@Override
	public String getPath(ItemDelegate delegate) throws InternalErrorException, RepositoryException {

		ItemDelegate userNode = null;
		try {
			userNode = getUserNode(delegate);
		} catch (ItemNotFoundException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
		return ((JCRWorkspaceFolder)getParent(userNode)).getPath(JCRRepository.getServlets().getItemById(userNode.getParentId()))
				+ workspace.getPathSeparator() + getNodeName(userNode);		
	}


	@SuppressWarnings("unchecked")
	@Override
	public List<String> getMembers() throws InternalErrorException {

		List<String> list = null;

		try {
			list = (ArrayList<String>) new XStream().fromXML(delegate.getProperties().get(NodeProperty.MEMBERS));
		} catch (Exception e) {
			throw new InternalErrorException(e);
		} 
		return list;
	}


	@SuppressWarnings("unchecked")
	@Override
	public List<String> getUsers() throws InternalErrorException {

		List<String> list = new ArrayList<String>();
		try {
			Map<String, String> users = (Map<String, String>) new XStream().fromXML(delegate.getProperties().get(NodeProperty.USERS));
			Set<String> set = users.keySet();
			for (String key:set)
				list.add(key);
		} catch (Exception e) {
			throw new InternalErrorException(e);
		} 
		return list;
	}


	@SuppressWarnings("unchecked")
	private void addUser(String user, String destinationFolderId) throws InternalErrorException, RepositoryException {

		logger.trace("addUser("+delegate.getPath()+", " + user +", " + destinationFolderId +");");
		//	System.out.println("addUser("+delegate.getPath()+", " + user +", " + destinationFolderId +");");

		JCRServlets	servlets = JCRRepository.getServlets();
		try {

			HomeManager homeManager = workspace.getHome().getHomeManager();
			WorkspaceFolder userRoot = (WorkspaceFolder)homeManager.getHome(user).getWorkspace().getItem(destinationFolderId);

			ItemDelegate rootNode = servlets.getItemById(userRoot.getId());
			String sharedFolderName = userRoot.getUniqueName(delegate.getTitle(), false);

			Map<NodeProperty, String> properties = delegate.getProperties();
			Map<String, String> usersNode = null;

			try{
				usersNode = (Map<String, String>) new XStream().fromXML(properties.get(NodeProperty.USERS));
			}catch (Exception e) {
				logger.info("USERS not set on " + delegate.getPath());
				usersNode = new HashMap<String, String>();
			}
			//			System.out.println("AFTER usersNode: " + usersNode.toString());

			String pathUser = null;
			if (applicationName != null){

				pathUser = rootNode.getPath() + workspace.getPathSeparator() + sharedFolderName;
				logger.trace("clone from " + delegate.getPath()+ workspace.getPathSeparator() + itemName + " to "+ pathUser);
				ItemDelegate cloned = JCRRepository.getServlets().clone(delegate.getPath()+ workspace.getPathSeparator() + itemName, pathUser, false);
				logger.trace("CLONE " + cloned.getPath());

			}else {		
				pathUser = rootNode.getPath() + workspace.getPathSeparator() + sharedFolderName;
				try {
					if (usersNode.get(user) != null){
						return;
					}
				} catch (Exception e) {
					logger.debug("User is not present");
				}

				JCRRepository.getServlets().clone(delegate.getPath(), pathUser, false);
				logger.trace("Clone from " + delegate.getPath() + " to "+ pathUser);
			}

			String value = userRoot.getId() + workspace.getPathSeparator() 
					+ sharedFolderName;

			logger.trace("usersNode: " + delegate.getPath() + " - set value " + value + " to: " + user);

			usersNode.put(user, value);

			properties.put(NodeProperty.USERS, new XStream().toXML(usersNode));
			DelegateManager wrap = new DelegateManager(delegate, workspace.getOwner().getPortalLogin());
			wrap.save();

		} catch (Exception e) {
			throw new InternalErrorException(e);
		}

	}


	@Override
	public void addUser(String user) throws InsufficientPrivilegesException,
	InternalErrorException {

		JCRServlets servlets = JCRRepository.getServlets();
		try {	 
			ItemDelegate sharedNode = servlets.getItemById(getId());

			@SuppressWarnings("unchecked")
			Map<String, String> usersMap = (Map<String, String>) new XStream().fromXML(sharedNode.getProperties().get(NodeProperty.USERS));

			try {
				if (usersMap.get(user) != null){
					logger.trace(user + " is already in share");
					return;
				}
			} catch (Exception e) {
				logger.debug("User "+ user + " is not present");
			}

			HomeManager homeManager = workspace.getHome().getHomeManager();
			Home home = homeManager.getHome(user);

			if (isVreFolder())
				addUser(user, home.getWorkspace().getMySpecialFolders().getId());
			else
				addUser(user, home.getWorkspace().getRoot().getId());


		} catch (Exception e) {
			throw new InternalErrorException(e);
		} 
	}

	@Override
	public WorkspaceItemType getType() {
		return WorkspaceItemType.SHARED_FOLDER;
	}

	@Override
	public WorkspaceFolder unShare() throws InternalErrorException {

		try {
			ItemDelegate item = JCRRepository.getServlets().getItemById(getId());	
			ItemDelegate userNode = getUserNode(item);	

			ItemDelegate unsharedNode = null;
			JCRWorkspaceFolder folder = null;

			boolean flag = false;
			if (getOwner().equals(workspace.getOwner().getPortalLogin()) || (getACLUser().equals(ACLType.ADMINISTRATOR))){
				flag= true;
			}
			try{
				unsharedNode = unShareNode(item, userNode.getParentId(), flag);
			}catch (Exception e) {
				// TODO: handle exception
			}

			if (unsharedNode!=null){
				folder = new JCRWorkspaceFolder(workspace, unsharedNode);

				logger.trace("remove clones");
				removeClones(item);

				logger.trace("remove sharedNode: " + item.getPath());
				DelegateManager wrap = new DelegateManager(item, workspace.getOwner().getPortalLogin());
				wrap.remove();
				//				session.save();
			}else {

				//remove user from ACL
				JCRAccessManager accessManager = new JCRAccessManager();
				List<String> userToRemove = new ArrayList<String>();
				userToRemove.add(workspace.getOwner().getPortalLogin());
				accessManager.deleteAces(getAbsolutePath(), userToRemove);	
			}
			//create a local copy
			return folder;

		} catch (Exception e) {
			throw new InternalErrorException(e);
		} 

	}




	@SuppressWarnings("unchecked")
	@Override
	public WorkspaceFolder unShare(String user) throws InternalErrorException {
		JCRServlets servlets = JCRRepository.getServlets();
		try {
			//			ItemDelegate sharedDelegate = servlets.getItemById(getId());	


			if (!user.startsWith(JCRRepository.JCR_NAMESPACE) &&
					!user.startsWith(JCRRepository.HL_NAMESPACE) && 
					!user.equals(workspace.getOwner().getPortalLogin()))	{	
				//remove clone
				try{
					ItemDelegate cloneNode = getUserNode(delegate, user);
					logger.trace("remove clone " + cloneNode.getPath());
					servlets.removeItem(cloneNode.getPath());

				}catch (Exception e) {
					logger.error("Error removing clone ");
				}
				//remove user in userNode
				Map<String, String> userNode =  null;
				try{	
					userNode = (Map<String, String>) new XStream().fromXML(delegate.getProperties().get(NodeProperty.USERS));

					userNode.put(user, (String)null);
					delegate.getProperties().put(NodeProperty.USERS, new XStream().toXML(userNode));
					DelegateManager wrap = new DelegateManager(delegate, workspace.getOwner().getPortalLogin());
					wrap.save();

					logger.trace(user + "  deleted from share " + delegate.getPath());
				}catch (Exception e) {					
					logger.error("Error removing user from node Users ");
				}
			}

			//remove user from ACL
			JCRAccessManager accessManager = new JCRAccessManager();
			List<String> userToRemove = new ArrayList<String>();
			userToRemove.add(user);
			accessManager.deleteAces(getAbsolutePath(), userToRemove);	

			//set history			
			setUnshareHistory(user);
			//			return new JCRWorkspaceFolder(workspace, unsharedNode);
			return null;
		} catch (Exception e) {
			throw new InternalErrorException(e);
		}

	}


	@Override
	public WorkspaceSharedFolder share(List<String> usersList) throws InsufficientPrivilegesException,
	WrongDestinationException, InternalErrorException {

		List<String> userIds = listUsers(usersList);

		for (String user : userIds) {
			addUser(user);
		}

		return (WorkspaceSharedFolder) this;
	}



	@Override
	public String getName(String user) throws InternalErrorException {

		try {
			ItemDelegate userNode = getUserNode(delegate, user);
			return getNodeName(userNode);
		} catch (RepositoryException | ItemNotFoundException e) {
			throw new InternalErrorException(e);
		} 
	}




	@Override
	public ACLType getPrivilege() throws InternalErrorException {

		String absPath = null;
		JCRAccessManager accessManager = null;
		Map<String, List<String>> aclMap = null;		

		try{
			accessManager = new JCRAccessManager();
			absPath = getAbsolutePath();
			aclMap  = accessManager.getEACL(absPath);

			Set<String> keys = aclMap.keySet();

			for (final String user : keys){

				JCRUserManager um = new JCRUserManager();
				GCubeGroup group = null;
				try{

					//if the user is a group and this is empty, skip
					group = um.getGroup(user);
					if (group!=null){
						if (group.getMembers().isEmpty()){
							continue;	
						}
					}

					ACLType aclType = WorkspaceUtil.getACLTypeByKey(aclMap.get(user));

					if (!aclType.equals(ACLType.ADMINISTRATOR))
						return aclType;

				}catch (Exception e) {
					logger.error(e.getMessage());
				}
			} 

		}catch (Exception e) {
			logger.error("an error occurred setting ACL on: " + absPath);
		}
		return ACLType.WRITE_OWNER;

	}








	public void removeClones(ItemDelegate sharedNode) throws InternalErrorException, RepositoryException {

	JCRServlets servlets = JCRRepository.getServlets();
		try {
			Map<NodeProperty, String> properties = sharedNode.getProperties();
			@SuppressWarnings("unchecked")
			Map<String, String> userNode = (Map<String, String>) new XStream().fromXML(properties.get(NodeProperty.USERS));
			Set<String> usersList = userNode.keySet();
			for (String user: usersList){

				if (user.startsWith("jcr:"))
					continue;
				//				System.out.println("remove clone for user " + user);
				logger.trace("user " + user);
				logger.trace("workspace.getOwner().getPortalLogin() " + workspace.getOwner().getPortalLogin());

				if (!user.startsWith(JCRRepository.JCR_NAMESPACE) &&
						!user.startsWith(JCRRepository.HL_NAMESPACE))	{	
					//remove clones
					try{
						ItemDelegate cloneNode = getUserNode(sharedNode, user);
						logger.trace("remove clone " + cloneNode.getPath());
						servlets.removeItem(cloneNode.getPath());

					}catch (Exception e) {
						logger.error("Error removing clone " + e);
					}

					try{		
						logger.trace("remove user in userList");
						logger.trace(user + "  deleted from share " + sharedNode.getPath());
						userNode.put(user, (String)null);

						sharedNode.getProperties().put(NodeProperty.USERS, new XStream().toXML(userNode));

						logger.trace(user + "  deleted from share " + sharedNode.getPath());
					}catch (Exception e) {		
						//						e.printStackTrace();
						logger.error("Error removing user from node Users " + e);
					}
					//remove user in memberList
					try{
						@SuppressWarnings("unchecked")
						Map<String, String> memberNode = (Map<String, String>) new XStream().fromXML(sharedNode.getProperties().get(NodeProperty.MEMBERS));
						memberNode.remove(user);
					}catch (Exception e) {
						logger.error("Error removing user from members node");
					}

					DelegateManager wrap = new DelegateManager(sharedNode, workspace.getOwner().getPortalLogin());
					wrap.save();

				}
			}

		} catch (Exception e) {
			throw new InternalErrorException(e);
		} 

	}


	public void removeUserSharedFolder(ItemDelegate sharedNode) throws InternalErrorException, RepositoryException {
		try {

			ItemDelegate userNode = getUserNode(sharedNode);

			// Remove sharedNode from user workspace
			JCRRepository.getServlets().removeItem(userNode.getPath());

			// Remove user in sharingSet
			try{
				@SuppressWarnings("unchecked")
				Map<String,String> usersNode = (Map<String, String>) new XStream().fromXML(sharedNode.getProperties().get(NodeProperty.USERS));
				usersNode.put(workspace.getOwner().getPortalLogin(), new XStream().toXML((String)null));
				sharedNode.getProperties().put(NodeProperty.USERS, new XStream().toXML(usersNode));
			}catch (Exception e) {
				logger.error("Error removing user from users node");
			}

			// Remove user in member node
			try{
				@SuppressWarnings("unchecked")
				Map<String,String> memberNode = (Map<String, String>) new XStream().fromXML(sharedNode.getProperties().get(NodeProperty.MEMBERS));
				memberNode.remove(workspace.getOwner().getPortalLogin());
			}catch (Exception e) {
				logger.error("Error removing user from members node");
			}

			DelegateManager wrap = new DelegateManager(sharedNode, workspace.getOwner().getPortalLogin());
			wrap.save();

		}catch (Exception e) {
			throw new InternalErrorException(e);
		}
	}

	@Override
	public boolean isVreFolder() {
		Boolean flag = false;
		try{
			flag = (Boolean) new XStream().fromXML(delegate.getProperties().get(NodeProperty.IS_VRE_FOLDER));
		}catch (Exception e) {}

		return flag;
	}

	@Override
	public String getDisplayName() {
		String display = "";
		try{
			display = delegate.getProperties().get(NodeProperty.DISPLAY_NAME);
		}catch (Exception e) {
		}
		return display;
	}


	//	public List<String> getGroupIds(Node node) throws PathNotFoundException, RepositoryException {
	//		List<String> groupIdsList = null;
	//
	//		try{
	//			groupIdsList = new ArrayList<String>();
	//			NodeIterator groups = node.getNode(GROUP_IDS).getNodes();
	//			while (groups.hasNext())
	//				groupIdsList.add(groups.nextNode().getName());
	//		}catch (Exception e) {
	//			// TODO: handle exception
	//		}
	//
	//		return groupIdsList;
	//	}

	//	@Override
	//	public List<String> getGroupIds() {
	//		return groupIds;
	//	}



	@Override
	public boolean addAdmin(final String username) throws InsufficientPrivilegesException,
	InternalErrorException {
		if (isAdmin(workspace.getOwner().getPortalLogin())){
			//		if (!getUsers().contains(username))
			try {
				share(new ArrayList<String>(){/**
				 * 
				 */
					private static final long serialVersionUID = 1L;

					{add(username);}});
			} catch (WrongDestinationException e) {
				throw new InternalErrorException(e);
			}

			try {
				List<String> administator = new ArrayList<String>();
				administator.add(username);
				this.setACL(administator, ACLType.ADMINISTRATOR);

			}catch (Exception e) {
				logger.error(e.getMessage());
				return false;
			}
			return true;
		}
		throw new InsufficientPrivilegesException("Insufficient Privileges to set administrators");

	}

	@Override
	public List<String> getAdministrators() throws InternalErrorException {
		List<String> list = null;
		try{
			list = getACLOwner().get(ACLType.ADMINISTRATOR);
		}catch (Exception e) {
			logger.error("no administrators");
		}
		return list;
	}


	public boolean isAdmin(String username) throws InternalErrorException {
		if (getACLUser().equals(ACLType.ADMINISTRATOR))
			return true;
		return false;
	}



	@Override
	public boolean setAdmins(List<String> logins)
			throws InsufficientPrivilegesException, InternalErrorException {
		logger.trace("setAdmins: " + logins.toString() + " on shared folder: " + getAbsolutePath() );
		if (isAdmin(workspace.getOwner().getPortalLogin())){
			try{
				List<String> notAdmins = getAdministrators();

				try{
					notAdmins.removeAll(logins);
				}catch (Exception e) {
					logger.trace("Admins not alredy set on " + getAbsolutePath());
				}
				try{
					notAdmins.remove(getOwner().getPortalLogin());
				}catch (Exception e) {
					logger.trace("Admins not alredy set on " + getAbsolutePath());
				}

				ACLType privilege = getPrivilege();

				logger.trace("Set " + privilege + " on users " +  notAdmins);
				JCRAccessManager accessManager = new JCRAccessManager();

				//set default ACL for users not more included in share
				logger.trace("Setting " + privilege + " on users " +  notAdmins);
				try{
					accessManager.deleteAces(getAbsolutePath(), notAdmins);
				}catch (Exception e) {
					logger.error("Error deleting aces on " + getAbsolutePath());
				}
				try{
					this.setACL(notAdmins, privilege);
				}catch (Exception e) {
					logger.error("Error setting aces on " + getAbsolutePath());
				}

				try{
					//set admin ACL for new admins
					logins.removeAll(notAdmins);
				}catch (Exception e) {
					logger.error("error removing all");
				}
				logger.trace("Setting Admin on users " +  logins);

				for (String user: logins)
					addAdmin(user);

				return true;
			}catch (Exception e) {
				logger.error("Error setting admins on node " +getAbsolutePath(), e);
				return false;
			}
		}
		throw new InsufficientPrivilegesException("Insufficient Privileges to edit administrators");


	}


	@Override
	public void setACL(List<String> users, ACLType privilege) throws InternalErrorException {

		List<String> notAdmins = new ArrayList<String>(users);
		List<String> admins = getAdministrators();
		if (admins!=null){
			notAdmins.removeAll(admins);
			logger.info("notAdmin users " + notAdmins.toString());
		}else
			logger.info("No Administrators on " + getAbsolutePath());

		//		String absPath = null;


		boolean flag = false;
		JCRAccessManager accessManager = new JCRAccessManager();

		int i = 0;
		while ((flag==false) && (i<3)){
			i++;
			try{
				//				absPath = getAbsolutePath();

				if (getAbsolutePath() == null)
					throw new InternalErrorException("Absolute path cannot be null setting ACL");

				switch(privilege){

				case READ_ONLY:
					if (notAdmins.size()>0)
						flag = accessManager.setReadOnlyACL(notAdmins, getAbsolutePath());		
					break;
				case WRITE_OWNER:	
					if (notAdmins.size()>0)
						flag = accessManager.setWriteOwnerACL(notAdmins, getAbsolutePath());		
					break;
				case WRITE_ALL:
					if (notAdmins.size()>0)
						flag = accessManager.setWriteAllACL(notAdmins, getAbsolutePath());	
					break;
				case ADMINISTRATOR:
					flag = accessManager.setAdminACL(users, getAbsolutePath());	
					break;
				default:
					break;
				}

				if (flag==false)
					Thread.sleep(1000);

			}catch (Exception e) {
				logger.error("an error occurred setting ACL on: " + getAbsolutePath());
			}
		}


		logger.info("Has ACL been modified correctly for users " + users.toString() + "in path " + getAbsolutePath() + "? " + flag);
		//set administators

		setAdministrators(accessManager, admins);

	}

	private void setAdministrators(JCRAccessManager accessManager, List<String> admins) throws InternalErrorException {
		if (!isVreFolder()){
			if (admins==null){	
				boolean isSet = false;

				int j = 0;
				while ((isSet==false) && (j<3)){
					j++;
					try{
						String owner = workspace.getOwner().getPortalLogin();
						List<String> adminList =new ArrayList<String>();
						adminList.add(owner);
						logger.info("Set " + owner + " ad administrator");
						isSet = accessManager.setAdminACL(adminList, getAbsolutePath());
						logger.info("Has ACL been modified correctly for users " + adminList.toString() + "in path " + getAbsolutePath() + "? " + isSet);
						if (isSet==false)
							Thread.sleep(1000);
					}catch (Exception e) {
						logger.error("Error setting administators on " + getAbsolutePath());
					}
				}
			}
		}

	}

	@Override
	public List<String> getGroups() throws InternalErrorException {
		List<String> groups = new ArrayList<String>();

		UserManager gm = HomeLibrary
				.getHomeManagerFactory().getUserManager();

		List<String> members = getMembers();
		for (String member: members){
			if (gm.isGroup(member))
				groups.add(member);					
		}
		return groups;	
	}


	/**
	 * Share folder
	 * @throws RepositoryException
	 * @throws InternalErrorException 
	 */
	public void share() throws RepositoryException, InternalErrorException {
		//resolve groupIds
		List<String> usersList = listUsers(users);
		//		System.out.println("SHARE WITH USERS " + usersList.toString());
		shareWithUses(usersList);

	}


}
