package org.gcube.common.homelibrary.jcr.repository.external;

import java.io.IOException;
import java.io.InputStream;

import org.gcube.common.homelibrary.jcr.repository.JCRRepository;
import org.gcube.common.homelibrary.jcr.workspace.util.Utils;
import org.gcube.common.scope.api.ScopeProvider;
import org.gcube.contentmanagement.blobstorage.resource.MyFile;
import org.gcube.contentmanagement.blobstorage.service.IClient;
import org.gcube.contentmanagement.blobstorage.transport.backend.RemoteBackendException;
import org.gcube.contentmanager.storageclient.model.protocol.smp.Handler;
import org.gcube.contentmanager.storageclient.wrapper.AccessType;
import org.gcube.contentmanager.storageclient.wrapper.StorageClient;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class GCUBEStorage {

	private static Logger logger = LoggerFactory.getLogger(GCUBEStorage.class);

	IClient storage = null;


	public GCUBEStorage(String portalLogin){

		logger.trace("Calling GCUBEStorage from user: " + portalLogin);
		try {

			String currentScope = ScopeProvider.instance.get();
			String scope = Utils.getRootScope(currentScope);			
			ScopeProvider.instance.set(scope);	

			logger.trace("GCUBE Storage scope : " + scope);

			//			storage = new StorageClient("org.gcube.portlets.user","home-library",
			//					portalLogin, AccessType.SHARED, currentScope, false).getClient();

			storage = new StorageClient("org.gcube.portlets.user", JCRRepository.serviceName,
					portalLogin, AccessType.SHARED, currentScope, false).getClient();

			Handler.activateProtocol();

		} catch (Exception e) {
			logger.error("GCUBE Storage initialization failed",e);
		}
	}

	public String putStream(InputStream is, String remotePath, String portalLogin, String mimeType) throws RemoteBackendException, IOException {

		logger.trace("GCUBE Storage - putStream - remotePath: " + remotePath);
		String storageId = null;
		try {
			storageId = storage.put(true, mimeType).LFile(is).RFile(remotePath);
		} catch (Exception e) {
			logger.error("GCUBE Storage error in putStream operation",e);
		}finally{
			if (is!=null){
				is.close();
				logger.trace("GCUBE Storage inputStream closed");
			}
		}
		return storageId;
//		return storage.getUrl(true).RFile(remotePath);
	}



	public String putStream(String old_remotePath, String new_remotePath, String portalLogin, String mimeType) throws RemoteBackendException {
		logger.trace("GCUBE Storage -putStream - remotePath: " + new_remotePath);
		return storage.put(true, mimeType).LFile(old_remotePath).RFile(new_remotePath);
//		return storage.getUrl().RFile(new_remotePath);
	}

	public long getRemoteFileSize(String remotePath, String portalLogin) throws RemoteBackendException {
		logger.trace("GCUBE Storage getRemoteFileSize - storage.getSize().RFile(" + remotePath+ ")");
		long size =storage.getSize().RFile(remotePath);
		return size;
	}

	/**
	 * delete file	
	 * @param remotePath
	 * @param portalLogin
	 * @throws RemoteBackendException
	 */
	public void removeRemoteFile(String remotePath, String portalLogin) throws RemoteBackendException {
		logger.trace("GCUBE Storage - removeRemoteFile - remotePath: " + remotePath);
		storage.remove().RFile(remotePath);
	}

	/**
	 * Delete folder
	 * @param remotePath
	 * @param portalLogin
	 * @throws RemoteBackendException
	 */
	public void removeRemoteFolder(String remotePath, String portalLogin) throws RemoteBackendException {
		logger.trace("GCUBE Storage - storage.removeDir().RDir(" + remotePath + ")");
		storage.removeDir().RDir(remotePath);
	}

	/**
	 * Get size of a file by remotepath
	 * @param remotePath
	 * @param portalLogin
	 * @param serviceName
	 * @return size of a file
	 * @throws RemoteBackendException
	 */
	public long getRemoteFileSize(String remotePath,
			String portalLogin, String serviceName) throws RemoteBackendException {
		logger.trace("GCUBE Storage getRemoteFileSize - storage.getSize().RFile(" + remotePath+ ")");
		return storage.getSize().RFile(remotePath);
	}

	/**
	 * Get the disk usage amount of a user 
	 * @param portalLogin
	 * @return the disk usage amount of a user 
	 * @throws RemoteBackendException
	 */
	public long getDiskUsageByUser(String portalLogin) throws RemoteBackendException {
		logger.trace("GCUBE Storage - get volume used by user: " + portalLogin + "");
		return Long.parseLong(storage.getTotalUserVolume());
	}

	/**
	 * Get tot items of a user
	 * @param portalLogin
	 * @return tot items of a user
	 * @throws RemoteBackendException
	 */
	public int getTotalItemsByUser(String portalLogin) throws RemoteBackendException {
		logger.trace("GCUBE Storage - get tot items of user " + portalLogin + "");
		return Integer.parseInt(storage.getUserTotalItems());
	}
	/**
	 * Get folder size
	 * @param remotePath
	 * @param portalLogin
	 * @return folder size
	 * @throws RemoteBackendException
	 */
	public long getFolderTotalVolume(String remotePath, String portalLogin) throws RemoteBackendException {
		logger.trace("GCUBE Storage - get volume (" + remotePath + ")");
		return Long.parseLong(storage.getFolderTotalVolume().RDir(remotePath).toString());
	}

	/**
	 * Get last update of a folder
	 * @param remotePath of the folder
	 * @param portalLogin
	 * @return folder size
	 * @throws RemoteBackendException
	 */
	public String getFolderLastUpdate(String remotePath, String portalLogin) throws RemoteBackendException {
		logger.trace("GCUBE Storage - get volume (" + remotePath + ")");;
		return storage.getFolderLastUpdate().RDir(remotePath);
	}

	/**
	 * Get total items in a folder
	 * @param remotePath of the folder
	 * @param portalLogin
	 * @return total items in a folder
	 * @throws RemoteBackendException
	 */
	public int getFolderTotalItems(String remotePath, String portalLogin) throws RemoteBackendException {
		logger.trace("GCUBE Storage - get volume (" + remotePath + ")");
		return Integer.parseInt(storage.getFolderTotalItems().RDir(remotePath).toString());
	}

	/**
	 * Get the inpustream by a remote path
	 * @param remotePath
	 * @param portalLogin
	 * @return the inpustream by a remote path
	 */
	public InputStream getRemoteFile(String remotePath, String portalLogin) throws RemoteBackendException{
		logger.trace("GCUBE Storage -getRemoteFile - stream = storage.get().RFileAsInputStream(" + remotePath+ ")");
		InputStream stream = null;
		stream = storage.get().RFileAsInputStream(remotePath);
		return stream;
	}


	/**
	 * Get public link by remote path
	 * @param remotePath
	 * @param portalLogin
	 * @return public link 
	 */
	public String getPublicLink(String remotePath, String portalLogin) throws RemoteBackendException{
		logger.trace("GCUBE Storage - Generate Public Link for " + remotePath);
		return storage.getUrl().RFile(remotePath);
	}



	/**
	 * Get storage ID by remote path
	 * @param remotePath
	 * @param portalLogin
	 * @return storage ID
	 */
	public String getStorageId(String remotePath, String portalLogin) throws RemoteBackendException{
		logger.trace("GCUBE Storage - Generate Public Link for " + remotePath);
		MyFile file = storage.getMetaFile().RFile(remotePath);
		return file.getId();
	}

	/**
	 * move dir
	 * @param oldPath
	 * @param newPath
	 * @param portalLogin
	 */
	public void moveRemoteFolder(String oldPath, String newPath, String portalLogin) throws RemoteBackendException{
		logger.trace("GCUBE Storage - storage.moveDir().from(" + oldPath + ").to(" + newPath + ")");
		storage.moveDir().from(oldPath).to(newPath);

	}

	/**
	 * move file
	 * @param oldPath
	 * @param newPath
	 * @param portalLogin
	 */
	public void moveRemoteFile(String oldPath, String newPath, String portalLogin) throws RemoteBackendException{
		logger.trace("GCUBE Storage - storage.moveFile().from(" + oldPath + ").to(" + newPath + ")");
		storage.moveFile().from(oldPath).to(newPath);
	}

	/**
	 * copy file
	 * @param oldPath
	 * @param newPath
	 * @param portalLogin
	 */
	public void copyRemoteFile(String oldPath, String newPath, String portalLogin) throws RemoteBackendException{
		logger.trace("GCUBE Storage - storage.copyFile().from(" + oldPath + ").to(" + newPath + ")");
		storage.copyFile().from(oldPath).to(newPath);
	}

	/**
	 * use remotepath to access storage
	 * @param srcPath
	 * @param destPath
	 * @param portalLogin
	 */
	public void createHardLink(String srcPath, String destPath, String portalLogin) throws RemoteBackendException{
		logger.trace("GCUBE Storage - storage.linkFile().from(" + srcPath + ").to(" + destPath + ")");
		storage.linkFile().from(srcPath).to(destPath);
	}


	/**
	 * delete folder
	 * @param remotePath
	 * @param portalLogin
	 * @param serviceName
	 * @throws RemoteBackendException
	 */
	public void removeRemoteFolder(String remotePath, String portalLogin, String serviceName) throws RemoteBackendException {
		logger.trace("GCUBE Storage - storage.removeDir().RDir(" + remotePath + ")");
		storage.removeDir().RDir(remotePath);
	}

	/**
	 * Save inpustream in storage using a remotepath
	 * @param is
	 * @param new_remotePath
	 * @param portalLogin
	 * @param serviceName
	 * @return remote path
	 * @throws RemoteBackendException
	 */
	public String putStream(InputStream is,  String new_remotePath, String portalLogin, String serviceName, String mimeType) throws RemoteBackendException {

		logger.trace("GCUBE Storage - putStream - remotePath: " + new_remotePath);
		storage.put(true, mimeType).LFile(is).RFile(new_remotePath);
		try {
			is.close();
			logger.trace("GCUBE Storage inputStream closed");
		} catch (IOException e) {
			logger.error("GCUBE Storage error during to close inputStream",e);
		}

		return storage.getUrl().RFile(new_remotePath);
	}

}
