package org.gcube.common.keycloak.model;

import java.util.Base64;

import org.gcube.com.fasterxml.jackson.annotation.JsonInclude.Include;
import org.gcube.com.fasterxml.jackson.core.JsonProcessingException;
import org.gcube.com.fasterxml.jackson.databind.ObjectMapper;
import org.gcube.com.fasterxml.jackson.databind.ObjectWriter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ModelUtils {

    protected static final Logger logger = LoggerFactory.getLogger(ModelUtils.class);

    private static final ObjectMapper mapper = new ObjectMapper();

    static {
        mapper.setSerializationInclusion(Include.NON_NULL);
    }

    public static String toJSONString(Object object) {
        return toJSONString(object, false);
    }

    public static String toJSONString(Object object, boolean prettyPrint) {
        ObjectWriter writer = prettyPrint ? mapper.writerWithDefaultPrettyPrinter() : mapper.writer();
        try {
            return writer.writeValueAsString(object);
        } catch (JsonProcessingException e) {
            logger.error("Cannot pretty print object", e);
            return null;
        }
    }

    private static byte[] getDecodedPayload(String value) {
        return getBase64Decoded(getEncodedPayload(value));
    }

    public static String getAccessTokenPayloadStringFrom(TokenResponse tokenResponse) throws Exception {
        return getAccessTokenPayloadStringFrom(tokenResponse, true);
    }

    public static String getAccessTokenPayloadStringFrom(TokenResponse tokenResponse, boolean prettyPrint) throws Exception {
        return toJSONString(getAccessTokenFrom(tokenResponse, Object.class), prettyPrint);
    }

    public static AccessToken getAccessTokenFrom(TokenResponse tokenResponse) throws Exception {
        return getAccessTokenFrom(tokenResponse, RefreshToken.class);
    }

    private static <T> T getAccessTokenFrom(TokenResponse tokenResponse, Class<T> clazz) throws Exception {
        return mapper.readValue(getDecodedPayload(tokenResponse.getAccessToken()), clazz);
    }

    public static String getRefreshTokenPayloadStringFrom(TokenResponse tokenResponse) throws Exception {
        return getRefreshTokenPayloadStringFrom(tokenResponse, true);
    }

    public static String getRefreshTokenPayloadStringFrom(TokenResponse tokenResponse, boolean prettyPrint) throws Exception {
        return toJSONString(getRefreshTokenFrom(tokenResponse, Object.class), prettyPrint);
    }

    public static RefreshToken getRefreshTokenFrom(TokenResponse tokenResponse) throws Exception {
        return getRefreshTokenFrom(tokenResponse, RefreshToken.class);
    }

    private static <T> T getRefreshTokenFrom(TokenResponse tokenResponse, Class<T> clazz) throws Exception {
        return mapper.readValue(getDecodedPayload(tokenResponse.getRefreshToken()), clazz);
    }

    protected static byte[] getBase64Decoded(String string) {
        return Base64.getDecoder().decode(string);
    }

    protected static String splitAndGet(String encodedJWT, int index) {
        String[] split = encodedJWT.split("\\.");
        if (split.length == 3) {
            return split[index];
        } else {
            return null;
        }
    }

    public static String getEncodedHeader(String encodedJWT) {
        return splitAndGet(encodedJWT, 0);
    }

    public static String getEncodedPayload(String encodedJWT) {
        return splitAndGet(encodedJWT, 1);
    }

    public static String getEncodedSignature(String encodedJWT) {
        return splitAndGet(encodedJWT, 2);
    }

}
