package org.gcube.common.keycloak;

import java.net.URL;
import java.util.List;

import org.gcube.common.keycloak.model.TokenResponse;
import org.gcube.common.scope.api.ScopeProvider;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public interface KeycloakClient {

    Logger logger = LoggerFactory.getLogger(KeycloakClient.class);

    String CATEGORY = "Auth";
    String NAME = "IAM";
    String DESCRIPTION = "oidc-token endpoint";

    /**
     * Finds the keycloak endpoint {@link URL} discovering it in the current scope provided by {@link ScopeProvider}
     * 
     * @return the keycloak endpoint URL in the current scope
     * @throws KeycloakClientException if something goes wrong discovering the endpoint URL
     */
    URL findTokenEndpointURL() throws KeycloakClientException;

    /**
     * Queries an UMA token from the Keycloak server, by using provided authorization, for the given audience (context),
     * in URLEncoded form or not, and optionally a list of permissions.
     * 
     * @param tokenUrl the token endpoint {@link URL} of the OIDC server
     * @param authorization the authorization to be set as header (e.g. a "Basic ...." auth or an encoded JWT access token preceded by the "Bearer " string)
     * @param audience the audience (context) where to request the issuing of the ticket (URLEncoded)
     * @param permissions a list of permissions, can be <code>null</code>
     * @return the issued token as {@link TokenResponse} object
     * @throws KeycloakClientException if something goes wrong performing the query
     */
    TokenResponse queryUMAToken(URL tokenURL, String authorization, String audience, List<String> permissions)
            throws KeycloakClientException;

    /**
     * Queries an UMA token from the Keycloak server, by using provided clientId and client secret for the given audience
     * (context), in URLEncoded form or not, and optionally a list of permissions.
     * 
     * @param tokenURL the token endpoint {@link URL} of the Keycloak server
     * @param clientId the client id
     * @param clientSecret the client secret
     * @param audience the audience (context) where to request the issuing of the ticket
     * @param permissions a list of permissions, can be <code>null</code>
     * @return the issued token as {@link TokenResponse} object
     * @throws KeycloakClientException if something goes wrong performing the query
     */
    TokenResponse queryUMAToken(URL tokenURL, String clientId, String clientSecret, String audience,
            List<String> permissions)
            throws KeycloakClientException;

    /**
     * Queries an UMA token from the Keycloak server discovered in the current scope, by using provided clientId and client secret
     * for the given audience (context), in URLEncoded form or not, and optionally a list of permissions.
     * 
     * @param clientId the client id
     * @param clientSecret the client secret
     * @param audience the audience (context) where to request the issuing of the ticket
     * @param permissions a list of permissions, can be <code>null</code>
     * @return the issued token as {@link TokenResponse} object
     * @throws KeycloakClientException if something goes wrong performing the query
     */
    TokenResponse queryUMAToken(String clientId, String clientSecret, String audience, List<String> permissions)
            throws KeycloakClientException;

    /**
     * Queries an UMA token from the Keycloak server discovered in the current scope, by using provided clientId and client secret
     * for the current scope audience (context), in URLEncoded form or not, and optionally a list of permissions.
     * 
     * @param clientId the client id
     * @param clientSecret the client secret
     * @param permissions a list of permissions, can be <code>null</code>
     * @return the issued token as {@link TokenResponse} object
     * @throws KeycloakClientException if something goes wrong performing the query
     */
    TokenResponse queryUMAToken(String clientId, String clientSecret, List<String> permissions)
            throws KeycloakClientException;

    /**
     * Refreshes a previously issued token from the Keycloak server discovered in the current scope using the refresh
     * token JWT encoded string in the token response object.
     * 
     * Client id will be read from "issued for" access token's claim and client secret will be not sent.
     * <br><b>NOTE</b>: For <code>public</code> clients types only.
     * 
     * @param tokenResponse the previously issued token as {@link TokenResponse} object
     * @return the refreshed token as {@link TokenResponse} object
     * @throws KeycloakClientException if something goes wrong performing the refresh query
     */
    TokenResponse refreshToken(TokenResponse tokenResponse) throws KeycloakClientException;

    /**
     * Refreshes a previously issued token from the Keycloak server using the refresh token JWT encoded string in the
     * token response object.
     * 
     * Client id will be read from "issued for" access token's claim and client secret will be not sent.
     * <br><b>NOTE</b>: For <code>public</code> clients types only.
     * 
     * @param tokenUrl the token endpoint {@link URL} of the OIDC server
     * @param tokenResponse the previously issued token as {@link TokenResponse} object
     * @return the refreshed token as {@link TokenResponse} object
     * @throws KeycloakClientException if something goes wrong performing the refresh query
     */
    TokenResponse refreshToken(URL tokenURL, TokenResponse tokenResponse) throws KeycloakClientException;

    /**
     * Refreshes a previously issued token from the Keycloak server discovered in the current scope using the refresh
     * token JWT encoded string in the token response object and the provided client id.
     * 
     * Client secret will be not sent.
     * <br><b>NOTE</b>: For <code>public</code> clients types only.
     * 
     * @param clientId the requestor client id, may be <code>null</code> and in this case will be take from the access token "issued for" claim
     * @param tokenResponse the previously issued token as {@link TokenResponse} object
     * @return the refreshed token as {@link TokenResponse} object
     * @throws KeycloakClientException if something goes wrong performing the refresh query
     */
    TokenResponse refreshToken(String clientId, TokenResponse tokenResponse) throws KeycloakClientException;

    /**
     * Refreshes a previously issued token from the Keycloak server discovered in the current scope using the refresh
     * token JWT encoded string in the token response object and the provided client id and secret.
     * 
     * @param clientId the requestor client id, may be <code>null</code> and in this case will be take from the access token "issued for" claim
     * @param clientSecret the requestor client secret, may be <code>null</code> for non-confidential clients
     * @param tokenResponse the previously issued token as {@link TokenResponse} object
     * @return the refreshed token as {@link TokenResponse} object
     * @throws KeycloakClientException if something goes wrong performing the refresh query
     */
    TokenResponse refreshToken(String clientId, String clientSecret, TokenResponse tokenResponse)
            throws KeycloakClientException;

    /**
     * Refreshes a previously issued token from the Keycloak server using the refresh token JWT encoded string in the
     * token response object and the provided client id and secret.
     * 
     * @param clientId the requestor client id, may be <code>null</code> and in this case will be take from the access token "issued for" claim
     * @param clientSecret the requestor client secret, may be <code>null</code> for non-confidential clients
     * @param tokenResponse the previously issued token as {@link TokenResponse} object
     * @return the refreshed token as {@link TokenResponse} object
     * @throws KeycloakClientException if something goes wrong performing the refresh query
     */
    TokenResponse refreshToken(URL tokenURL, String clientId, String clientSecret, TokenResponse tokenResponse)
            throws KeycloakClientException;

    /**
     * Refreshes a previously issued token from the Keycloak server discovered in the current scope using the provided
     * client id and the refresh token JWT encoded string obtained with the access token in the previous token response.
     * 
     * Client secret will be not used.
     * <br><b>NOTE</b>: For <code>public</code> clients types only.
     * 
     * @param clientId the requestor client id
     * @param refreshTokenJWTString the previously issued refresh token JWT string taken from the same token response of the access token parameter
     * @return the refreshed token as {@link TokenResponse} object
     * @throws KeycloakClientException if something goes wrong performing the refresh query
     */
    TokenResponse refreshToken(String clientId, String refreshTokenJWTString) throws KeycloakClientException;

    /**
     * Refreshes a previously issued token from the Keycloak server discovered in the current scope using the provided
     * client id and secret and the refresh token JWT encoded string obtained with the access token in the previous
     * token response.
     * 
     * @param clientId the requestor client id
     * @param clientSecret the requestor client secret, may be <code>null</code> for non-confidential clients
     * @param refreshTokenJWTString the previously issued refresh token JWT string taken from the same token response of the access token parameter
     * @return the refreshed token as {@link TokenResponse} object
     * @throws KeycloakClientException if something goes wrong performing the refresh query
     */
    TokenResponse refreshToken(String clientId, String clientSecret, String refreshTokenJWTString)
            throws KeycloakClientException;

    /**
     * Refreshes a previously issued token from the Keycloak server by using the client id and secret
     * and the refresh token JWT encoded string obtained with the access token in the previous token response.
     * 
     * @param tokenUrl the token endpoint {@link URL} of the OIDC server
     * @param clientId the requestor client id
     * @param clientSecret the requestor client secret, may be <code>null</code> for non-confidential clients
     * @param refreshTokenJWTString the previously issued refresh token JWT string
     * @return the refreshed token as {@link TokenResponse} object
     * @throws KeycloakClientException if something goes wrong performing the refresh query
     */
    TokenResponse refreshToken(URL tokenURL, String clientId, String clientSecret, String refreshTokenJWTString)
            throws KeycloakClientException;

}