package org.gcube.contentmanagement.codelistmanager.entities;

import java.util.Calendar;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.Map;
import java.util.UUID;
import org.gcube.common.dbinterface.Order;
import org.gcube.common.dbinterface.Specification;
import org.gcube.common.dbinterface.Order.OrderType;
import org.gcube.common.dbinterface.attributes.SimpleAttribute;
import org.gcube.common.dbinterface.persistence.ObjectPersistency;
import org.gcube.common.dbinterface.persistence.ObjectStateControl;
import org.gcube.common.dbinterface.persistence.annotations.FieldDefinition;
import org.gcube.common.dbinterface.persistence.annotations.TableRootDefinition;
import org.gcube.common.dbinterface.pool.DBSession;
import org.gcube.common.dbinterface.queries.CreateTableFromSelect;
import org.gcube.common.dbinterface.queries.Select;
import org.gcube.common.dbinterface.tables.SimpleTable;
import org.gcube.contentmanagement.codelistmanager.entities.TableField.ColumnType;
import org.gcube.contentmanagement.codelistmanager.exception.CurationNotFinishedException;
import org.gcube.contentmanagement.codelistmanager.exception.SerializationException;
import org.gcube.contentmanagement.codelistmanager.managers.CodeListCuration;
import org.gcube.contentmanagement.codelistmanager.util.CodeListType;
import org.gcube.contentmanagement.codelistmanager.util.Constants;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;


@TableRootDefinition
public class CodeList extends ObjectStateControl{

	private static final Logger logger = LoggerFactory.getLogger(CodeList.class);
	
	@FieldDefinition(precision={40}, specifications={Specification.NOT_NULL, Specification.PRIMARY_KEY})
	private String id;
	
	@FieldDefinition(precision={200}, specifications={Specification.NOT_NULL})
	private String name;

	@FieldDefinition(specifications={Specification.NOT_NULL})
	private String description;

	@FieldDefinition(precision={100}, specifications={Specification.NOT_NULL})
	private String agency;

	@FieldDefinition(specifications={Specification.NOT_NULL})
	private boolean isFinal;

	@FieldDefinition(specifications={Specification.NOT_NULL})
	private Calendar creationDate;
	
	@FieldDefinition(specifications={Specification.NOT_NULL})
	private Calendar modificationDate;
	
	@FieldDefinition(specifications={Specification.NOT_NULL})
	private boolean approved;
	
	@FieldDefinition(precision={4,2}, specifications={Specification.NOT_NULL})
	private float version=0.0f;
	
	@FieldDefinition
	private String versionLog;
	
	@FieldDefinition(precision={40})
	private String parentCodeListId= null;
	
	@FieldDefinition()
	private Hashtable<String, TableField > labelFieldMapping;
	
	@FieldDefinition()
	private SimpleTable table;
	
	@FieldDefinition()
	private CodeListType codelistType;
	
	@FieldDefinition(precision={60})
	private String tableName;
	
	@FieldDefinition
	private boolean informationInherited = false;
	
	
	
	public CodeList(){
		this.id = UUID.randomUUID().toString();	
		Calendar now= Calendar.getInstance();
		this.modificationDate= now;
		this.creationDate = now;
		this.informationInherited = true;
	}
	
	public CodeList(String agencyId, String name, String description, boolean isFinal) {
		this.id = UUID.randomUUID().toString();
		this.version=0.0f;
		this.approved= false;
		Calendar now= Calendar.getInstance();
		this.modificationDate= now;
		this.creationDate = now;
		this.name = name;
		this.isFinal = isFinal;
		this.description = description;
		this.agency= agencyId;
		this.informationInherited = false;
	}

	public boolean initialize(CodeListCuration mapper){
		try {
			if (!mapper.isMappingFinished()) throw new CurationNotFinishedException();
			this.table = createTable(mapper.getTable());
			this.tableName = table.getTableName();
			table.initializeCount();
			table.initializeFieldMapping();
			this.parentCodeListId = mapper.getId();
			this.labelFieldMapping = mapper.getLabelFieldMapping();
			this.codelistType = mapper.getCodelistType();
			this.version = mapper.getVersion();
			if (this.informationInherited){
				this.name = mapper.getName();
				this.description = mapper.getDescription();
				this.isFinal = mapper.isFinal();
				this.agency = mapper.getAgencyId();
			}
			if (!this.store()) throw new SerializationException();
		} catch (Exception e) {
			logger.error("error initializing the codelist",e);
			return false;
		}
		return true;
	}
	
		
	/**
	 * @return the codelistType
	 */
	public CodeListType getCodelistType() {
		return codelistType;
	}

	/**
	 * @return the table
	 */
	public SimpleTable getTable() {
		return table;
	}

	
	
	/**
	 * @return the version
	 */
	public float getVersion() {
		return version;
	}

	/**
	 * @return the parentCodeListId
	 */
	public String getParentCodeListId() {
		return parentCodeListId;
	}

	public String getId() {
		return id;
	}

	public void setId(String id) {
		this.id = id;
	}

	public String getName() {
		return name;
	}

	public void setName(String name) {
		this.name = name;
	}
		
	/**
	 * @return the concept
	 */
	public String getDescription() {
		return description;
	}

	/**
	 * @param concept the concept to set
	 */
	public void setDescription(String description) {
		this.description = description;
	}

	/**
	 * @return the agency
	 */
	public String getAgency() {
		return agency;
	}

	/**
	 * @param agency the agency to set
	 */
	public void setAgency(String agency) {
		this.agency = agency;
	}

	/**
	 * @return the isFinal
	 */
	public boolean isFinal() {
		return isFinal;
	}

	/**
	 * @param isFinal the isFinal to set
	 */
	public void setFinal(boolean isFinal) {
		this.isFinal = isFinal;
	}

	public String getRelatedTableName() {
		if (table==null) return null;
		return table.getTableName();
	}
		
	/**
	 * @return the labelFieldMapping
	 */
	public Hashtable<String, TableField > getLabelFieldMapping() {
		return labelFieldMapping;
	}

	
	
	
	/**
	 * @return the modificationDate
	 */
	public Calendar getModificationDate() {
		return modificationDate;
	}

	/**
	 * @param modificationDate the modificationDate to set
	 */
	public void setModificationDate(Calendar modificationDate) {
		this.modificationDate = modificationDate;
	}

	/**
	 * @return the approved
	 */
	public boolean isApproved() {
		return approved;
	}
	
	/**
	 * @return the creationDate
	 */
	public Calendar getCreationDate() {
		return creationDate;
	}

	/**
	 * @param version the version to set
	 */
	public void setVersion(float version) {
		this.version = version;
	}

	/**
	 * @param approved the approved to set
	 */
	public void setApproved(boolean approved) {
		this.approved = approved;
	}

	public int getCount() throws Exception{
		SimpleTable table = new SimpleTable(this.getRelatedTableName());
		table.initializeCount();
		return table.getCount();
	}
	
	public String getCodeColumnId(){
		for (TableField tf : this.labelFieldMapping.values())
			if (tf.getColumnReference().getType()==ColumnType.Code) return tf.getId();
		return null;
	}
	
	public boolean addLine(){
		
		return false;
	}
	
	public boolean removeLine(){
		return false;
	}
	
	public void remove()  throws Exception {
		ObjectPersistency.get(CodeList.class).deleteByKey(id);
		
	}

	public boolean store(){
		try{
			ObjectPersistency<CodeList> pers = ObjectPersistency.get(CodeList.class);
			if (!pers.existsKey(this.id))
				ObjectPersistency.get(CodeList.class).insert(this);
			else {
				this.modificationDate = Calendar.getInstance();
				ObjectPersistency.get(CodeList.class).update(this);
			}
		}catch (Exception e) {
			logger.error("error storing on DB",e);
			return false;
		}
		return true;
	}
	
	
	private SimpleTable createTable(SimpleTable mapperTable) throws SerializationException{
		DBSession session=null;
		try{
			session= DBSession.connect();
			CreateTableFromSelect createTable = DBSession.getImplementation(CreateTableFromSelect.class);
			createTable.setTableName(Constants.RD_PREFIX+this.id.replaceAll("-",""));
			Select select = DBSession.getImplementation(Select.class);
			select.setTables(mapperTable);
			select.setOrders(new Order(OrderType.ASC,new SimpleAttribute(Constants.ID_LABEL)));
			createTable.setSelect(select);
			SimpleTable table =createTable.execute(session);
			return table;
		}catch (Exception e) {
			throw new SerializationException();
		}finally{
			if (session!=null) session.release();
		}
	}

	
	
	public static CodeList get(String id) throws Exception{
		return ObjectPersistency.get(CodeList.class).getByKey(id);
	}
	
	public static Iterator<CodeList> getByName(String name) throws Exception{
		return ObjectPersistency.get(CodeList.class).getObjectByField("name", name).iterator();
	}
	
	public static Iterator<CodeList> getByType(CodeListType type) throws Exception{
		return ObjectPersistency.get(CodeList.class).getObjectByField("codelistType", type.toString()).iterator();
	}
	
	public static Iterator<CodeList> getAll() throws Exception{
		return ObjectPersistency.get(CodeList.class).getAll().iterator();
	}
	
	public static String getTableName() throws Exception {
		return ObjectPersistency.get(CodeList.class).getTable().getTableName();
	}
	
	public static Map<String, String> getInfo() throws Exception {
		return ObjectPersistency.get(CodeList.class).getInfo();
	}
	
	
}
