package org.gcube.contentmanagement.codelistmanager.managers;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;
import java.util.UUID;
import org.gcube.common.dbinterface.Specification;
import org.gcube.common.dbinterface.TableAlreadyExistsException;
import org.gcube.common.dbinterface.attributes.SimpleAttribute;
import org.gcube.common.dbinterface.conditions.IntArray;
import org.gcube.common.dbinterface.conditions.IsInOperator;
import org.gcube.common.dbinterface.persistence.ObjectPersistency;
import org.gcube.common.dbinterface.persistence.ObjectStateControl;
import org.gcube.common.dbinterface.persistence.annotations.FieldDefinition;
import org.gcube.common.dbinterface.persistence.annotations.TableRootDefinition;
import org.gcube.common.dbinterface.pool.DBSession;
import org.gcube.common.dbinterface.queries.CreateTableLike;
import org.gcube.common.dbinterface.queries.Delete;
import org.gcube.common.dbinterface.queries.InsertFromSelect;
import org.gcube.common.dbinterface.queries.Select;
import org.gcube.common.dbinterface.tables.SimpleTable;
import org.gcube.common.dbinterface.types.Type;
import org.gcube.common.dbinterface.utils.Utility;
import org.gcube.contentmanagement.codelistmanager.entities.CodeList;
import org.gcube.contentmanagement.codelistmanager.entities.TableField;
import org.gcube.contentmanagement.codelistmanager.entities.TableField.ColumnType;
import org.gcube.contentmanagement.codelistmanager.managers.CodeListImport.Status;
import org.gcube.contentmanagement.codelistmanager.managers.handlers.CodeListCurationHandler;
import org.gcube.contentmanagement.codelistmanager.managers.handlers.CodeListHandler;
import org.gcube.contentmanagement.codelistmanager.managers.handlers.HierarchicalCodeListHandler;
import org.gcube.contentmanagement.codelistmanager.util.CodeListType;
import org.gcube.contentmanagement.codelistmanager.util.ColumnReference;
import org.gcube.contentmanagement.codelistmanager.util.Constants;
import org.gcube.contentmanagement.codelistmanager.util.RowIterator;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@TableRootDefinition
public class CodeListCuration extends ObjectStateControl{

	private static final Logger logger = LoggerFactory.getLogger(CodeListCuration.class);
	
	@FieldDefinition(precision={40},specifications={Specification.NOT_NULL, Specification.PRIMARY_KEY})
	protected String id;
	
	@FieldDefinition
	protected Hashtable<String, TableField > labelFieldMapping;
	
	@FieldDefinition(precision={100}, specifications={Specification.NOT_NULL})
	protected String agencyId;
	
	@FieldDefinition(precision={200}, specifications={Specification.NOT_NULL})
	protected String name;

	@FieldDefinition(specifications={Specification.NOT_NULL})
	protected String description;
	
	@FieldDefinition(specifications={Specification.NOT_NULL})
	protected Calendar creationDate;
	
	@FieldDefinition(specifications={Specification.NOT_NULL})
	protected Calendar lastModifyDate;
	
	@FieldDefinition(specifications={Specification.NOT_NULL})
	protected boolean isFinal;
	
	@FieldDefinition(precision={4,2},specifications={Specification.NOT_NULL})
	protected float version;
	
	@FieldDefinition(specifications={Specification.NOT_NULL})
	protected CodeListType codelistType;
	
	@FieldDefinition(precision={40})
	protected String codeListImportId;
	
	@FieldDefinition(precision={40},specifications={Specification.NOT_NULL})
	protected SimpleTable table;

		
	
	
	public CodeListCuration(){
		this.id = UUID.randomUUID().toString();	
		Calendar now =Calendar.getInstance();
		this.creationDate = now;
		this.lastModifyDate = now;
	}
	
	public CodeListCuration(String name, String description, String agencyId, boolean isFinal, float version, CodeListType type){
		this.id = UUID.randomUUID().toString();
		this.name = name;
		this.description = description;
		this.agencyId = agencyId;
		this.isFinal = isFinal;
		Calendar now =Calendar.getInstance();
		this.creationDate = now;
		this.lastModifyDate = now;
		this.codelistType = type;
		this.version=version;
	}
		
	public CodeListCuration(String name, String description, String agencyId, boolean isFinal, float version){
		this.id = UUID.randomUUID().toString();;
		this.name = name;
		this.description = description;
		this.agencyId = agencyId;
		this.isFinal = isFinal;
		Calendar now =Calendar.getInstance();
		this.creationDate = now;
		this.lastModifyDate = now;
		this.version = version;
		this.codelistType = CodeListType.Unknown;
	}
	
	/**
	 * @return the id
	 */
	public String getId() {
		return id;
	}

	
	protected CodeListCurationHandler getHandler(){
		if (this.codelistType==CodeListType.Hierarchical)
			return HierarchicalCodeListHandler.get();
		else return CodeListHandler.get();
			
		
	}
	
	
	
	/**
	 * @return the table
	 */
	public SimpleTable getTable() {
		return table;
	}

	/**
	 * @return the count
	 */
	public int getCount() {
		if (table==null) return 0;
		try {
			return table.getCount();
		} catch (Exception e) {
			return 0;
		}
	}

	public void updateCount() {
		try {
			this.table.initializeCount();
		} catch (Exception e) {
			logger.warn("error initiliazing count",e);
		}
	}
	
	
	
	/**
	 * @return the version
	 */
	public float getVersion() {
		return version;
	}

	/**
	 * @param version the version to set
	 */
	public void setVersion(float version) {
		this.version = version;
	}

	/**
	 * @param agencyId the agencyId to set
	 */
	public void setAgencyId(String agencyId) {
		this.agencyId = agencyId;
	}

	/**
	 * @param isFinal the isFinal to set
	 */
	public void setFinal(boolean isFinal) {
		this.isFinal = isFinal;
	}

	/**
	 * @return the codelistType
	 */
	public CodeListType getCodelistType() {
		return codelistType;
	}

	/**
	 * @param codelistType the codelistType to set
	 */
	public void setCodelistType(CodeListType codelistType) {
		this.codelistType = codelistType;
	}

	/**
	 * @return the isFinal
	 */
	public boolean isFinal() {
		return isFinal;
	}

	/**
	 * @return the codeListImportId
	 */
	public String getCodeListImportId() {
		return codeListImportId;
	}

	/**
	 * @return the creationDate
	 */
	public Calendar getCreationDate() {
		return creationDate;
	}

	/**
	 * @return the lastModified
	 */
	public Calendar getLastModifyDate() {
		return lastModifyDate;
	}

	/**
	 * @return the agencyId
	 */
	public String getAgencyId() {
		return agencyId;
	}

	/**
	 * @return the labelFieldMapping
	 */
	public Hashtable<String, TableField > getLabelFieldMapping() {
		return labelFieldMapping;
	}

	

	/**
	 * @param labelFieldMapping the labelFieldMapping to set
	 */
	public void setLabelFieldMapping(Hashtable<String, TableField> labelFieldMapping) {
		this.labelFieldMapping = labelFieldMapping;
	}

	/**
	 * @return the tableName
	 */
	public String getTableName() {
		if (this.table==null) return null;
		return this.table.getTableName();
	}

	
	/**
	 * @return the name
	 */
	public String getName() {
		return name;
	}

	/**
	 * @param name the name to set
	 */
	public void setName(String name) {
		this.name = name;
	}

	/**
	 * @return the description
	 */
	public String getDescription() {
		return description;
	}

	/**
	 * @param description the description to set
	 */
	public void setDescription(String description) {
		this.description = description;
	}

	/**
	 * 
	 * @param importManager
	 * @param parent
	 * @param fieldsMask
	 * @return
	 */
	public boolean start(CodeListImport importManager){
		this.codeListImportId = importManager.getId();
		DBSession session= null;
		try{
			if (importManager.getStatus()== Status.Failed) throw new Exception("cannot start a curation from a failed import");			
			session = DBSession.connect();
			session.disableAutoCommit();
			CreateTableLike createLike= DBSession.getImplementation(CreateTableLike.class);		
			createLike.setTableLike(new SimpleTable(importManager.getTableName()));
			createLike.setTableName(Constants.MAPPING_PREFIX+id.replace("-",""));
			try{
				table = createLike.execute(session);
			}catch (TableAlreadyExistsException e) {
				logger.info("the table already exist, recreating it");
				Utility.drop(importManager.getTableName()).execute(session);
				this.table = createLike.execute(session);
			}
			Select select = DBSession.getImplementation(Select.class);
			select.setTables(new SimpleTable(importManager.getTableName()));
			select.setUseDistinct(true);
		
	
			InsertFromSelect insert= DBSession.getImplementation(InsertFromSelect.class);
			insert.setSubQuery(select);
			insert.setTable(table);
			insert.execute(session);
			session.commit();
			logger.trace("the table name is:"+table.getTableName());
			this.labelFieldMapping=importManager.getFieldsName();
			if (importManager.getCodelistType()== CodeListType.Hierarchical){
				for (TableField tf: this.labelFieldMapping.values())
					if (tf.getColumnReference().getType()== ColumnType.HLParentCode || tf.getColumnReference().getType()== ColumnType.HLChildCode){
						Iterator<CodeList> codelistIt =CodeList.getByName(tf.getColumnReference().getCodelistReferenceName());
						if (codelistIt.hasNext())
							tf.getColumnReference().setCodelistReferenceId(codelistIt.next().getId());
						//else throw new Exception("no codelist retrieved with the specified name");
					}
						
			}
			if (importManager.isFromSDMXImport()){
				this.setAgencyId(importManager.getAgencyId());
				this.setDescription(importManager.getDescription());
				this.setFinal(importManager.isFinal());
				this.setName(importManager.getName());
				this.setVersion(importManager.getVersion());
				this.setCodelistType(importManager.getCodelistType());
			}
			table.initializeCount();
			this.store();
			session.commit();
		}catch (Exception e) {
			logger.error("error starting the mapping ",e);
			try{
				if(session!=null)session.rollback();
			}catch(Exception e1){
				logger.warn("error cleaning the table");
			}
			return false;
		}finally{
			if (session!=null) session.release();
		}
		return true;
	}
	
	
	public List<String> getFieldsIdByColumnType(ColumnType columnType){
		List<String> toReturn = new ArrayList<String>();
		for(TableField tf: this.labelFieldMapping.values()){
			if (tf.getColumnReference().getType()==columnType)
				toReturn.add(tf.getId());
		}
		return toReturn;
	}
	
	/**
	 * 
	 * @param fieldId
	 * @return
	 */
	public boolean removeColumn(String fieldId){
		DBSession session = null;
		try{
			session= DBSession.connect();
			Utility.dropColumn(fieldId,this.table).execute(session);
			this.getLabelFieldMapping().remove(fieldId);
			this.table.initializeFieldMapping(session);
		}catch (Exception e) {
			logger.error("error dropping column",e);
			return false;
		}finally{
			if (session!=null) session.release();
		}
		return true;
	}

	/**
	 * 
	 * @param rowsId
	 * @return
	 */
	public boolean removeRows(int ... rowsId){
		DBSession session=null;
		try{
			Delete deleteRows= DBSession.getImplementation(Delete.class);
			deleteRows.setFilter(new IsInOperator(new SimpleAttribute(Constants.ID_LABEL), new IntArray(rowsId)));
			deleteRows.setTable(this.table);
			deleteRows.execute(session);
			this.table.initializeCount(session);
		}catch (Exception e) {
			logger.error("error deleting rows ",e);
			return false;
		}finally{
			if (session!=null) session.release();
		}
		return true;
	}
	
	/**
	 * 
	 * @return
	 * @throws Exception
	 */
	public Iterator<List<String>> getData() throws Exception{
		Select select = DBSession.getImplementation(Select.class);
		select.setTables(this.table);
		return getData(select);
		
	}
	
	/**
	 * 
	 * @param rowsIds
	 * @return
	 * @throws Exception
	 */
	public Iterator<List<String>> getRowsData(Integer ... rowsIds) throws Exception{
		Select select = DBSession.getImplementation(Select.class);
		select.setTables(this.table);
		select.setFilter(new IsInOperator(new SimpleAttribute(Constants.ID_LABEL),new IntArray(rowsIds)));
		return getData(select);
	}
	
	
	/**
	 * 
	 * @param query
	 * @return
	 * @throws Exception
	 */
	private Iterator<List<String>> getData(Select query) throws Exception{
		DBSession session= null;
		try{
			session= DBSession.connect();
			return new RowIterator(query);
		}finally{
			if (session!=null) session.release();
		}
	}
	
	
	public boolean store(){
		try{
			ObjectPersistency<CodeListCuration> op = ObjectPersistency.get(CodeListCuration.class);
			if (!op.existsKey(this.getId()))
				op.insert(this);
			else op.update(this);
		}catch (Exception e) {
			logger.error("error storing on DB",e);
			return false;
		}
		return true;
	}
	
	/**
	 * 
	 * @return
	 */
	public boolean remove(){
		try {
			Utility.drop(this.getTableName());
			ObjectPersistency.get(CodeListCuration.class).deleteByKey(id);
		} catch (Exception e) {
			return false;
		}
		return true;
	}	
	
	
	/**
	 * 
	 * @return
	 */
	public boolean isMappingFinished(){
		return getHandler().isMappingFinished(table, labelFieldMapping);
	}
	

	
	/**
	 * 
	 * @return
	 * @throws Exception
	 */
	public  static  Iterator<CodeListCuration> getAll() throws Exception{
		return ObjectPersistency.get(CodeListCuration.class).getAll().iterator();
	}
	
	/**
	 * 
	 * @param id
	 * @return
	 * @throws Exception
	 */
	public static CodeListCuration get(String id) throws Exception{
		return ObjectPersistency.get(CodeListCuration.class).getByKey(id);
	}
	
	/**
	 * 
	 * @param id
	 * @throws Exception
	 */
	public static void destroy(String id) throws Exception{
		ObjectPersistency.get(CodeListCuration.class).deleteByKey(id);
	}

	public void changeColumnType(String fieldId, ColumnType columnType, String ... codelistReferenceId) throws Exception {
		if (codelistReferenceId==null)
			getHandler().changeColumnType(fieldId, columnType, table, labelFieldMapping);
		else getHandler().changeColumnType(fieldId, columnType, table, labelFieldMapping, codelistReferenceId);
	}

	public Integer[] getInvalidRows(String fieldId, ColumnReference columnDefinition) throws Exception{
		return getHandler().checkInvalidValues(fieldId, columnDefinition,  table, labelFieldMapping);
	}

	public Integer[] getInvalidRows(String fieldId, Type type) throws Exception{
		return getHandler().checkInvalidValues(fieldId, type,  table, labelFieldMapping);
	}
	
	public void replaceValue(String fieldId, String newValue, int row) throws Exception{
		getHandler().replaceValue(fieldId, newValue, row, table, labelFieldMapping);
		
	}
	
	
}
