package org.gcube.contentmanagement.codelistmanager.managers;

import java.io.File;
import java.net.URI;
import java.util.Calendar;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;
import java.util.UUID;

import org.gcube.common.dbinterface.ColumnDefinition;
import org.gcube.common.dbinterface.Specification;
import org.gcube.common.dbinterface.TableAlreadyExistsException;
import org.gcube.common.dbinterface.persistence.ObjectPersistency;
import org.gcube.common.dbinterface.persistence.ObjectStateControl;
import org.gcube.common.dbinterface.persistence.annotations.FieldDefinition;
import org.gcube.common.dbinterface.persistence.annotations.TableRootDefinition;
import org.gcube.common.dbinterface.pool.DBSession;
import org.gcube.common.dbinterface.queries.CreateTable;
import org.gcube.common.dbinterface.tables.SimpleTable;
import org.gcube.common.dbinterface.utils.Utility;
import org.gcube.contentmanagement.codelistmanager.entities.TableField;
import org.gcube.contentmanagement.codelistmanager.util.CodeListType;
import org.gcube.contentmanagement.codelistmanager.util.Constants;
import org.gcube.contentmanagement.codelistmanager.util.ImporterInterface;
import org.gcube.contentmanagement.codelistmanager.util.csv.CSVImport;
import org.gcube.contentmanagement.codelistmanager.util.csv.ImportLineProcessor;
import org.gcube.contentmanagement.codelistmanager.util.csv.InitializerProcessor;
import org.gcube.contentmanagement.codelistmanager.util.csv.ProgresChangedEvent;
import org.gcube.contentmanagement.codelistmanager.util.opensdmx.CLSDMXImport;
import org.gcube.contentmanagement.codelistmanager.util.opensdmx.HCLSDMXImport;
import org.gcube.contentmanagement.codelistmanager.util.opensdmx.SDMXImport;
import org.opensdmx.client.sdmx20.direct.Sdmx20CustomRegistryClient;
import org.sdmx.resources.sdmxml.schemas.v2_0.message.Structure;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@TableRootDefinition
public class CodeListImport extends ObjectStateControl{

	private ProgresChangedEvent onProgressChanged = new ProgresChangedEvent() {
		
		@Override
		public void onProgresChanged(int value) {
			progress = value;
			if (progress % 30 == 0)
				store();
		}
	};
	
	private static final Logger logger = LoggerFactory.getLogger(CodeListImport.class);
	
	public enum Status{InProgress, Initialized, Failed, Complete}; 
	
	@FieldDefinition(precision={40}, specifications={Specification.NOT_NULL})
	private String tableName;
	
	@FieldDefinition(precision={40}, specifications={Specification.NOT_NULL, Specification.PRIMARY_KEY})
	private String id;
		
	@FieldDefinition(precision={100}, specifications={Specification.NOT_NULL})
	private String agencyId;
	
	@FieldDefinition(precision={200}, specifications={Specification.NOT_NULL})
	private String name;

	@FieldDefinition(specifications={Specification.NOT_NULL})
	private String description;

	@FieldDefinition(specifications={Specification.NOT_NULL})
	private boolean isFinal;
	
	@FieldDefinition(precision={4,2}, specifications={Specification.NOT_NULL})
	private float version;
	
	@FieldDefinition(precision={20})
	private String validFrom;
	
	@FieldDefinition(precision={20})
	private String validTo;
	
	@FieldDefinition(specifications={Specification.NOT_NULL})
	private Status status;
		
	@FieldDefinition(specifications={Specification.NOT_NULL})	
	private Calendar lastUpdate;
	
	@FieldDefinition(specifications={Specification.NOT_NULL})
	private int totalLines;
	
	@FieldDefinition(specifications={Specification.NOT_NULL})
	private int progress;
	
	@FieldDefinition(specifications={Specification.NOT_NULL})
	private CodeListType codelistType;
	
	@FieldDefinition()	
	private Hashtable<String, TableField > fieldsName;
	@FieldDefinition()
	private List<ColumnDefinition> tableDefinition;
	
	@FieldDefinition()
	private boolean fromSDMXImport = false;
	
	public CodeListImport(){
		this.id = UUID.randomUUID().toString();;
		this.lastUpdate = Calendar.getInstance();
		this.tableName= Constants.IMPORT_PREFIX+id.replaceAll("-", "");
		this.status = Status.Initialized;
		this.totalLines = 0;
		this.progress = 0;
		this.isFinal= false;
		this.agencyId="undefined";
		this.name="undefined";
		this.description="undefined";
		this.codelistType = CodeListType.Unknown;
		this.version=0.0f;
	}

	public String getTableName() {
		return tableName;
	}

	public Hashtable<String, TableField > getFieldsName() {
		return fieldsName;
	}


	public List<ColumnDefinition> getTableDefinition() {
		return tableDefinition;
	}


	public String getId() {
		return id;
	}
	
	

	/**
	 * @return the codelistType
	 */
	public CodeListType getCodelistType() {
		return codelistType;
	}

	/**
	 * @param codelistType the codelistType to set
	 */
	public void setCodelistType(CodeListType codelistType) {
		this.codelistType = codelistType;
	}

	/**
	 * @return the isFinal
	 */
	public boolean isFinal() {
		return isFinal;
	}

	/**
	 * @param isFinal the isFinal to set
	 */
	public void setFinal(boolean isFinal) {
		this.isFinal = isFinal;
	}

	/**
	 * @return the version
	 */
	public float getVersion() {
		return version;
	}

	/**
	 * @param version the version to set
	 */
	public void setVersion(float version) {
		this.version = version;
	}

	/**
	 * @return the name
	 */
	public String getName() {
		return name;
	}

	/**
	 * @param name the name to set
	 */
	public void setName(String name) {
		this.name = name;
	}

	/**
	 * @return the description
	 */
	public String getDescription() {
		return description;
	}

	/**
	 * @param description the description to set
	 */
	public void setDescription(String description) {
		this.description = description;
	}

	/**
	 * @return the agencyId
	 */
	public String getAgencyId() {
		return agencyId;
	}

	/**
	 * @param agencyId the agencyId to set
	 */
	public void setAgencyId(String agencyId) {
		this.agencyId = agencyId;
	}
	
	/**
	 * @return the status
	 */
	public Status getStatus() {
		return status;
	}

	/**
	 * @param status the status to set
	 */
	public void setStatus(Status status) {
		this.status = status;
	}

	/**
	 * @return the lastUpdate
	 */
	public Calendar getLastUpdate() {
		return lastUpdate;
	}

	/**
	 * @param lastUpdate the lastUpdate to set
	 */
	public void setLastUpdate(Calendar lastUpdate) {
		this.lastUpdate = lastUpdate;
	}

	protected void setFieldsName(Hashtable<String, TableField> tableFieldMapping) {
		this.fieldsName = tableFieldMapping;
	}

	protected void setTableDefinition(List<ColumnDefinition> tableDefinition) {
		this.tableDefinition = tableDefinition;
	}

	
	
	/**
	 * @return the totalLines
	 */
	public int getTotalLines() {
		return totalLines;
	}

	/**
	 * @return the progress
	 */
	public int getProgress() {
		return progress;
	}
	
	/**
	 * 
	 * @return
	 */
	public boolean isFromSDMXImport() {
		return fromSDMXImport;
	}
 
	public boolean store(){
		try{
			ObjectPersistency<CodeListImport> op = ObjectPersistency.get(CodeListImport.class);
			if (!op.existsKey(this.getId()))
				op.insert(this);
			else op.update(this);
		}catch (Exception e) {
			logger.error("error storing on DB",e);
			return false;
		}
		return true;
	}
	
	public boolean importSDMX(URI uri) { 
		try{
			Sdmx20CustomRegistryClient sdmxClient = new Sdmx20CustomRegistryClient();
			Structure structure = sdmxClient.retrieveStructure(uri.toURL());
			return parseSDMXStructure(structure);
		}catch (Exception e) {
			this.setStatus(Status.Failed);
			logger.error("import process failed",e);
			this.store();
			return false;
		}
	}
	
	public boolean importSDMX(File file) {
		try{
			Sdmx20CustomRegistryClient sdmxClient = new Sdmx20CustomRegistryClient();
			Structure structure = sdmxClient.retrieveStructure(file);
			return parseSDMXStructure(structure);
		}catch (Exception e) {
			this.setStatus(Status.Failed);
			logger.error("import process failed",e);
			this.store();
			return false;
		}
	}
	
	private boolean parseSDMXStructure(Structure structure) throws Exception{
		SDMXImport sdmxImport;
		if (structure.getCodeLists()!=null){
			sdmxImport= new CLSDMXImport(structure.getCodeLists().getCodeLists().get(0));
			this.codelistType = CodeListType.Simple;
		}else if (structure.getHierarchicalCodelists()!=null){
			sdmxImport= new HCLSDMXImport(structure.getHierarchicalCodelists().getHierarchicalCodelists().get(0));
			this.codelistType = CodeListType.Hierarchical;
		}
		else{
			logger.error("no valid file for sdmx");
			return false;
		}
		this.setAgencyId(sdmxImport.getAgencyId());
		this.setDescription(sdmxImport.getDescription());
		this.setName(sdmxImport.getName());
		this.setFinal(sdmxImport.isFinal());
		this.setVersion(sdmxImport.getVersion());
		this.fromSDMXImport = true;
		return importData(sdmxImport);
	}
	
	public boolean importCSV(File file, String encoding, boolean hasHeader, char delimiter, boolean deleteFileWhenFinished){
		boolean result =importData(new CSVImport(file, hasHeader, delimiter, encoding, new InitializerProcessor(), new ImportLineProcessor() ));
		if (deleteFileWhenFinished) file.delete();
		return result;
	}
	
	
	private boolean importData(ImporterInterface importerInterface){
		SimpleTable table= null;
		DBSession session= null;
		try{
			this.setStatus(Status.InProgress);
			if (!this.store()) throw new Exception("error storing import");
			session = DBSession.connect();
			session.disableAutoCommit();
			importerInterface.initialize();
			try{
				table= createTable(importerInterface.getTableDefinition(), tableName, session);
			}catch (TableAlreadyExistsException e) {
				Utility.drop(this.tableName).execute(session);
				table= createTable(importerInterface.getTableDefinition(), tableName, session );
			}
			session.commit();
			this.totalLines = importerInterface.getTotalLines();
			if (!this.store()) throw new Exception("error storing import");
			importerInterface.process(table, onProgressChanged);
			this.setTableDefinition(importerInterface.getTableDefinition());
			this.setFieldsName(importerInterface.getFieldsName());
			session.commit();
			
			this.setStatus(Status.Complete);
			if (!this.store()) throw new Exception("error storing import");
		}catch (Exception e) {
			logger.error("import process failed",e);
			try{
				if (session!=null){
					session.rollback();
					if (table!=null) Utility.drop(this.tableName).execute(session); 
				}
				this.setStatus(Status.Failed);
				this.store();
			}catch(Exception e1){
				logger.warn("clean table failed");
			}
			
			return false;
		}finally{
			if (session!=null) session.release();
		}
		return true;
	}
	
	public boolean remove(){
		try {
			Utility.drop(this.getTableName());
			ObjectPersistency.get(CodeListImport.class).deleteByKey(id);
		} catch (Exception e) {
			return false;
		}
		return true;
	}
	
	private SimpleTable createTable(final List<ColumnDefinition> tableDefintion, String tableName, DBSession session) throws Exception{
		SimpleTable toReturn= null;
		CreateTable create= DBSession.getImplementation(CreateTable.class);
		create.setTableName(tableName);
		create.setColumnsDefinition(tableDefintion.toArray(new ColumnDefinition[tableDefintion.size()]));
		toReturn= create.execute(session);
		return toReturn;
	}
	
	public static CodeListImport get(String id) throws Exception{
		return ObjectPersistency.get(CodeListImport.class).getByKey(id);
	}
	
	public static Iterator<CodeListImport> getAll() throws Exception{
		return ObjectPersistency.get(CodeListImport.class).getAll().iterator();
	}
	
			
}
