package org.gcube.contentmanagement.timeseriesservice.impl.codelist;

import java.util.ArrayList;
import java.util.Arrays;
import  java.util.Iterator;
import java.util.List;

import org.gcube.common.core.faults.GCUBEFault;
import org.gcube.common.core.faults.GCUBERetryEquivalentFault;
import org.gcube.common.core.types.VOID;
import org.gcube.common.core.utils.logging.GCUBELog;
import org.gcube.common.dbinterface.Limit;
import org.gcube.common.dbinterface.Order;
import org.gcube.common.dbinterface.Order.OrderType;
import org.gcube.common.dbinterface.attributes.SimpleAttribute;
import org.gcube.common.dbinterface.persistence.ObjectNotFoundException;
import org.gcube.common.dbinterface.pool.DBSession;
import org.gcube.common.dbinterface.queries.Select;
import org.gcube.common.dbinterface.types.Type;
import org.gcube.contentmanagement.codelistmanager.entities.TableField;
import org.gcube.contentmanagement.codelistmanager.entities.TableField.ColumnType;
import org.gcube.contentmanagement.codelistmanager.managers.CodeListCuration;
import org.gcube.contentmanagement.codelistmanager.managers.CodeListImport;
import org.gcube.contentmanagement.codelistmanager.util.CodeListType;
import org.gcube.contentmanagement.codelistmanager.util.ColumnReference;
import org.gcube.contentmanagement.timeseriesservice.impl.codelist.wrappers.CLCurationWrapper;
import org.gcube.contentmanagement.timeseriesservice.impl.codelist.wrappers.CLImportItemWrapper;
import org.gcube.contentmanagement.timeseriesservice.impl.context.ServiceContext;
import org.gcube.contentmanagement.timeseriesservice.impl.exceptions.EditNotFinishedException;
import org.gcube.contentmanagement.timeseriesservice.impl.exceptions.NotInEditModeException;
import org.gcube.contentmanagement.timeseriesservice.impl.utils.Util;
import org.gcube.contentmanagement.timeseriesservice.stubs.CLUnderCurationItem;
import org.gcube.contentmanagement.timeseriesservice.stubs.CLUnderCurationItemsArray;
import org.gcube.contentmanagement.timeseriesservice.stubs.ChangeColumnTypeRequest;
import org.gcube.contentmanagement.timeseriesservice.stubs.CheckNullValuesOnDataTypeChangeRequest;
import org.gcube.contentmanagement.timeseriesservice.stubs.EnterInEditModeRequest;
import org.gcube.contentmanagement.timeseriesservice.stubs.ReplaceValueRequest;
import org.gcube.contentmanagement.timeseriesservice.stubs.SetCLLabelRequest;
import org.gcube.contentmanagement.timeseriesservice.stubs.StartCurationModeRequest;
import org.gcube.contentmanagement.timeseriesservice.stubs.RemoveRowsRequest;
import org.gcube.contentmanagement.timeseriesservice.stubs.RemoveColumnRequest;
import org.gcube.contentmanagement.timeseriesservice.stubs.SetPropertiesRequest;
import org.gcube.contentmanagement.timeseriesservice.stubs.codelist.types.CodelistColumnDefinition;
import org.gcube.contentmanagement.timeseriesservice.stubs.codelist.types.CodelistColumnType;
import org.gcube.contentmanagement.timeseriesservice.stubs.codelist.types.CodelistColumnsDefinitionArray;
import org.gcube.contentmanagement.timeseriesservice.stubs.codelist.types.CodelistIdentifier;
import org.gcube.contentmanagement.timeseriesservice.stubs.codelist.types.DataAsJsonRequest;
import org.gcube.contentmanagement.timeseriesservice.stubs.types.AccessFault;
import org.gcube.contentmanagement.timeseriesservice.stubs.types.ItemNotFoundFault;
import org.gcube.contentmanagement.timeseriesservice.stubs.types.NotInEditModeFault;
import org.gcube.contentmanagement.timeseriesservice.stubs.types.Status;


public class CodelistCurationManager {

	private static GCUBELog logger = new GCUBELog(CodelistCurationManager.class);
	
	/**
	 * 
	 * @param request
	 * @return
	 * @throws GCUBEFault
	 */
	public String startCurationMode(StartCurationModeRequest request) throws GCUBEFault {
		final CodeListImport importer;
		try{
			CLImportItemWrapper importWrapper = CLImportItemWrapper.get(request.getImportId());
			if (!importWrapper.getOwner().equals(request.getUser()) || !ServiceContext.getContext().getScope().toString().equals(importWrapper.getScope().toString()))
				throw new AccessFault();
			importer = importWrapper.getCodelist();
		}catch (ObjectNotFoundException e) {
			throw new ItemNotFoundFault();
		} catch (Exception e) {
			throw new GCUBEFault(e);
		}	
		
			
		final CodeListCuration curation = new CodeListCuration(importer.getName(), importer.getDescription(), importer.getAgencyId(), importer.isFinal(), importer.getVersion(),
				importer.getCodelistType()==CodeListType.Unknown?CodeListType.valueOf(request.getCodeListType().toString()):importer.getCodelistType());
		curation.store();
		final CLCurationWrapper curationWrapper = new CLCurationWrapper(request.getUser(), curation.getId(), ServiceContext.getContext().getScope());
		curationWrapper.store();
		final Thread curationThread = new Thread(){
			public void run(){
				curationWrapper.setStatus(org.gcube.contentmanagement.timeseriesservice.stubs.types.Status.Open);
				curationWrapper.store();
				if (curation.start(importer))
					curationWrapper.setStatus(org.gcube.contentmanagement.timeseriesservice.stubs.types.Status.Close);
				else curationWrapper.setStatus(org.gcube.contentmanagement.timeseriesservice.stubs.types.Status.Error);
				curationWrapper.store();
			}
		};
		try {
			ServiceContext.getContext().useServiceCredentials(curationThread);
		} catch (Exception e) {
			logger.error("error starting curation",e);
			throw new GCUBEFault(e);
		}
		
		curationThread.start();
	
		return curation.getId();
	}
	
	/**
	 * 
	 * @param request
	 * @return
	 * @throws GCUBEFault
	 */
	public String getDataAsJson(DataAsJsonRequest request) throws GCUBEFault{
		try {
			CLCurationWrapper clCurationWrapper = CLCurationWrapper.get(request.getIdentifier().getId());
			if (!clCurationWrapper.getScope().toString().equals(ServiceContext.getContext().getScope().toString()) || !request.getIdentifier().getUser().equals(clCurationWrapper.getOwner())) throw new AccessFault();
			Select selectQuery= DBSession.getImplementation(Select.class);
			selectQuery.setLimit(new Limit(request.getLimit().getLowerLimit(), request.getLimit().getUpperLimit()));
			selectQuery.setOrders(new Order((request.getOrder().getOrder()==org.gcube.contentmanagement.timeseriesservice.stubs.types.OrderType.Ascending)?OrderType.ASC:OrderType.DESC, new SimpleAttribute(request.getOrder().getField())));
			return clCurationWrapper.getDataAsJson(selectQuery);
		}catch (ObjectNotFoundException e) {
			logger.error("item not found", e);
			throw new ItemNotFoundFault();
		} catch (Exception e1) {
			logger.error("error getting data as json",e1);
			throw new GCUBEFault(e1);
		}
	}
	

	/**
	 * 
	 * @param user
	 * @return
	 * @throws GCUBEFault
	 */
	public CLUnderCurationItemsArray getUnderCurationCodelists(String user) throws GCUBEFault{
		List<CLUnderCurationItem> clItems = new ArrayList<CLUnderCurationItem>();
		
		logger.debug("calling  getUnderCurationCodelists with user "+user);
		
		try {
			Iterator<CLCurationWrapper> clCurationWrapperIterator = CLCurationWrapper.getByUser(user);
			
			while (clCurationWrapperIterator.hasNext()){
				CLCurationWrapper wrapperItem = clCurationWrapperIterator.next();
				logger.debug("scope is null ? "+(wrapperItem.getScope()==null));
				logger.debug("service context is null ? "+(ServiceContext.getContext()==null));
				if (wrapperItem.getScope().equals(ServiceContext.getContext().getScope())){
					try{
						CodeListCuration item = wrapperItem.getCodelist();
						clItems.add(new CLUnderCurationItem(item.isFinal(), item.getAgencyId(), org.gcube.contentmanagement.timeseriesservice.stubs.codelist.types.CodeListType.fromString(item.getCodelistType().toString()), item.getCreationDate(), item.getDescription(),  item.getId(),
							item.getLastModifyDate(), item.getName(), wrapperItem.getOwner(), item.getCount(), wrapperItem.getStatus(), item.getVersion()));
					}catch (ObjectNotFoundException e) {
						logger.warn("the wrapper with id "+wrapperItem.getCodelistId()+" is not bound to a codeListCuration");
						wrapperItem.remove();
					}
				}
			}
		} catch (Exception e) {
			logger.error("error getting imported items",e);
			throw new GCUBEFault("error retrieving codelist imported items");
		}
		return new CLUnderCurationItemsArray(clItems.toArray(new CLUnderCurationItem[clItems.size()]));
	
	}
	
	/**
	 * 
	 * @param identifier
	 * @return
	 * @throws GCUBEFault
	 */
	public CLUnderCurationItem getCLUnderCurationItem(CodelistIdentifier identifier) throws GCUBEFault{
		try {
			CLCurationWrapper clCurationWrapper = CLCurationWrapper.get(identifier.getId());
			if (!clCurationWrapper.getScope().toString().equals(ServiceContext.getContext().getScope().toString()) || !identifier.getUser().equals(clCurationWrapper.getOwner())) throw new AccessFault();
			CodeListCuration item = clCurationWrapper.getCodelist();
			return new CLUnderCurationItem(item.isFinal(), item.getAgencyId() , org.gcube.contentmanagement.timeseriesservice.stubs.codelist.types.CodeListType.fromString(item.getCodelistType().toString()), item.getCreationDate(), item.getDescription(),  item.getId(),
					item.getLastModifyDate(), item.getName(), clCurationWrapper.getOwner(), item.getCount(), clCurationWrapper.getStatus(),  item.getVersion());
		}catch (ObjectNotFoundException e) {
			logger.error("item not found", e);
			throw new ItemNotFoundFault();
		} catch (Exception e1) {
			logger.error("error getting imported item with id "+identifier.getId(),e1);
			throw new GCUBEFault(e1);
		}
	}
	
	public boolean checkCodeListCurationFinished(CodelistIdentifier identifier) throws GCUBEFault{
		try {
			CLCurationWrapper clCurationWrapper = CLCurationWrapper.get(identifier.getId());
			if (!clCurationWrapper.getScope().toString().equals(ServiceContext.getContext().getScope().toString()) || !identifier.getUser().equals(clCurationWrapper.getOwner())) throw new AccessFault();
			if (clCurationWrapper.isUnderEditing()) return false;
			CodeListCuration item = clCurationWrapper.getCodelist();
			return item.isMappingFinished();
		}catch (ObjectNotFoundException e) {
			logger.error("item not found", e);
			throw new ItemNotFoundFault();
		} catch (Exception e1) {
			logger.error("error getting imported item with id "+identifier.getId(),e1);
			throw new GCUBEFault(e1);
		}
	}
	
	/**
	 * 
	 * @param identifier
	 * @return
	 * @throws GCUBEFault
	 */
	public CodelistColumnsDefinitionArray getColumnsDefinition(CodelistIdentifier identifier) throws GCUBEFault{
		List<CodelistColumnDefinition> columnDefinitionList= new ArrayList<CodelistColumnDefinition>();
		try {
			CLCurationWrapper clCurationWrapper = CLCurationWrapper.get(identifier.getId());
			if (!clCurationWrapper.getScope().toString().equals(ServiceContext.getContext().getScope().toString()) || !identifier.getUser().equals(clCurationWrapper.getOwner())) throw new AccessFault();
			TableField[] returnList = clCurationWrapper.getCodelist().getLabelFieldMapping().values().toArray(new TableField[0]);
			Arrays.sort(returnList);
			for (TableField cd : returnList)
				columnDefinitionList.add(new CodelistColumnDefinition(CodelistColumnType.fromString(cd.getColumnReference().getType().toString()),Util.mapSqlToJava(cd.getDataType()),  cd.getId(), cd.getFieldName(), cd.getColumnReference().getCodelistReferenceId()));
			return new CodelistColumnsDefinitionArray(columnDefinitionList.toArray(new CodelistColumnDefinition[columnDefinitionList.size()]));
		}catch (ObjectNotFoundException e) {
			logger.error("item not found", e);
			throw new ItemNotFoundFault();
		} catch (Exception e1) {
			logger.error("error getting column definition",e1);
			throw new GCUBEFault(e1);
		}
	}
	
	/**
	 * 
	 * @param identifier
	 * @return
	 * @throws GCUBEFault
	 */
	public boolean remove(CodelistIdentifier identifier) throws GCUBEFault{
		try {
			CLCurationWrapper clCurationWrapper = CLCurationWrapper.get(identifier.getId());
			if (!clCurationWrapper.getScope().toString().equals(ServiceContext.getContext().getScope().toString()) || !identifier.getUser().equals(clCurationWrapper.getOwner())) throw new AccessFault();
			return clCurationWrapper.remove();
		}catch (ObjectNotFoundException e) {
			logger.error("item not found", e);
			throw new ItemNotFoundFault();
		} catch (Exception e1) {
			logger.error("error removing imported codelist",e1);
			throw new GCUBEFault(e1);
		}
	}
	
	/**
	 * 
	 * @param request
	 * @return
	 * @throws GCUBEFault
	 */
	public synchronized VOID changeColumnType(ChangeColumnTypeRequest request) throws GCUBEFault{
		try {
			logger.debug("changing columnType");
			CLCurationWrapper clCurationWrapper = CLCurationWrapper.get(request.getIdentifier().getId());
			if (!clCurationWrapper.getScope().toString().equals(ServiceContext.getContext().getScope().toString()) || !request.getIdentifier().getUser().equals(clCurationWrapper.getOwner())) throw new AccessFault();
			CodeListCuration codelist= clCurationWrapper.getCodelist();
			codelist.changeColumnType(request.getFieldId(), ColumnType.valueOf(request.getColumnType().toString()), request.getRelatedCodeListId());
			if (!codelist.store()) throw new GCUBERetryEquivalentFault();
			return new VOID();
		}catch (ObjectNotFoundException e) {
			logger.error("item not found", e);
			throw new ItemNotFoundFault();
		}  catch (GCUBEFault f1) {
			logger.error("error changing column type",f1);
			throw  f1;
		}	catch (Exception e1) {
			logger.error("error changing column type",e1);
			throw new GCUBEFault(e1);
		}	
	}
	
	/**
	 * 
	 * @param request
	 * @return
	 * @throws GCUBEFault
	 */
	public int checkNullValuesOnDataTypeChange(CheckNullValuesOnDataTypeChangeRequest request) throws GCUBEFault{
		try {
			CLCurationWrapper clCurationWrapper = CLCurationWrapper.get(request.getIdentifier().getId());
			if (!clCurationWrapper.getScope().toString().equals(ServiceContext.getContext().getScope().toString()) || !request.getIdentifier().getUser().equals(clCurationWrapper.getOwner())) throw new AccessFault();
			if (request.getColumnType()==null)
				return clCurationWrapper.getCodelist().getInvalidRows(request.getFieldId(), new Type(Util.mapJavaToSql(request.getDataType()))).length;
			else return clCurationWrapper.getCodelist().getInvalidRows(request.getFieldId(), new ColumnReference(ColumnType.valueOf(request.getColumnType().toString()), null,  request.getCodelistReferenceId())).length;
		}catch (ObjectNotFoundException e) {
			logger.error("item not found", e);
			throw new ItemNotFoundFault();
		} catch (Exception e1) {
			logger.error("error checking the curation",e1);
			throw new GCUBEFault(e1);
		}
	}
	
	/**
	 * 
	 * @param request
	 * @return
	 * @throws GCUBEFault
	 */
	public synchronized VOID enterInEditMode(EnterInEditModeRequest request) throws GCUBEFault{
		try {
			logger.debug("entering in edit mode");
			CLCurationWrapper clCurationWrapper = CLCurationWrapper.get(request.getIdentifier().getId());
			if (!clCurationWrapper.getScope().toString().equals(ServiceContext.getContext().getScope().toString()) || !request.getIdentifier().getUser().equals(clCurationWrapper.getOwner())) throw new AccessFault();
			if (request.getDataType() ==null)
				clCurationWrapper.edit(request.getFieldId());
			else clCurationWrapper.edit(request.getFieldId(), new Type(Util.mapJavaToSql(request.getDataType())));
		}catch (ObjectNotFoundException e) {
			logger.error("item not found", e);
			throw new ItemNotFoundFault();
		} catch (Exception e1) {
			logger.error("error checking the curation",e1);
			throw new GCUBEFault(e1);
		}
		return new VOID();
	}
	
	
	/**
	 * 
	 * @param identifier
	 * @return
	 * @throws GCUBEFault
	 */
	public synchronized VOID saveEdit(CodelistIdentifier identifier) throws GCUBEFault{
		try {
			logger.debug("saving Edit for id "+identifier.getId());
			CLCurationWrapper clCurationWrapper = CLCurationWrapper.get(identifier.getId());
			if (!clCurationWrapper.getScope().toString().equals(ServiceContext.getContext().getScope().toString()) || !identifier.getUser().equals(clCurationWrapper.getOwner())) throw new AccessFault();
			clCurationWrapper.saveEdit();
		}catch (ObjectNotFoundException e) {
			logger.error("item not found saving edit", e);
			e.printStackTrace();
			throw new ItemNotFoundFault();
		}catch (EditNotFinishedException enf) {
			logger.error("edit not finished exception", enf);
			throw new GCUBEFault("the Edit is not yet complete"); 
		}catch (NotInEditModeException nie) {
			logger.error("not in edit mode", nie);
			throw new NotInEditModeFault(); 
		} catch (Exception e1) {
			logger.error("error checking the curation",e1);
			throw new GCUBEFault(e1);
		}
		return new VOID();
	}
	
	
	/**
	 * 
	 * @param identifier
	 * @return
	 * @throws GCUBEFault
	 */
	public synchronized VOID removeEdit(CodelistIdentifier identifier) throws GCUBEFault {
		try {
			CLCurationWrapper clCurationWrapper = CLCurationWrapper.get(identifier.getId());
			if (!clCurationWrapper.getScope().toString().equals(ServiceContext.getContext().getScope().toString()) || !identifier.getUser().equals(clCurationWrapper.getOwner())) throw new AccessFault();
			clCurationWrapper.removeEdit();
			clCurationWrapper.store();
		}catch (ObjectNotFoundException e) {
			logger.error("item not found", e);
			throw new ItemNotFoundFault();
		}catch (NotInEditModeException nie) {
			logger.error("not in edit mode", nie);
			throw new NotInEditModeFault(); 
		} catch (Exception e1) {
			logger.error("error checking the curation",e1);
			throw new GCUBEFault(e1);
		}
		return new VOID();
	}
	
	
	/**
	 * 
	 * @param identifier
	 * @return
	 * @throws GCUBEFault
	 */
	public Status getEditingStatus(CodelistIdentifier identifier) throws GCUBEFault{
		try {
			CLCurationWrapper clCurationWrapper = CLCurationWrapper.get(identifier.getId());
			if (!clCurationWrapper.getScope().toString().equals(ServiceContext.getContext().getScope().toString()) || !identifier.getUser().equals(clCurationWrapper.getOwner())) throw new AccessFault();
			return clCurationWrapper.getEditStatus();
		}catch (ObjectNotFoundException e) {
			logger.error("item not found", e);
			throw new ItemNotFoundFault();
		} catch (Exception e1) {
			logger.error("error checking the curation",e1);
			throw new GCUBEFault(e1);
		}
		
	}
	
	/**
	 * 
	 * @param identifier
	 * @return
	 * @throws GCUBEFault
	 */
	public boolean isUnderEditing(CodelistIdentifier identifier) throws GCUBEFault{
		try {
			CLCurationWrapper clCurationWrapper = CLCurationWrapper.get(identifier.getId());
			if (!clCurationWrapper.getScope().toString().equals(ServiceContext.getContext().getScope().toString()) || !identifier.getUser().equals(clCurationWrapper.getOwner())) throw new AccessFault();
			return clCurationWrapper.isUnderEditing();
		}catch (ObjectNotFoundException e) {
			logger.error("item not found", e);
			throw new ItemNotFoundFault();
		} catch (Exception e1) {
			logger.error("error checking the curation",e1);
			throw new GCUBEFault(e1);
		}
	}
	
	/**
	 * 
	 * @param request
	 * @return
	 * @throws GCUBEFault
	 */
	public synchronized VOID replaceValue(ReplaceValueRequest request) throws GCUBEFault {
		try {
			CLCurationWrapper clCurationWrapper = CLCurationWrapper.get(request.getIdentifier().getId());
			if (!clCurationWrapper.getScope().toString().equals(ServiceContext.getContext().getScope().toString()) || !request.getIdentifier().getUser().equals(clCurationWrapper.getOwner())) throw new AccessFault();
			clCurationWrapper.replaceValue(request.getFieldId(),request.getRowId(), request.getNewValue());
		}catch (ObjectNotFoundException e) {
			logger.error("item not found replacing value", e);
			throw new ItemNotFoundFault();
		} catch (NotInEditModeException nie) {
			logger.error("not in edit mode", nie);
			throw new NotInEditModeFault(); 
		}catch (Exception e1) {
			logger.error("error checking the curation",e1);
			throw new GCUBEFault(e1);
		}
		return new VOID();
	}
	
	/**
	 * 
	 * @param request
	 * @return
	 * @throws GCUBEFault
	 */
	public VOID setLabel(SetCLLabelRequest request) throws GCUBEFault{
		try {
			CLCurationWrapper clCurationWrapper = CLCurationWrapper.get(request.getIdentifier().getId());
			if (!clCurationWrapper.getScope().toString().equals(ServiceContext.getContext().getScope().toString()) || !request.getIdentifier().getUser().equals(clCurationWrapper.getOwner())) throw new AccessFault();
			CodeListCuration codelist = clCurationWrapper.getCodelist();
			codelist.getLabelFieldMapping().get(request.getFieldId()).setFieldName(request.getLabel());
			codelist.store();
		}catch (ObjectNotFoundException e) {
			logger.error("item not found setting label", e);
			throw new ItemNotFoundFault();
		} catch (Exception e1) {
			logger.error("error checking the curation",e1);
			throw new GCUBEFault(e1);
		}
		return new VOID();
	}
	
	/**
	 * 
	 * @param identifier
	 * @return
	 * @throws GCUBEFault
	 */
	public Status getCurationStatus(CodelistIdentifier identifier) throws GCUBEFault{
		try {
			CLCurationWrapper clCurationWrapper = CLCurationWrapper.get(identifier.getId());
			if (!clCurationWrapper.getScope().toString().equals(ServiceContext.getContext().getScope().toString()) || !identifier.getUser().equals(clCurationWrapper.getOwner())) throw new AccessFault();
			return clCurationWrapper.getStatus();
		}catch (ObjectNotFoundException e) {
			logger.error("item not found getting the curation status", e);
			throw new ItemNotFoundFault();
		} catch (Exception e1) {
			logger.error("error checking the curation",e1);
			throw new GCUBEFault(e1);
		}
	}
	
	public synchronized VOID removeAllErrors(CodelistIdentifier identifier) throws GCUBEFault{
		try {
			CLCurationWrapper clCurationWrapper = CLCurationWrapper.get(identifier.getId());
			if (!clCurationWrapper.getScope().toString().equals(ServiceContext.getContext().getScope().toString()) || !identifier.getUser().equals(clCurationWrapper.getOwner())) throw new AccessFault();
			clCurationWrapper.removeAllErrors();
		}catch (ObjectNotFoundException e) {
			logger.error("item not found removing all error", e);
			throw new ItemNotFoundFault();
		} catch (NotInEditModeException nie) {
			logger.error("not in edit mode", nie);
			throw new NotInEditModeFault(); 
		} catch (Exception e1) {
			logger.error("error checking the curation",e1);
			throw new GCUBEFault(e1);
		}
		return new VOID();
	}
	
	/**
	 * 
	 * @param identifier
	 * @return
	 * @throws GCUBEFault
	 */
	public synchronized CodelistColumnDefinition getColumnDefinitionUnderEdit(CodelistIdentifier identifier) throws GCUBEFault{
		try {
			CLCurationWrapper clCurationWrapper = CLCurationWrapper.get(identifier.getId());
			if (!clCurationWrapper.getScope().toString().equals(ServiceContext.getContext().getScope().toString()) || !identifier.getUser().equals(clCurationWrapper.getOwner())) throw new AccessFault();
			TableField tf = clCurationWrapper.getColumnInEditMode();
			return new CodelistColumnDefinition(CodelistColumnType.fromString(tf.getColumnReference().getType().toString()),Util.mapSqlToJava(tf.getDataType()), tf.getId(), tf.getFieldName(), tf.getColumnReference().getCodelistReferenceId());
		}catch (ObjectNotFoundException e) {
			logger.error("item not found getting column under edit", e);
			throw new ItemNotFoundFault();
		}catch (NotInEditModeException nie) {
			logger.error("not in edit mode", nie);
			throw new NotInEditModeFault(); 
		} catch (Exception e1) {
			logger.error("error checking the curation",e1);
			throw new GCUBEFault(e1);
		}
	}
	
	
	/**
	 * 
	 * @return
	 * @throws Exception
	 */
	public synchronized int getErrorCount(CodelistIdentifier identifier) throws GCUBEFault{
		try {
			CLCurationWrapper clCurationWrapper = CLCurationWrapper.get(identifier.getId());
			if (!clCurationWrapper.getScope().toString().equals(ServiceContext.getContext().getScope().toString()) || !identifier.getUser().equals(clCurationWrapper.getOwner())) throw new AccessFault();
			return clCurationWrapper.getErrorCount();
		}catch (ObjectNotFoundException e) {
			logger.error("item not found getting error count", e);
			throw new ItemNotFoundFault();
		}catch (NotInEditModeException nie) {
			logger.error("not in edit mode", nie);
			throw new NotInEditModeFault(); 
		} catch (Exception e1) {
			logger.error("error checking the curation",e1);
			throw new GCUBEFault(e1);
		}
	}
	
	/**
	 * 
	 * @return
	 * @throws Exception
	 */
	public int getSize(CodelistIdentifier identifier) throws GCUBEFault{
		try {
			CLCurationWrapper clCurationWrapper = CLCurationWrapper.get(identifier.getId());
			if (!clCurationWrapper.getScope().toString().equals(ServiceContext.getContext().getScope().toString()) || !identifier.getUser().equals(clCurationWrapper.getOwner())) throw new AccessFault();
			return clCurationWrapper.getSize();
		}catch (ObjectNotFoundException e) {
			logger.error("item not found getting the size", e);
			throw new ItemNotFoundFault();
		} catch (Exception e1) {
			logger.error("error checking the curation",e1);
			throw new GCUBEFault(e1);
		}
	}
	
	public String getCodes(DataAsJsonRequest request) throws GCUBEFault{
		try {
			CLCurationWrapper clCurationWrapper = CLCurationWrapper.get(request.getIdentifier().getId());
			if (!clCurationWrapper.getScope().toString().equals(ServiceContext.getContext().getScope().toString()) || !request.getIdentifier().getUser().equals(clCurationWrapper.getOwner())) throw new AccessFault();
			Order order= new Order((request.getOrder().getOrder()==org.gcube.contentmanagement.timeseriesservice.stubs.types.OrderType.Ascending)?OrderType.ASC:OrderType.DESC, new SimpleAttribute(request.getOrder().getField()));
			Limit limit = new Limit(request.getLimit().getLowerLimit(), request.getLimit().getUpperLimit());
			return clCurationWrapper.getCodesAsJson(limit, order);
		}catch (ObjectNotFoundException e) {
			logger.error("item not found getting the size", e);
			throw new ItemNotFoundFault();
		} catch (Exception e1) {
			logger.error("error checking the curation",e1);
			throw new GCUBEFault(e1);
		}
	}
	
	public boolean removeRows(RemoveRowsRequest request) throws GCUBEFault{
		try {
			CLCurationWrapper clCurationWrapper = CLCurationWrapper.get(request.getIdentifier().getId());
			if (!clCurationWrapper.getScope().toString().equals(ServiceContext.getContext().getScope().toString()) || !request.getIdentifier().getUser().equals(clCurationWrapper.getOwner())) throw new AccessFault();
			//TODO: implement it
			return false;
		}catch (ObjectNotFoundException e) {
			logger.error("item not found getting the size", e);
			throw new ItemNotFoundFault();
		} catch (Exception e1) {
			logger.error("error removing rows",e1);
			throw new GCUBEFault(e1);
		}
	}
	
	public boolean removeColumn(RemoveColumnRequest request) throws GCUBEFault{
		try {
			CLCurationWrapper clCurationWrapper = CLCurationWrapper.get(request.getIdentifier().getId());
			if (!clCurationWrapper.getScope().toString().equals(ServiceContext.getContext().getScope().toString()) || !request.getIdentifier().getUser().equals(clCurationWrapper.getOwner())) throw new AccessFault();
			CodeListCuration codelist = clCurationWrapper.getCodelist();
			if (clCurationWrapper.isUnderEditing()) throw new Exception("the remove column cannot be done in error edit mode ");
			boolean toReturn = codelist.removeColumn(request.getFieldId());
			codelist.store();
			return toReturn;
		}catch (ObjectNotFoundException e) {
			logger.error("item not found getting the size", e);
			throw new ItemNotFoundFault();
		} catch (Exception e1) {
			logger.error("error removing column",e1);
			throw new GCUBEFault(e1);
		}
	}
	
	public VOID setProperties(SetPropertiesRequest request) throws GCUBEFault{
		try {
			CLCurationWrapper clCurationWrapper = CLCurationWrapper.get(request.getIdentifier().getId());
			if (!clCurationWrapper.getScope().toString().equals(ServiceContext.getContext().getScope().toString()) || !request.getIdentifier().getUser().equals(clCurationWrapper.getOwner())) throw new AccessFault();
			CodeListCuration codelist = clCurationWrapper.getCodelist();
			codelist.setName(request.getName());
			codelist.setDescription(request.getDescription());
			codelist.setAgencyId(request.getAgencyId());
			codelist.setFinal(request.isIsFinal());
			codelist.setVersion(request.getVersion());
			codelist.store();
			return new VOID();
		}catch (ObjectNotFoundException e) {
			logger.error("item not found getting the size", e);
			throw new ItemNotFoundFault();
		} catch (Exception e1) {
			logger.error("error setting properties",e1);
			throw new GCUBEFault(e1);
		}
	}
}
