/**
 * 
 */
package org.gcube.data.gml.elements;

import static org.gcube.data.gml.constants.Labels.*;

import java.util.Iterator;
import java.util.List;

import javax.xml.bind.annotation.XmlElement;
import javax.xml.bind.annotation.XmlRootElement;
import javax.xml.bind.annotation.XmlTransient;

/**
 * An annotation for a {@link GCubeDocument}.
 * 
 * @author Federico De Faveri defaveri@isti.cnr.it
 * @author Fabio Simeoni (University of Strathclyde)
 *
 */
@XmlRootElement
public class GCubeAnnotation extends BaseInnerElement {

	@XmlElement(name=PREVIOUS)
	private String previousID;

	@XmlTransient
	GCubeAnnotation previous;

	/**
	 * Creates an instance.
	 */
	public GCubeAnnotation() {} 

	/**
	 * Creates an instance that act as a proxy for a given annotation.
	 * @param id the annotation identifier.
	 */
	public GCubeAnnotation(String id) {
		super(id);
	}

	/**{@inheritDoc}*/
	@Override
	public void postBinding(GCubeDocument d) throws IllegalStateException, Exception {
		super.postBinding(d);
		if (previousID()!=null && d.annotations().contains(previousID()))
			_setPrevious(d.annotations().get(previousID()));
	}

	/**
	 * Sets the annotation that precedes this annotation in an annotation thread.
	 * @param a the previous annotation, or <code>null</code> to unlink this annotation from any other.
	 * @throws IllegalStateException if this annotation is not associated with a document or is a proxy of an existing annotation.
	 * @throws IllegalArgumentException if the input annotation is new or is bound to a different document than this annotation's.
	 */
	public void setPrevious(GCubeAnnotation a) throws IllegalStateException, IllegalArgumentException {

		//for time being, we allow unlinking with null parameters.
		if (a!=null) {
			if (document()==null)
				throw new IllegalArgumentException("cannot link an annotation that is not associated with a document");

			if(!isNew())
				throw new IllegalStateException("cannot link an annotation proxy, such as "+id());
			
			if(a.isNew())
				throw new IllegalArgumentException("cannot link to a new annotation, such as "+a);

			if(!document().annotations().contains(a.id()))
				throw new IllegalArgumentException("cannot link to an annotation in an other document, such as \n"+a);
		}
		
		_setPrevious(a);
		
	}
	
	
	public void _setPrevious(GCubeAnnotation a)  {

		if (a!=null) 
			previousID = a.id();
		previous = a;
	}

	/**
	 * Returns the annotation that precedes this annotation in an annotation thread, if available in the containing document.
	 * @return the previous part, or <code>null</code> if this annotation does not follow another in an annotation thread or
	 * if it does but is not available in the containing document.
	 */
	public GCubeAnnotation previous() {
		return previous;
	}

	/**
	 * Returns the identifier of the annotation that precedes this element in an annotation thread.
	 * @return the identifier, or <code>null</code> if this annotation does not follow another in an annotation thread.
	 */
	public String previousID() {
		return previousID;
	}	

	/**{@inheritDoc}*/ 
	@Override
	public String toString() {
		StringBuilder builder = new StringBuilder(super.toString());
		builder.append(", previous=");
		builder.append(previous()!=null?previous():previousID);
		builder.append("]");
		return builder.toString();
	}

	@Override
	public int hashCode() {
		final int prime = 31;
		int result = super.hashCode();
		result = prime * result
		+ ((previous == null) ? 0 : previous.hashCode());
		return result;
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (!super.equals(obj))
			return false;
		if (!(obj instanceof GCubeAnnotation))
			return false;
		GCubeAnnotation other = (GCubeAnnotation) obj;
		if (previousID == null) {
			if (other.previousID != null)
				return false;
		} else if (!previousID.equals(other.previousID))
			return false;
		return true;
	}
	
	protected static String toString(List<GCubeAnnotation> annotations, boolean asThread)
	{
		StringBuilder sb = new StringBuilder();
		sb.append("[");
		Iterator<GCubeAnnotation> iterator = annotations.iterator();
		while(iterator.hasNext()){
			sb.append(iterator.next().id());
			if (iterator.hasNext()) sb.append(asThread?"<-":",");
		}
		sb.append("]");
		return sb.toString();
	}
}
