/**
 * 
 */
package org.gcube.data.gml.elements;

import static org.gcube.data.gml.constants.Labels.*;

import javax.xml.bind.annotation.XmlElement;
import javax.xml.bind.annotation.XmlRootElement;
import javax.xml.bind.annotation.XmlTransient;

/**
 * A part of a {@link GCubeDocument}.
 * 
 *@author Federico De Faveri defaveri@isti.cnr.it
 *@author Fabio Simeoni (University of Strathclyde)
 */
@XmlRootElement
public class GCubePart extends BaseInnerElement {
	
	@XmlElement(name = ORDER)
	private int order;

	@XmlTransient
	GCubePart previous;
	

	//see base class for an explanation of constructors
	
	public GCubePart() {} 
	
	public GCubePart(String id) {
		super(id);
	}
	
	/**
	 * Returns the order of the part within the document.
	 * @return the order.
	 */
	public int order() {
		return order;
	}
	
	/**
	 * Sets the order of the part within the document.
	 * @param o the order.
	 * @throws IllegalStateException if the part is not associated with a document.
	 */
	public void setOrder(int o) throws IllegalArgumentException,IllegalStateException {
		if (document()==null)
			throw new IllegalStateException("cannot set order on a part that is not associated with a document, such as "+this);
		order=o;
		try{
			findPart();
		}
		catch(IllegalStateException e) { //convert to right exception in this context
			throw new IllegalArgumentException(e.getMessage());
		}
	}
	
	/**{@inheritDoc}*/
	@Override
	public void postBinding(GCubeDocument d) throws IllegalStateException, Exception {
		super.postBinding(d);
		findPart();
	}
	
	//used internally
	private void findPart() throws IllegalStateException {
		if (order()>1) {
			for (GCubePart p : document().parts())
				if (p.order()==order)
					throw new IllegalStateException("part order clashes with "+p);
				else if (p.order()==order()-1) {
					previous=p;
					return;
				}
			
			throw new IllegalArgumentException("no part in order "+(order()-1));
		}
	}

	/**
	 * Returns the part that precedes this part.
	 * @return the previous part, or <code>null</code> if this part has no previous
	 * or if the previous part is not available in the containing document.
	 */
	public GCubePart previous() {
		return previous;
	}
	
	/**
	 * {@inheritDoc}
	 */ 
	@Override
	public String toString() {
		StringBuilder builder = new StringBuilder(super.toString());
		builder.append(", order=");
		builder.append(order);
		builder.append("]");
		return builder.toString();
	}

	@Override
	public int hashCode() {
		final int prime = 31;
		int result = super.hashCode();
		result = prime * result + order;
		return result;
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (!super.equals(obj))
			return false;
		if (!(obj instanceof GCubePart))
			return false;
		GCubePart other = (GCubePart) obj;
		if (order != other.order)
			return false;
		return true;
	}
	
	
}
