/**
 * 
 */
package org.gcube.data.gml.elements;

import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;

/**
 * Group of typed elements of the document.
 * @author Fabio Simeoni (University of Strathclyde)
 *
 * @param <T> the element type.
 */
public class InnerElements<T extends BaseInnerElement> implements Iterable<T> {
	
	GCubeDocument inner;
	List<T> elements;
	
	InnerElements(GCubeDocument doc,List<T> es) {
		inner = doc;
		elements =es;
	}
	
	/**
	 * Adds an element to the document, binding the element to the document.
	 * InnerElements that have identifiers replace elements with the same identifier already in the document, if any.
	 * @param e the element.
	 * @return the element replaced by this element, or <code>null</code> if no element was replaced.
	 * @throws IllegalArgumentException if the element is not new (has an identifier) but the document is already tracked for changes.
	 * @throws IllegalStateException if the element is already bound to another document
	 * */
	public T add(T e) throws IllegalArgumentException, IllegalStateException {
		
		T replaced = null;
		
		//new element?
		if (e.isNew()) {
			
			if (!inner.isNew() && !inner.isTracked()) //can only be added to new documents or to untracked proxies
				throw new IllegalArgumentException("cannot add new elements to a document which is not tracked for changes");
		}
		//old element
		else {
		
			if (inner.isNew()) //cannot be added to new document
				throw new IllegalArgumentException("cannot add element proxies to new document");
			
			else if (inner.isTracked()) //cannot be added to tracked element 
				throw new IllegalArgumentException("cannot add elements proxy to a document which is tracked for changes");
			
			else
				for (T t : elements) 
					if (t.id().equals(e.id())) {
						replaced=t;
						break;
					}
		}
		
		if (replaced!=null)
			elements.remove(replaced);
		
		elements.add(e);
		
		e.setDocument(inner);
	
		return replaced;
	}

	/**
	 * Returns an element.
	 * @param id the element's identifier.
	 * @return the element.
	 * @throws IllegalStateException if an element with the given identifier does not exist.
	 */
	public T get(String id) throws IllegalStateException {
		for (T e : elements)
			if (e.id().equals(id))
				return e;
		
		throw new IllegalStateException("unknown element with identifier "+id);
	}
	
	/**
	 * Indicates the existence of an element.
	 * @param id the element's identifier.
	 * @return <code>true</code> if an element with the given identifier exists, <code>false</code> otherwise.
	 */
	public boolean contains(String id) {
		try {
			get(id);
			return true;
		}
		catch(IllegalStateException e) {
			return false;
		}
	}
	
	/**
	 * Removes an element.
	 * @param id the element's identifier.
	 * @return the removed element.
	 * @throws IllegalStateException if an element with the given identifier does not exist.
	 */
	public T remove(String id) throws IllegalStateException {
		T removed = get(id);
		remove(removed);
		return removed;
	}
	
	/**
	 * Removes an element from the document.
	 * @param e the element.
	 * @throws IllegalStateException if the element does not exist.
	 **/
	public void remove(T e) throws IllegalStateException {
		if (!elements.remove(e))
			throw new IllegalStateException("unknown element "+e);
	}
	
	/**
	 * Returns the elements.
	 * @return the elements. 
	 */
	public List<T> toList() {
		return new LinkedList<T>(elements); //clone
	}
	
	/**{@inheritDoc}*/
	public Iterator<T> iterator() {
		return elements.iterator();
	}
	
	/**
	 * Returns the number of elements.
	 * @return the number of elements
	 */
	public int size() {
		return elements.size();
	}
	
	/**{@inheritDoc}*/
	@Override
	public String toString() {
		return elements.toString();
	}
	
}
