package org.apache.jackrabbit.j2ee.workspacemanager;

import java.util.HashMap;
import java.util.Map;

import javax.jcr.ItemNotFoundException;
import javax.jcr.Node;
import javax.jcr.PathNotFoundException;
import javax.jcr.Property;
import javax.jcr.PropertyIterator;
import javax.jcr.RepositoryException;
import javax.jcr.Session;

import org.apache.jackrabbit.util.Text;
import org.gcube.common.homelibary.model.items.ItemDelegate;
import org.gcube.common.homelibary.model.items.type.NodeProperty;
import org.gcube.common.homelibary.model.items.type.PrimaryNodeType;
import org.gcube.common.homelibary.model.util.WorkspaceItemAction;
import org.gcube.common.homelibrary.model.exceptions.InternalErrorException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.thoughtworks.xstream.XStream;

public class JCRWorkspaceItem{

	public static final String HL_NAMESPACE					= "hl:";
	public static final String JCR_NAMESPACE				= "jcr:";
	public static final String REP_NAMESPACE				= "rep:";
	private static final String NT_NAMESPACE 				= "nt:";
	private static final String SEPARATOR 					= "/";

	protected static Logger logger = LoggerFactory.getLogger(JCRWorkspaceItem.class);

	protected ItemDelegate item;

	public JCRWorkspaceItem(Node node, String login) throws RepositoryException {

		item = new ItemDelegate();


		item.setId(node.getIdentifier());
		item.setName(node.getName());
		item.setPath(node.getPath());
		item.setLocked(node.isLocked());

		item.setPrimaryType(node.getPrimaryNodeType().getName());

		try{
			item.setParentId(node.getParent().getIdentifier());
		}catch (ItemNotFoundException e) {
			logger.info("Root node doesn't have a parent");
		}

		try{			
			if (node.hasProperty(NodeProperty.TITLE.toString()))
				item.setTitle(node.getProperty(NodeProperty.TITLE.toString()).getString());
			if (node.hasProperty(NodeProperty.DESCRIPTION.toString()))
				item.setDescription(node.getProperty(NodeProperty.DESCRIPTION.toString()).getString());
			if (node.hasProperty(NodeProperty.CREATED.toString()))
				item.setCreationTime(node.getProperty(NodeProperty.CREATED.toString()).getDate());
			if (node.hasProperty(NodeProperty.LAST_ACTION.toString()))
				item.setLastAction(WorkspaceItemAction.valueOf(node.getProperty(NodeProperty.LAST_ACTION.toString()).getString()));
			if (node.hasProperty(NodeProperty.LAST_MODIFIED_BY.toString()))
				item.setLastModifiedBy(node.getProperty(NodeProperty.LAST_MODIFIED_BY.toString()).getString());
			if (node.hasProperty(NodeProperty.LAST_MODIFIED.toString()))
				item.setLastModificationTime(node.getProperty(NodeProperty.LAST_MODIFIED.toString()).getDate());

			try{
				item.setMetadata(getMetadata(node));
			}catch (Exception e) {
				logger.info("Error setting metadata");
			}

			item.setLocked(node.isLocked());

		}catch (Exception e) {
			logger.error("error setting basic properties on node: " + node.getPath(), e);
		}

		try {
			item.setProperties(new HashMap<NodeProperty, String>());

		} catch (Exception e) {
			logger.error("error setting metadata", e);
		}


		String parentPath = null;
		try {
			String idSharedFolder = getIdSharedFolder(node);
			if (idSharedFolder!=null){
				item.setShared(true);
				item.getProperties().put(NodeProperty.SHARED_ROOT_ID, idSharedFolder);
				//				System.out.println("idSharedFolder " + idSharedFolder );

				try{
					Session session = node.getSession();
					Node sharedNode = session.getNodeByIdentifier(idSharedFolder);
					//					System.out.println("sharedNode " + sharedNode.getPath());
					Map<String, String> users = getMembers(sharedNode);

					item.getProperties().put(NodeProperty.USERS, new XStream().toXML(users));

					String value = users.get(login);
					//					System.out.println("value " + value);
					parentPath = getUserParentPath(value, node.getSession());

					item.setParentPath(parentPath);
					//					System.out.println("path " + parentSharedId);
					//					Node parentSharedNode = session.getNodeByIdentifier(parentSharedId);
					//					System.out.println(parentSharedNode.getPath());

				} catch (Exception e) {
					logger.info(" Error setting parent path");
				}
				//				try {
				//					item.setParentPath(getUserParentPath(users.get(login), node.getSession()));
				//				} catch (Exception e) {
				//					logger.info(" Error setting parent path");
				//				}

			}else
				item.setShared(false);
		} catch (Exception e) {
			logger.error("error setting isShared");
		}	


		if (parentPath==null)
			item.setParentPath(node.getPath());

		Node nodeOwner;
		//get Owner
		try{
			item.setOwner(node.getProperty(NodeProperty.PORTAL_LOGIN.toString()).getString());
		}catch (Exception e) {
			try {
				nodeOwner = node.getNode(NodeProperty.OWNER.toString());
				item.setOwner(nodeOwner.getProperty(NodeProperty.USER_ID.toString()).getString());
				//				this.userId = nodeOwner.getProperty(USER_ID).getString();
				//				this.portalLogin = nodeOwner.getProperty(PORTAL_LOGIN).getString();
				//				node.getSession().save();
			} catch (PathNotFoundException e1) {
				logger.error("error setting owner");
			}

		}	
	}


	private String getUserParentPath(String value, Session session) throws Exception {
		String[] values = value.split(SEPARATOR);
		if (values.length < 2)
			throw new Exception("Path node corrupt");

		String parentId = values[0];
		String nodeName = values[1];

		Node parentNode = session.getNodeByIdentifier(parentId);

		return parentNode.getPath() + SEPARATOR + nodeName;
	}

	private Map<String, String> getMembers(Node node) {
		Map<String, String> list = new HashMap<String, String>();
		try {
			Node usersNode = node.getNode(NodeProperty.USERS.toString());
			for (PropertyIterator iterator = usersNode.getProperties(); iterator.hasNext();) {
				Property property  = iterator.nextProperty();
				String key = property.getName();
				String value = property.getString();
				if (!(key.startsWith(JCR_NAMESPACE)) &&
						!(key.startsWith(HL_NAMESPACE)) &&
						!(key.startsWith(NT_NAMESPACE)))
					list.put(key, value);
			} 
		} catch (RepositoryException e) {
			logger.error("error getting members node");
		} 
		return list;

	}

	public ItemDelegate getItemDelegate() {
		return item;
	}

	/**
	 * Get the id Shared Folder Root
	 * @param node
	 * @return the id Shared Folder Root
	 * @throws InternalErrorException
	 * @throws RepositoryException
	 */
	public String getIdSharedFolder(Node node) throws InternalErrorException, RepositoryException {
		if (node.getPrimaryNodeType().getName().equals(PrimaryNodeType.NT_WORKSPACE_SHARED_FOLDER)){
			return node.getIdentifier();
		}
		if (isRoot(node)) {
			return null;
		}
		return  getIdSharedFolder(node.getParent());
	}


	/**
	 * Check if the node is root in Jackrabbit
	 * @param node
	 * @return true if the node is root
	 * @throws InternalErrorException
	 * @throws ItemNotFoundException
	 * @throws RepositoryException
	 */
	public boolean isRoot(Node node) throws InternalErrorException, RepositoryException { 
		Node parent = null;
		try{
			parent = node.getParent();
		}catch (ItemNotFoundException e){
			//			logger.error("Root node doesn't have a parent");
			return true;
		}
		return (parent == null);
	}


	private Map<String,String> getMetadata(Node itemNode) throws InternalErrorException, RepositoryException {
		Map<String,String> map = new HashMap<String,String>();
		try {

			Node propertiesNode = itemNode.getNode(NodeProperty.METADATA.toString());

			for (PropertyIterator iterator = propertiesNode.getProperties();
					iterator.hasNext();) {
				Property property = iterator.nextProperty();
				if(!property.getName().startsWith("jcr:")) {
					String unescapeName = Text.unescape(property.getName());
					map.put(unescapeName,
							property.getValue().getString());
				}
			}

			return map;
		} catch (PathNotFoundException e) {
			if (itemNode!= null) {
				try {
					itemNode.addNode(NodeProperty.METADATA.toString());
					itemNode.getSession().save();
				} catch (Exception e1) {
					logger.error("Error adding metadata node to " + itemNode.getPath());
					//					throw new InternalErrorException(e1.getMessage());
				}
			} else {
				throw new InternalErrorException(e.getMessage());
			}
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		}
		return map; 

	}

}
