package org.gcube.data.tml.clients;

import static org.gcube.data.streams.dsl.Streams.*;
import static org.gcube.data.tml.utils.Utils.*;
import static org.gcube.data.trees.patterns.Patterns.*;
import static org.gcube.data.trees.streams.TreeStreams.*;

import java.net.URI;
import java.net.URISyntaxException;
import java.util.Arrays;

import org.apache.axis.message.addressing.EndpointReferenceType;
import org.gcube.common.clients.Client;
import org.gcube.common.clients.DiscoveryException;
import org.gcube.common.clients.gcore.GCoreDiscoveryClient;
import org.gcube.common.clients.gcore.GCorePassThroughClient;
import org.gcube.common.clients.gcore.StatefulQuery;
import org.gcube.common.core.faults.GCUBEFault;
import org.gcube.common.core.faults.GCUBEUnrecoverableException;
import org.gcube.data.streams.Stream;
import org.gcube.data.tm.stubs.GetByIDParams;
import org.gcube.data.tm.stubs.GetByIDsParams;
import org.gcube.data.tm.stubs.GetParams;
import org.gcube.data.tm.stubs.InvalidTreeFault;
import org.gcube.data.tm.stubs.Path;
import org.gcube.data.tm.stubs.TReaderPortType;
import org.gcube.data.tm.stubs.UnknownPathFault;
import org.gcube.data.tm.stubs.UnknownTreeFault;
import org.gcube.data.tm.stubs.UnsupportedOperationFault;
import org.gcube.data.tm.stubs.UnsupportedRequestFault;
import org.gcube.data.tml.clients.APIUtils.TReaderCall;
import org.gcube.data.tml.clients.providers.TReaderProvider;
import org.gcube.data.tml.clients.queries.QueryBuilder;
import org.gcube.data.tml.exceptions.InvalidTreeException;
import org.gcube.data.tml.exceptions.UnknownPathException;
import org.gcube.data.tml.exceptions.UnknownTreeException;
import org.gcube.data.tml.exceptions.UnsupportedOperationException;
import org.gcube.data.tml.exceptions.UnsupportedRequestException;
import org.gcube.data.tml.uri.URIs;
import org.gcube.data.tml.utils.TMStreams;
import org.gcube.data.trees.data.Node;
import org.gcube.data.trees.data.Tree;
import org.gcube.data.trees.io.Bindings;
import org.gcube.data.trees.patterns.Pattern;
import org.oasis.wsrf.faults.BaseFaultType;
import org.oasis.wsrf.lifetime.Destroy;
import org.w3c.dom.Element;

/**
 * A {@link TReaderClient} implementation.
 * 
 * @author Fabio Simeoni
 *
 */
public class ReaderClient implements TReaderClient  {

	//inner service client that executes calls
	private final Client<TReaderPortType> client;

	
	public ReaderClient(Client<TReaderPortType> client) {
		this.client=client;
	}

	/**
	 * Creates an instance for reading from a data source discovered with a given query.
	 * @param query the query.
	 */
	public ReaderClient(StatefulQuery query) {
		this(new GCoreDiscoveryClient<TReaderPortType>(TReaderProvider.INSTANCE,query));
	}
	
	/**
	 * Creates an instance for reading from a given data source.
	 * @param sourceId the source identifier
	 */
	public ReaderClient(String sourceId) {
		this(QueryBuilder.findReadSource().withId(sourceId).build());
	}
	
	/**
	 * Creates an instance to read from a service at a given endpoint.
	 * @param endpoint the service endpoint
	 */
	public ReaderClient(EndpointReferenceType endpoint) {
		this(new GCorePassThroughClient<TReaderPortType>(TReaderProvider.INSTANCE,endpoint));
	}
	
	/**
	 * Resolves a content URI in a scope and, optionally, with a given security manager.
	 * @param uri the URI
	 * @return the {@link Node} identified by the URI
	 * @throws IllegalArgumentException if the input scope is <code>null</code>
	 * @throws URISyntaxException if the URI is not a content URI
	 * @throws DiscoveryException if a resolver for the URI cannot be found in the given scope
	 * @throws UnsupportedOperationException if the target plugin does not support the operation
	 * @throws UnknownPathException if the URI cannot be resolved to a tree or tree node
	 * @throws Exception if the operation fails for an unexpected error
	 */
	public static Node resolve(URI uri) throws IllegalArgumentException, URISyntaxException, DiscoveryException, UnsupportedOperationException, UnknownPathException, Exception {
		
		String sourceId = URIs.sourceId(uri);
		TReaderClient call = new ReaderClient(sourceId);
		
		return call.getNode(URIs.nodeIDs(uri));
	}

	
	@Override
	public Element getAsElement(final String id, final Pattern pattern) throws UnknownTreeException,UnsupportedOperationException, DiscoveryException, Exception {
		
		TReaderCall<Element> call = new TReaderCall<Element>() {
			public Element call(TReaderPortType service) throws Exception {
				
				GetByIDParams params = new GetByIDParams(toHolder(pattern),id);
				
				try {
					return toElement(service.getByID(params)); 
				} 
				catch(UnsupportedOperationFault f) {
					throw new UnsupportedOperationException("operation is not supported by target plugin");
				}
				catch(UnsupportedRequestFault f) {
					throw new UnsupportedRequestException("request is not supported by target plugin");
				}
				catch(UnknownTreeFault f) {
					throw new UnknownTreeException(id+" does not identify a tree",f.toException().getCause());
				}
				catch(InvalidTreeFault f) {
					throw new InvalidTreeException(pattern+" does not match tree with identifier "+id,f.toException().getCause());
				}
				catch(GCUBEFault f) {
					throw f.toException(); 
				}
			}
		};
		
		return client.make(call);
		
	}
	
	@Override
	public Tree get(String id) throws UnknownTreeException, DiscoveryException,UnsupportedOperationException, Exception {
		return get(id,tree());
	}
	
	@Override
	public Tree get(final String id, Pattern pattern) throws UnknownTreeException, DiscoveryException, UnsupportedOperationException, Exception {	
		return Bindings.fromElement(this.getAsElement(id,pattern));
	}
	
	@Override
	public Element getAsElement(String id) throws DiscoveryException, UnsupportedOperationException, UnknownTreeException,  Exception {
		return getAsElement(id,tree());
	}
	
	
	@Override
	public Stream<Tree> get(final URI loc, final Pattern pattern) throws DiscoveryException,UnsupportedOperationException,Exception {
		
		TReaderCall<String> call = new TReaderCall<String>() {
			public String call(TReaderPortType service) throws Exception {
				
				GetByIDsParams params = new GetByIDsParams(loc.toString(),toHolder(pattern));
				
				try {
					return service.getByIDs(params); 
				}
				catch(UnsupportedOperationFault f) {
					throw new UnsupportedOperationException("operation is not supported by target plugin");
				}
				catch(UnsupportedRequestFault f) {
					throw new UnsupportedRequestException("request is not supported by target plugin");
				}
				catch(GCUBEFault f) {
					throw f.toException(); 
				}
			}
		};
		
		String locator = client.make(call); 
		
		return treesIn(URI.create(locator));
			
	}
	
	@Override
	public Stream<Tree> get(Stream<String> identifiers, Pattern pattern) throws DiscoveryException,UnsupportedOperationException,Exception {
		
		URI rs = publishStringsIn(identifiers).withDefaults();
		return get(rs,pattern);
	}
	
	@Override
	public Stream<Tree> get(URI loc) throws DiscoveryException, UnsupportedOperationException,Exception {
		   return get(loc,tree());
	}
	
	@Override
	public Stream<Tree> get(Stream<String> identifiers) throws DiscoveryException, UnsupportedOperationException,Exception {
		 return get(identifiers,tree());
	}
	
	@Override
	public Stream<Tree> get(final Pattern pattern) throws DiscoveryException, UnsupportedOperationException,Exception {
		
		TReaderCall<String> call = new TReaderCall<String>() {
			public String call(TReaderPortType service) throws Exception {
				try {
					return service.get(new GetParams(toHolder(pattern))); 
				}
				catch(UnsupportedOperationFault f) {
					throw new UnsupportedOperationException("operation is not supported by target plugin");
				}
				catch(UnsupportedRequestFault f) {
					throw new UnsupportedRequestException("request is not supported by target plugin");
				}
				catch(GCUBEFault f) {
					throw f.toException(); 
				}
			}
		};
		
		String locator = client.make(call); 
		return treesIn(URI.create(locator));
	}
		
	@Override
	public Stream<Tree> get() throws DiscoveryException,UnsupportedOperationException,Exception {
		return get(tree());
	}
	
	
	@Override
	public Element getNodeAsElement(final String ... ids) throws IllegalArgumentException,DiscoveryException,UnsupportedOperationException, UnknownPathException, Exception {
		
		if(ids==null || ids.length==0)
			throw new IllegalArgumentException("no ids specified");
		
		TReaderCall<Element> call = new TReaderCall<Element>() {
			public Element call(TReaderPortType service) throws Exception {
				try {
					return toElement(service.getNode(new Path(ids))); 
				} 
				catch(UnsupportedOperationFault f) {
					throw new UnsupportedOperationException("operation is not supported by target plugin");
				}
				catch(UnsupportedRequestFault f) {
					throw new UnsupportedRequestException("request is not supported by target plugin");
				}
				catch(UnknownPathFault f) {
					throw new UnknownPathException(Arrays.asList(ids)+" do not identify a node",f.toException().getCause());
				}
				catch(GCUBEFault f) {
					throw f.toException(); 
				}
			}
		};
		
		
		return client.make(call);
		
	}
	
	@Override
	public Node getNode(String ... path) throws IllegalArgumentException,DiscoveryException, UnsupportedOperationException,UnknownPathException, Exception {
		return Bindings.nodeFromElement(getNodeAsElement(path));
	}

	@Override
	public Stream<Node> getNodes(final Stream<Path> paths) throws DiscoveryException,UnsupportedOperationException,Exception {
		URI pathRs = TMStreams.publishPathsIn(paths).withDefaults();
		return getNodes(pathRs);
	}
	
	@Override
	public Stream<Node> getNodes(final URI paths) throws DiscoveryException,UnsupportedOperationException,Exception {
		
		TReaderCall<String> call = new TReaderCall<String>() {
			public String call(TReaderPortType service) throws Exception {
				try {
					return service.getNodes(paths.toString()); 
				}
				catch(UnsupportedOperationFault f) {
					throw new UnsupportedOperationException("operation is not supported by target plugin");
				}
				catch(UnsupportedRequestFault f) {
					throw new UnsupportedRequestException("request is not supported by target plugin");
				}
				catch(GCUBEFault f) {
					throw f.toException(); 
				}
			}
		};
		
		String locator = client.make(call); 
		return nodesIn(URI.create(locator));
	}
	
	/**{@inheritDoc}*/
	synchronized void delete() throws Exception {
		
		TReaderCall<Void> call = new TReaderCall<Void>() {
			public Void call(TReaderPortType service) throws Exception {
				try{
					service.destroy(new Destroy());
					return null;
				}
				catch (BaseFaultType f) {
					throw new GCUBEUnrecoverableException(f);
				}
			}
		};
		
		client.make(call);
		
	}
	
}
