package org.gcube.data.tml.clients.queries;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.gcube.common.clients.gcore.StatefulQuery;
import org.gcube.data.tml.Constants;
import org.gcube.data.tml.clients.TBinderClient;
import org.gcube.data.tml.clients.providers.TBinderProvider;

/**
 * Builds queries for T-Binder services.
 * 
 * @author Fabio Simeoni
 * @see TBinderClient
 *
 */
public class BinderQueryBuilder {

	static private final String basePluginCondition = 
			"some $plugin in $result/child::*[local-name()='"+Constants.BINDER_PLUGIN_RPNAME+"'] satisfies (%1$s)";
	
	static private final String pluginCondition = 
			"$plugin/child::*[local-name()='%1$s'] eq '%2$s'";
	
	static private final String pluginPropertyCondition = 
			"some $prop in $plugin/child::*[local-name()='"+Constants.PROPERTY_RPNAME+"'] satisfies (%1$s)";
	
	static private final String propertyCondition = 
			"$prop/child::*[local-name()='%1$s'] eq '%2$s'";
	
	private String pluginName;
	private Map<String,String> properties = new HashMap<String, String>();

	
	protected BinderQueryBuilder() {}
	
	/**
	 * Builds the query.
	 * @return the query
	 */
	public StatefulQuery build() {
	
		StatefulQuery query = new StatefulQuery(TBinderProvider.INSTANCE);
		
		String condition = toString();
		if (!condition.isEmpty())
			query.query().addGenericCondition(condition);
		
		return query;
	}
	
	/**
	 * Sets the name of plugin on the query.
	 * @param name the plugin name
	 * @return this builder
	 */
	public BinderQueryBuilder withPlugin(String name) {
		this.pluginName=name;
		return this;
	}
	
	/**
	 * Sets the name of a plugin property on the query.
	 * @param name the property name
	 * @param value the property value
	 * @return this builder
	 */
	public BinderQueryBuilder withProperty(String name, String value) {
		this.properties.put(name,value);
		return this;
	}
	
	@Override
	public String toString() {
		
		List<String> conditions = new ArrayList<String>();
		
		if (pluginName!=null)
			conditions.add(String.format(pluginCondition,"name",pluginName));
		
		if (!properties.isEmpty()) {
			List<String> propConditions = new ArrayList<String>(); 
			for (Map.Entry<String,String> prop : properties.entrySet())
				propConditions.add(String.format(propertyCondition,prop.getKey(),prop.getValue()));
			conditions.add(String.format(pluginPropertyCondition,QueryBuilder.concatenate(propConditions)));
		}
		
		return conditions.isEmpty()?"":String.format(basePluginCondition,QueryBuilder.concatenate(conditions));
	}
}
