package org.gcube.data.tml.proxies;

import static org.gcube.common.clients.exceptions.FaultDSL.*;
import static org.gcube.data.tml.utils.Utils.*;
import static org.gcube.data.trees.patterns.Patterns.*;
import static org.gcube.data.trees.streams.TreeStreams.*;

import java.net.URI;

import org.gcube.common.clients.Call;
import org.gcube.common.clients.delegates.ProxyDelegate;
import org.gcube.common.clients.exceptions.ServiceException;
import org.gcube.common.clients.exceptions.UnsupportedRequestException;
import org.gcube.data.streams.Stream;
import org.gcube.data.streams.dsl.Streams;
import org.gcube.data.tm.stubs.AnyHolder;
import org.gcube.data.tm.stubs.GetByIDParams;
import org.gcube.data.tm.stubs.GetByIDsParams;
import org.gcube.data.tm.stubs.GetParams;
import org.gcube.data.tm.stubs.Path;
import org.gcube.data.tm.stubs.TReaderPortType;
import org.gcube.data.tml.exceptions.InvalidTreeException;
import org.gcube.data.tml.exceptions.UnknownPathException;
import org.gcube.data.tml.exceptions.UnknownTreeException;
import org.gcube.data.tml.utils.TMStreams;
import org.gcube.data.trees.data.Node;
import org.gcube.data.trees.data.Tree;
import org.gcube.data.trees.io.Bindings;
import org.gcube.data.trees.patterns.Pattern;
import org.w3c.dom.Element;

public class DefaultTReader implements TReader {

	private final ProxyDelegate<TReaderPortType> delegate;

	public DefaultTReader(ProxyDelegate<TReaderPortType> delegate) {
		this.delegate = delegate;
	}

	@Override
	public Tree get(String id) throws UnknownTreeException {

		try {
			return get(id,tree());
		}
		catch(InvalidTreeException e) {
			//cannot really happen under the tree() pattern, but we need to satisfy the compiler
			//without introducing false declarations in the signature
			throw again(e).asServiceException();
		}

	}

	@Override
	public Tree get(String id,Pattern pattern) throws UnknownTreeException, InvalidTreeException {

		Element element = this.getAsElement(id,pattern); 
		
		try {
			return Bindings.fromElement(element);
		}
		catch(Exception e) {
			throw again(e).asServiceException();
		}
	}
	
	private Element getAsElement(final String id, final Pattern pattern) throws UnknownTreeException, InvalidTreeException {
		
		notNull("identifier", id);
		notNull(pattern);
		
		final GetByIDParams params = new GetByIDParams(toHolder(pattern),id);
		
		Call<TReaderPortType,Element> call = new Call<TReaderPortType,Element>() {
			@Override
			public Element call(TReaderPortType endpoint) throws Exception {
				return toElement(endpoint.getByID(params));  
			}
		};
		
		try {
			
			Element e =  delegate.make(call);
			
			if (e==null)
				throw new ServiceException("unexpected null response");
			
			return e;
		}
		catch(Exception e) {
			
			throw again(e).as(UnknownTreeException.class,InvalidTreeException.class);
		}
		
	}
	

	@Override
	public Stream<Tree> get(Stream<String> ids,Pattern pattern) {
		
		URI locator = Streams.publishStringsIn(ids).withDefaults();
		
		return get(locator,pattern);
	}
	
	
	@Override
	public Stream<Tree> get(final URI locator, Pattern pattern) throws UnsupportedOperationException,
			UnsupportedRequestException, ServiceException {
		
		notNull("resultset locator",locator);
		notNull(pattern);
		
		final AnyHolder patternhHolder = toHolder(pattern);

		Call<TReaderPortType,String> call = new Call<TReaderPortType,String>() {
			@Override
			public String call(TReaderPortType endpoint) throws Exception {
				
				GetByIDsParams params = new GetByIDsParams(locator.toString(),patternhHolder);
				
				return endpoint.getByIDs(params);
			}
		};
		
		try {
			String outcomeLocator = delegate.make(call);
			
			if (outcomeLocator==null)
				throw new ServiceException("unexpected null response");
			
			return treesIn(URI.create(outcomeLocator));
		}
		catch(Exception e) {
			throw again(e).asServiceException();
		}
	}

	@Override
	public Stream<Tree> get(final Pattern pattern) {
		
		notNull(pattern);
		
		Call<TReaderPortType,String> call = new Call<TReaderPortType,String>() {
			@Override
			public String call(TReaderPortType endpoint) throws Exception {
				
				GetParams params = new GetParams(toHolder(pattern));
				
				return endpoint.get(params);
			}
		};
		
		try {
			
			String outcomeLocator = delegate.make(call);
			
			if (outcomeLocator==null)
				throw new ServiceException("unexpected null response");
			
			return treesIn(URI.create(outcomeLocator));
		}
		catch(Exception e) {
			throw again(e).asServiceException();
		}
	}
	
	private Element getNodeAsElement(final String... ids) throws UnknownPathException {
		
		notNull("identifiers",ids);
		
		Call<TReaderPortType,Element> call = new Call<TReaderPortType,Element>() {
			@Override
			public Element call(TReaderPortType service) throws Exception {
				return toElement(service.getNode(new Path(ids)));  
			}
		};
		
		try {
			Element e = delegate.make(call);
			
			if (e==null)
				throw new ServiceException("unexpected null response");
			
			return e;
		}
		catch(Exception e) {
			throw again(e).as(UnknownPathException.class);
		}
	}
	
	@Override
	public Node getNode(String... path) throws UnknownPathException {
		
		try {
			return Bindings.nodeFromElement(getNodeAsElement(path));
		}
		catch(Exception e) {
			throw again(e).as(UnknownPathException.class);
		}
	}
	
	@Override
	public Stream<Node> getNodes(Stream<Path> paths) {
		
		URI pathRs = TMStreams.publishPathsIn(paths).withDefaults();
		
		return getNodes(pathRs);
	}
	
	
	@Override
	public Stream<Node> getNodes(final URI paths) {
		
		Call<TReaderPortType,String> call = new Call<TReaderPortType,String>() {
			@Override
			public String call(TReaderPortType endpoint) throws Exception {
				return endpoint.getNodes(paths.toString());
			}
		};
		
		try {
			String locator = delegate.make(call); 
			
			return nodesIn(URI.create(locator));
		}
		catch(Exception e) {
			throw again(e).asServiceException();
		}
		
		
	}
}
