/**
 *
 */
package org.gcube.data.analysis.dminvocation;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;

import javax.xml.XMLConstants;
import javax.xml.bind.JAXBContext;
import javax.xml.bind.JAXBException;
import javax.xml.bind.Marshaller;
import javax.xml.bind.SchemaOutputResolver;
import javax.xml.bind.Unmarshaller;
import javax.xml.transform.Result;
import javax.xml.transform.stream.StreamResult;
import javax.xml.transform.stream.StreamSource;
import javax.xml.validation.Schema;
import javax.xml.validation.SchemaFactory;

import org.eclipse.persistence.jaxb.JAXBContextProperties;
import org.gcube.data.analysis.dminvocation.model.DataMinerInvocation;
import org.xml.sax.SAXException;



/**
 * The Class DataMinerInvocationManager.
 *
 * @author Francesco Mangiacrapa at ISTI-CNR (francesco.mangiacrapa@isti.cnr.it)
 * Dec 10, 2018
 */
public class DataMinerInvocationManager {

	private static DataMinerInvocationManager singleInstance = null;
    private JAXBContext jaxbContext;
	private Schema schema;

	/*
	 *
	 * JAXBContext is thread safe and should only be created once and reused to avoid the cost of initializing the metadata multiple times.
	 * Marshaller and Unmarshaller are not thread safe, but are lightweight to create and could be created per operation.
	*/

	/**
	 * Instantiates a new data miner invocation manager.
	 *
	 * @throws JAXBException the JAXB exception
	 * @throws IOException Signals that an I/O exception has occurred.
	 * @throws SAXException the SAX exception
	 */
	private DataMinerInvocationManager() throws JAXBException, IOException, SAXException{
		jaxbContext= JAXBContext.newInstance(DataMinerInvocation.class);
		schema = generateSchema();
	}


	/**
	 * Gets the single instance of DataMinerInvocationManager.
	 *
	 * @return single instance of DataMinerInvocationManager
	 * @throws JAXBException the JAXB exception
	 * @throws IOException Signals that an I/O exception has occurred.
	 * @throws SAXException the SAX exception
	 */
	public static DataMinerInvocationManager getInstance() throws JAXBException, IOException, SAXException {

		  if (singleInstance == null)
			  singleInstance = new DataMinerInvocationManager();

	        return singleInstance;
	}


	/**
	 * Marshaling.
	 *
	 * @param dmInvocation the dm invocation
	 * @param mediaType the media type see at {@link MediaType}
	 * @param validateModel the validate model. If true performs model validation against the xml schema generated for {@link DataMinerInvocation}
	 * @return the byte array output stream
	 * @throws JAXBException the JAXB exception
	 */
	public ByteArrayOutputStream marshaling(DataMinerInvocation dmInvocation, MediaType mediaType,  boolean validateModel) throws JAXBException
	{
		Marshaller jaxbMarshaller = jaxbContext.createMarshaller();

	    if(validateModel)
			jaxbMarshaller.setSchema(schema);

	    jaxbMarshaller.setProperty(Marshaller.JAXB_FORMATTED_OUTPUT, Boolean.TRUE);

	    if(mediaType==null)
	    	mediaType = MediaType.ApplicationXML;

	    switch (mediaType) {
		case ApplicationJSON:
			jaxbMarshaller.setProperty(JAXBContextProperties.JSON_INCLUDE_ROOT, true);
			jaxbMarshaller.setProperty(JAXBContextProperties.JSON_ATTRIBUTE_PREFIX, "@");
			break;
		case ApplicationXML:
		default:

		}

		jaxbMarshaller.setProperty(JAXBContextProperties.MEDIA_TYPE, mediaType.getMimeType());
	    ByteArrayOutputStream baos = new ByteArrayOutputStream();
	    jaxbMarshaller.marshal(dmInvocation, baos);
	    return baos;
	}


	/**
	 * Unmarshaling.
	 *
	 * @param dmInvocationXMLStream the dm invocation xml stream
	 * @param mediaType the media type see at {@link MediaType}
	 * @param validateModel the validate model. If true performs model validation against the xml schema generated for {@link DataMinerInvocation}
	 * @return the data miner invocation
	 * @throws JAXBException the JAXB exception
	 */
	public DataMinerInvocation unmarshaling(InputStream dmInvocationXMLStream, MediaType mediaType, boolean validateModel) throws JAXBException
	{
	    Unmarshaller jaxbUnmarshaller = jaxbContext.createUnmarshaller();

	    if(validateModel)
	    	jaxbUnmarshaller.setSchema(schema);

	    if(mediaType==null)
	    	mediaType = MediaType.ApplicationXML;

	    switch (mediaType) {
		case ApplicationJSON:
			jaxbUnmarshaller.setProperty(JAXBContextProperties.JSON_INCLUDE_ROOT, true);
			jaxbUnmarshaller.setProperty(JAXBContextProperties.JSON_ATTRIBUTE_PREFIX, "@");
			break;
		case ApplicationXML:
		default:

		}
	    jaxbUnmarshaller.setProperty(JAXBContextProperties.MEDIA_TYPE, mediaType.getMimeType());
	    return (DataMinerInvocation) jaxbUnmarshaller.unmarshal(dmInvocationXMLStream);

	}


	/**
	 * Generate schema.
	 *
	 * @return the schema
	 * @throws JAXBException the JAXB exception
	 * @throws IOException Signals that an I/O exception has occurred.
	 * @throws SAXException the SAX exception
	 */
	private Schema generateSchema() throws JAXBException, IOException, SAXException {

	    // generate schema
	    ByteArrayStreamOutputResolver schemaOutput = new ByteArrayStreamOutputResolver();
	    jaxbContext.generateSchema(schemaOutput);

	    // load schema
	    ByteArrayInputStream schemaInputStream = new ByteArrayInputStream(schemaOutput.getSchemaContent());
	    SchemaFactory sf = SchemaFactory.newInstance(XMLConstants.W3C_XML_SCHEMA_NS_URI);
	   return sf.newSchema(new StreamSource(schemaInputStream));
	}



	/**
	 * The Class ByteArrayStreamOutputResolver.
	 *
	 * @author Francesco Mangiacrapa at ISTI-CNR (francesco.mangiacrapa@isti.cnr.it)
	 * Dec 7, 2018
	 */
	private static class ByteArrayStreamOutputResolver extends SchemaOutputResolver {

	    private ByteArrayOutputStream schemaOutputStream;

	    /* (non-Javadoc)
    	 * @see javax.xml.bind.SchemaOutputResolver#createOutput(java.lang.String, java.lang.String)
    	 */
    	public Result createOutput(String namespaceURI, String suggestedFileName) throws IOException {

	        schemaOutputStream = new ByteArrayOutputStream();
	        StreamResult result = new StreamResult(schemaOutputStream);

	        // We generate single XSD, so generator will not use systemId property
	        // Nevertheless, it validates if it's not null.
	        result.setSystemId("");

	        return result;
	    }

	    /**
    	 * Gets the schema content.
    	 *
    	 * @return the schema content
    	 */
    	public byte[] getSchemaContent() {
	        return schemaOutputStream.toByteArray();
	    }
	}
}
