package org.gcube.data.analysis.statisticalmanager.proxies;

import static org.gcube.data.streams.dsl.Streams.publish;
import gr.uoa.di.madgik.grs.record.GenericRecord;
import gr.uoa.di.madgik.grs.record.GenericRecordDefinition;
import gr.uoa.di.madgik.grs.record.Record;
import gr.uoa.di.madgik.grs.record.RecordDefinition;
import gr.uoa.di.madgik.grs.record.field.FieldDefinition;
import gr.uoa.di.madgik.grs.record.field.StringField;
import gr.uoa.di.madgik.grs.record.field.StringFieldDefinition;

import java.io.File;
import java.net.URI;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import org.gcube.common.clients.Call;
import org.gcube.common.clients.delegates.AsyncProxyDelegate;
import org.gcube.common.clients.delegates.ProxyDelegate;
import org.gcube.common.clients.exceptions.FaultDSL;
import org.gcube.common.core.scope.GCUBEScope;
import org.gcube.common.core.types.VOID;
import org.gcube.common.scope.api.ScopeProvider;
import org.gcube.data.analysis.statisticalmanager.exception.ResourceNotFoundException;
import org.gcube.data.analysis.statisticalmanager.stubs.DataSpacePortType;
import org.gcube.data.analysis.statisticalmanager.stubs.SMCreateTableFromCSVRequest;
import org.gcube.data.analysis.statisticalmanager.stubs.SMCreateTableFromDataStreamRequest;
import org.gcube.data.analysis.statisticalmanager.stubs.SMCreatedTablesRequest;
import org.gcube.data.analysis.statisticalmanager.stubs.SMFiles;
import org.gcube.data.analysis.statisticalmanager.stubs.SMGetFilesRequest;
import org.gcube.data.analysis.statisticalmanager.stubs.SMImporters;
import org.gcube.data.analysis.statisticalmanager.stubs.SMImportersRequest;
import org.gcube.data.analysis.statisticalmanager.stubs.SMResources;
import org.gcube.data.analysis.statisticalmanager.stubs.SMTables;
import org.gcube.data.analysis.statisticalmanager.wrapper.RSWrapper;
import org.gcube.data.streams.Stream;
import org.gcube.data.streams.publishers.RecordFactory;
import org.gcube.dataanalysis.ecoengine.datatypes.enumtypes.TableTemplates;
import org.gcube_system.namespaces.data.analysis.statisticalmanager.types.SMFile;
import org.gcube_system.namespaces.data.analysis.statisticalmanager.types.SMImport;
import org.gcube_system.namespaces.data.analysis.statisticalmanager.types.SMResource;

import org.gcube.data.spd.model.binding.Bindings;
import org.gcube.data.spd.model.products.OccurrencePoint;

public class StatisticalManagerDefaultDataSpace implements
		StatisticalManagerDataSpace {

	private final AsyncProxyDelegate<DataSpacePortType> delegate;

	public StatisticalManagerDefaultDataSpace(
			ProxyDelegate<DataSpacePortType> delegate) {
		super();
		this.delegate = new AsyncProxyDelegate<DataSpacePortType>(delegate);
	}

	@Override
	public String createTableFromCSV(final File file, final boolean hasHeader, final String delimiter,
			final String comment, final String tableName, final TableTemplates tableTemplate,
			final String description, final String user) {

		String scope = ScopeProvider.instance.get();
		try {
			
			
			RSWrapper wrapper = new RSWrapper(GCUBEScope.getScope(scope));
			wrapper.add(file);
			final String locator = wrapper.getLocator().toString();

			Call<DataSpacePortType, String> call = new Call<DataSpacePortType, String>() {
				@Override
				public String call(DataSpacePortType endpoint) throws Exception {

					SMCreateTableFromCSVRequest request = new SMCreateTableFromCSVRequest();
					request.setUser(user);
					request.setFileName(file.getName());
					request.setDescription(description);
					request.setHasHeader(hasHeader);
					request.setDelimiter(delimiter);
					request.setCommentChar(comment);
					request.setRsLocator(locator);
					request.setTableName(tableName);
					request.setTableType(tableTemplate.toString());

					return String.valueOf(endpoint.createTableFromCSV(request));
				}
			};

			return delegate.make(call);
		} catch (Exception e) {
			throw FaultDSL.again(e).asServiceException();
		}

	}

	@Override
	public String createTableFromDataStream(Stream<OccurrencePoint> points,
			final String tableName, final String description, final String user) {

		final URI rs = publish(points).using(
				new RecordFactory<OccurrencePoint>() {

					@Override
					public Record newRecord(OccurrencePoint element)
							throws RuntimeException {

						GenericRecord gr = new GenericRecord();
						try {
							gr.setFields(new StringField[] { new StringField(
									Bindings.toXml(element)) });
						} catch (Exception e) {
							throw new RuntimeException(e);
						}

						return gr;
					}

					@Override
					public RecordDefinition[] definitions() {
						StringFieldDefinition fieldDefinition = new StringFieldDefinition(
								"result");
						RecordDefinition[] defs = new RecordDefinition[] { // A
																			// gRS
																			// can
																			// contain
																			// a
																			// number
																			// of
																			// different
																			// record
																			// definitions
						new GenericRecordDefinition((new FieldDefinition[] { // A
																				// record
																				// can
																				// contain
																				// a
																				// number
																				// of
																				// different
																				// field
																				// definitions
								fieldDefinition // The definition of the field
								})) };
						return defs;
					}

				}).withDefaults();

		Call<DataSpacePortType, String> call = new Call<DataSpacePortType, String>() {
			@Override
			public String call(DataSpacePortType endpoint) throws Exception {

				SMCreateTableFromDataStreamRequest request = new SMCreateTableFromDataStreamRequest();
				request.setUser(user);
				request.setDescription(description);
				request.setRsLocator(rs.toString());
				request.setTableName(tableName);
				request.setTableType(TableTemplates.OCCURRENCE_SPECIES
						.toString());

				return String.valueOf(endpoint
						.createTableFromDataStream(request));
			}
		};

		try {
			return delegate.make(call);
		} catch (Exception e) {
			throw FaultDSL.again(e).asServiceException();
		}
	}

	@Override
	public String getDBParameters(final String tableId) {

		Call<DataSpacePortType, String> call = new Call<DataSpacePortType, String>() {
			@Override
			public String call(DataSpacePortType endpoint) throws Exception {
				return endpoint.getDBParameters(tableId);
			}
		};

		try {
			return delegate.make(call);
		} catch (Exception e) {
			throw FaultDSL.again(e).asServiceException();
		}
	}

	private SMTables getTables(final SMCreatedTablesRequest request) {

		Call<DataSpacePortType, SMTables> call = new Call<DataSpacePortType, SMTables>() {

			@Override
			public SMTables call(DataSpacePortType endpoint) throws Exception {
				return endpoint.getTables(request);
			};
		};

		try {
			return delegate.make(call);
		} catch (Exception e) {
			throw FaultDSL.again(e).asServiceException();
		}
	}

	@Override
	public SMTables getTables(final String user) {

		SMCreatedTablesRequest request = new SMCreatedTablesRequest();
		request.setUser(user);
		return getTables(request);

	}

	@Override
	public SMTables getTables(final String user, final String template) {

		SMCreatedTablesRequest request = new SMCreatedTablesRequest();
		request.setUser(user);
		request.setTemplate(template);
		return getTables(request);
	}

	@Override
	public List<SMImport> getImports(final String user, final String template) {

		Call<DataSpacePortType, SMImporters> call = new Call<DataSpacePortType, SMImporters>() {

			@Override
			public SMImporters call(DataSpacePortType endpoint)
					throws Exception {
				return endpoint.getImporters(new SMImportersRequest(template,
						user));
			};
		};

		try {
			SMImporters imports = delegate.make(call);
			if (imports.getList() != null)
				return Arrays.asList(imports.getList());
			else
				return new ArrayList<SMImport>();		
			
		} catch (Exception e) {
			throw FaultDSL.again(e).asServiceException();
		}
	}

	@Override
	public SMImport getImporter(final String importId) {

		Call<DataSpacePortType, SMImport> call = new Call<DataSpacePortType, SMImport>() {

			@Override
			public SMImport call(DataSpacePortType endpoint)
					throws Exception {
				return endpoint.getImporter(importId);
			};
		};

		try {
			return delegate.make(call);
		} catch (Exception e) {
			throw FaultDSL.again(e).asServiceException();
		}

	}

	@Override
	public void removeImport(final String importerId) {
		Call<DataSpacePortType, VOID> call = new Call<DataSpacePortType, VOID>() {

			@Override
			public VOID call(DataSpacePortType endpoint)
					throws Exception {
				return endpoint.removeImporter(importerId);
			};
		};

		try {
			delegate.make(call);
		} catch (Exception e) {
			throw FaultDSL.again(e).asServiceException();
		}
		
	}

	@Override
	public List<SMResource> getResources(final String user, final String...template ) {
		
		Call<DataSpacePortType, SMResources> call = new Call<DataSpacePortType, SMResources>() {

			@Override
			public SMResources call(DataSpacePortType endpoint)
					throws Exception {
				SMCreatedTablesRequest request = new SMCreatedTablesRequest();
				request.setUser(user);
				if (template != null)
					request.setTemplate(template[0]);
				return endpoint.getResources(request);
			};
		};

		try {
			SMResources resources = delegate.make(call);
			if (resources.getList() != null)
				return Arrays.asList(resources.getList());
			else
				return new ArrayList<SMResource>();
		} catch (Exception e) {
			throw FaultDSL.again(e).asServiceException();
		}
	}


	@Override
	public void removeTable(final String tableId) {
		
		Call<DataSpacePortType, VOID> call = new Call<DataSpacePortType, VOID>() {

			@Override
			public VOID call(DataSpacePortType endpoint)
					throws Exception {
				return endpoint.removeTable(tableId);
			};
		};

		try {
			delegate.make(call);
		} catch (Exception e) {
			throw FaultDSL.again(e).asServiceException();
		}
	}

	@Override
	public File exportTable(final String tableId) throws ResourceNotFoundException{
		
		Call<DataSpacePortType, String> call = new Call<DataSpacePortType, String>() {

			@Override
			public String call(DataSpacePortType endpoint)
					throws Exception {
				return endpoint.exportTable(tableId);
			};
		};

		try {
			String locator = delegate.make(call);
			return RSWrapper.getStreamFromLocator(URI.create(locator));
		} catch (Exception e) {
			throw FaultDSL.again(e).as(ResourceNotFoundException.class);
		}
	}

	@Override
	public List<SMFile> getFiles(String user) {
		
		final SMGetFilesRequest request = new SMGetFilesRequest();
		request.setUser(user);
		
		Call<DataSpacePortType, SMFiles> call = new Call<DataSpacePortType, SMFiles>() {
			@Override
			public SMFiles call(DataSpacePortType endpoint) throws Exception {
				return endpoint.getFiles(request);
			};
		};

		try {
			SMFiles files = delegate.make(call);
			if (files.getList() != null)
				return Arrays.asList(files.getList());
			else
				return new ArrayList<SMFile>();		
		} catch (Exception e) {
			throw FaultDSL.again(e).asServiceException();
		}
	}

}
