package org.gcube.data.analysis.tabulardata.cube.tablemanagers.codelist;

import java.util.Collection;
import java.util.List;

import org.gcube.data.analysis.tabulardata.cube.data.DatabaseWrangler;
import org.gcube.data.analysis.tabulardata.cube.exceptions.TableCreationException;
import org.gcube.data.analysis.tabulardata.cube.metadata.CubeMetadataWrangler;
import org.gcube.data.analysis.tabulardata.cube.tablemanagers.DefaultTableCreator;
import org.gcube.data.analysis.tabulardata.cube.tablemanagers.TableManager;
import org.gcube.data.analysis.tabulardata.model.column.Column;
import org.gcube.data.analysis.tabulardata.model.column.ColumnType;
import org.gcube.data.analysis.tabulardata.model.column.IdColumn;
import org.gcube.data.analysis.tabulardata.model.idioms.ColumnIsOfType;
import org.gcube.data.analysis.tabulardata.model.table.Codelist;
import org.gcube.data.analysis.tabulardata.model.table.Table;

import com.google.common.collect.Collections2;
import com.google.common.collect.Lists;

public class CodelistCreator extends DefaultTableCreator {

	private List<Column> newColumns = Lists.newArrayList();

	// Following attributes are used for cloning
	private Table clonedCodelist = null;
	private List<Column> columnsToRemove = Lists.newArrayList();

	private boolean withData = false;

	public CodelistCreator(DatabaseWrangler dw, CubeMetadataWrangler cmw, TableManager tm) {
		super(dw, cmw, tm);
	}

	@Override
	protected Collection<Column> getAllColumns() {
		List<Column> columns = Lists.newArrayList();
		if (clonedCodelist != null) {
			Collection<Column> clonedColumns = Lists.newArrayList(clonedCodelist.getColumns());
			clonedColumns.removeAll(columnsToRemove);
			columns.addAll(clonedColumns);
		}
		columns.addAll(newColumns);
		// Make sure that IdColumn is not on the list
		while (columns.remove(new IdColumn()))
			;
		return columns;
	}

	protected Collection<Column> getNewColumns() {
		List<Column> columns = Lists.newArrayList();
		columns.addAll(newColumns);
		return columns;
	}

	protected void checkConsistency() throws TableCreationException {
		// Verify label uniqueness
		checkDuplicateLabels(getAllColumns());

		// Check that at least one CodeColumn is provided
		Collection<Column> codeColumns = Collections2.filter(getAllColumns(), new ColumnIsOfType(ColumnType.CODE));
		if (codeColumns.isEmpty())
			throw new TableCreationException("No column of type " + ColumnType.CODE + " was provided.");
		if (codeColumns.size() > 1)
			throw new TableCreationException("Too many column of type " + ColumnType.CODE + " were provided.");

		// Check that at least one AttributeColumn is provided
		Collection<Column> attributeColumns = Collections2.filter(getAllColumns(), new ColumnIsOfType(
				ColumnType.ATTRIBUTE));
		if (attributeColumns.isEmpty())
			throw new TableCreationException("No column of type " + ColumnType.ATTRIBUTE + " was provided.");
	}

	@Override
	protected boolean isAllowedColumn(Column column) {
		switch (column.getColumnType()) {
		case CODE:
		case ATTRIBUTE:
		case SYSTEM:
			return true;
		default:
			return false;
		}
	}

	@Override
	protected void addNewColumn(Column column) {
		newColumns.add(column);
	}

	@Override
	protected boolean isAllowedCloneableTable(Table table) {
		switch (table.getTableType()) {
		case CODELIST:
			return true;
		default:
			return false;
		}
	}

	@Override
	protected Table getTableToClone() {
		return clonedCodelist;
	}

	@Override
	protected void setTableToClone(Table table) {
		clonedCodelist = table;
	}

	@Override
	protected boolean isCloneWithData() {
		return withData;
	}

	@Override
	protected void setCloneWithData(boolean cloneWithData) {
		withData = cloneWithData;
	}

	@Override
	protected Collection<Column> getColumnsToRemove() {
		return columnsToRemove;
	}

	@Override
	protected void setColumnsToRemove(Collection<Column> columns) {
		columnsToRemove = Lists.newArrayList(columns);
	}

	@Override
	protected Table createBaseTable(String tableName, Collection<Column> columns) {
		return new Codelist(tableName, columns);
	}

	@Override
	protected void addIndexes(String tableName, Collection<Column> columns) {
		for (Column column : columns){
			switch(column.getColumnType()){
			case SYSTEM:
				return;
			default: 
				dbWrangler.createIndex(tableName, column.getName());
			}
			
		}
	}

}
