package org.gcube.data.analysis.tabulardata.operation.data.replace;

import java.sql.SQLException;

import org.gcube.data.analysis.tabulardata.cube.CubeManager;
import org.gcube.data.analysis.tabulardata.cube.data.connection.DatabaseConnectionProvider;
import org.gcube.data.analysis.tabulardata.expression.Expression;
import org.gcube.data.analysis.tabulardata.expression.MalformedExpressionException;
import org.gcube.data.analysis.tabulardata.expression.TableReferenceReplacer;
import org.gcube.data.analysis.tabulardata.expression.evaluator.EvaluatorException;
import org.gcube.data.analysis.tabulardata.expression.evaluator.sql.SQLExpressionEvaluatorFactory;
import org.gcube.data.analysis.tabulardata.model.column.Column;
import org.gcube.data.analysis.tabulardata.model.column.ColumnReference;
import org.gcube.data.analysis.tabulardata.model.table.Table;
import org.gcube.data.analysis.tabulardata.operation.SQLHelper;
import org.gcube.data.analysis.tabulardata.operation.invocation.OperationInvocation;
import org.gcube.data.analysis.tabulardata.operation.worker.exceptions.WorkerException;
import org.gcube.data.analysis.tabulardata.operation.worker.results.ImmutableWorkerResult;
import org.gcube.data.analysis.tabulardata.operation.worker.results.WorkerResult;
import org.gcube.data.analysis.tabulardata.operation.worker.types.DataWorker;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ReplaceByExpression extends DataWorker{

	private static final Logger log = LoggerFactory.getLogger(ReplaceByExpression.class);

	private CubeManager cubeManager;

	private DatabaseConnectionProvider connectionProvider;

	private SQLExpressionEvaluatorFactory sqlEvaluatorFactory;


	public ReplaceByExpression(OperationInvocation sourceInvocation,
			CubeManager cubeManager,
			DatabaseConnectionProvider connectionProvider,
			SQLExpressionEvaluatorFactory sqlEvaluatorFactory) {
		super(sourceInvocation);
		this.cubeManager = cubeManager;
		this.connectionProvider = connectionProvider;
		this.sqlEvaluatorFactory = sqlEvaluatorFactory;
	}

	private Table targetTable;
	private Column targetColumn;
	private Expression condition;
	private Expression value;

	private Table newTable;
	private Table diffTable;


	@Override
	protected WorkerResult execute() throws WorkerException {
		instantiateExecutionVariables();
		updateProgress(0.1f,"Initializing");
		diffTable = cubeManager.createTable(targetTable.getTableType()).addColumn(targetColumn).create();
		newTable = cubeManager.createTable(targetTable.getTableType()).like(targetTable, true).create();
		updateProgress(0.5f,"Updating");
		executeBatch();
		updateProgress(0.9f,"Finalizing");
		return new ImmutableWorkerResult(newTable, diffTable);
	}


	private void instantiateExecutionVariables()throws WorkerException{
		OperationInvocation invocation=getSourceInvocation();
		targetTable=cubeManager.getTable(invocation.getTargetTableId());
		targetColumn=targetTable.getColumnById(invocation.getTargetColumnId());
		condition=(Expression) invocation.getParameterInstances().get(ReplaceByExpressionFactory.CONDITION_PARAMETER.getIdentifier());
		value=(Expression) invocation.getParameterInstances().get(ReplaceByExpressionFactory.VALUE_PARAMETER.getIdentifier());
	}

	private String getUpdateStatement() throws WorkerException{
		//Actualize table references
		try{
			Expression actualCondition=updateTableReferences(condition);
			Expression actualValue=updateTableReferences(value);

			String stmt=String.format("WITH updated AS (UPDATE %1$s SET %2$s = %3$s WHERE %4$s RETURNING id) " +
					" INSERT INTO %5$s (id, %2$s) SELECT target.id, target.%2$s FROM %6$s as target, updated WHERE updated.id = target.id ", 
					newTable.getName(),targetColumn.getName(),
					sqlEvaluatorFactory.getEvaluator(actualValue).evaluate(),
					sqlEvaluatorFactory.getEvaluator(actualCondition).evaluate(),
					diffTable.getName(),
					targetTable.getName()
					);
			log.debug("To execute statement : "+stmt);
			return stmt;
		}catch(MalformedExpressionException e){
			throw new WorkerException("Expression is not well formed",e);
		}catch(EvaluatorException e){
			throw new WorkerException("Unable to evaluate expression",e);
		}
	}

	private void executeBatch()throws WorkerException{
		try {
			SQLHelper.executeSQLBatchCommands(connectionProvider, getUpdateStatement());
		} catch (SQLException e) {
			throw new WorkerException("Error occurred while executing SQL command", e);
		}
	}

	private Expression updateTableReferences(Expression e) throws MalformedExpressionException{
		TableReferenceReplacer replacer=new TableReferenceReplacer(e);
		for(ColumnReference original:replacer.getReferences(targetTable.getId())){
			String columnName=targetTable.getColumnById(original.getColumnId()).getName();
			replacer.replaceColumnReference(original, newTable.getColumnReference(newTable.getColumnByName(columnName)));
		}
		return replacer.getExpression();
	}

}
