package org.gcube.data.analysis.tabulardata.operation.data.transformation;

import java.util.Arrays;
import java.util.List;

import javax.inject.Inject;
import javax.inject.Singleton;

import org.gcube.data.analysis.tabulardata.cube.CubeManager;
import org.gcube.data.analysis.tabulardata.cube.data.connection.DatabaseConnectionProvider;
import org.gcube.data.analysis.tabulardata.expression.evaluator.sql.SQLExpressionEvaluatorFactory;
import org.gcube.data.analysis.tabulardata.model.datatype.DataType;
import org.gcube.data.analysis.tabulardata.model.datatype.IntegerType;
import org.gcube.data.analysis.tabulardata.model.datatype.NumericType;
import org.gcube.data.analysis.tabulardata.model.table.Table;
import org.gcube.data.analysis.tabulardata.operation.OperationHelper;
import org.gcube.data.analysis.tabulardata.operation.OperationId;
import org.gcube.data.analysis.tabulardata.operation.factories.types.TableTransformationWorkerFactory;
import org.gcube.data.analysis.tabulardata.operation.invocation.OperationInvocation;
import org.gcube.data.analysis.tabulardata.operation.parameters.Cardinality;
import org.gcube.data.analysis.tabulardata.operation.parameters.Parameter;
import org.gcube.data.analysis.tabulardata.operation.parameters.leaves.TargetColumnParameter;
import org.gcube.data.analysis.tabulardata.operation.worker.exceptions.InvalidInvocationException;
import org.gcube.data.analysis.tabulardata.operation.worker.types.DataWorker;
@Singleton
public class DenormalizationFactory extends TableTransformationWorkerFactory {

	private static final OperationId OPERATION_ID=new OperationId(3005);
	
//	public static TargetColumnParameter VALUE_COLUMN=new TargetColumnParameter("value_column", "Value column", "Values to aggregate", Cardinality.ONE,
//			Arrays.asList(new GenericTableType(),new DatasetTableType()),Arrays.asList((ColumnType)new MeasureColumnType()));
//	public static TargetColumnParameter ATTRIBUTE_COLUMN=new TargetColumnParameter("attribute_column", "Attribute Column", "Column by which denormalize", Cardinality.ONE,
//			Arrays.asList(new GenericTableType(),new DatasetTableType()),Arrays.asList(new AttributeColumnType(),new AnnotationColumnType()));
	
	
	public static TargetColumnParameter VALUE_COLUMN=new TargetColumnParameter("value_column", "Value column", "Values to aggregate", Cardinality.ONE);
	public static TargetColumnParameter ATTRIBUTE_COLUMN=new TargetColumnParameter("attribute_column", "Attribute Column", "Column by which denormalize", Cardinality.ONE);
	
	private static List<Parameter> params=Arrays.asList((Parameter)VALUE_COLUMN,ATTRIBUTE_COLUMN); 
	
	
	private CubeManager cubeManager;
	private SQLExpressionEvaluatorFactory evaluatorFactory;
	private DatabaseConnectionProvider connProvider;
	
	@Inject
	public DenormalizationFactory(CubeManager cubeManager,
			SQLExpressionEvaluatorFactory evaluatorFactory,DatabaseConnectionProvider connProvider) {
		super();
		this.cubeManager = cubeManager;
		this.evaluatorFactory = evaluatorFactory;
		this.connProvider = connProvider;
	}

	@Override
	public DataWorker createWorker(OperationInvocation arg0)
			throws InvalidInvocationException {
		performBaseChecks(arg0, cubeManager);
		performCustomChecks(arg0);
		return new DenormalizationWorker(arg0, cubeManager, evaluatorFactory,connProvider);
	}
	
	@Override
	public String describeInvocation(OperationInvocation toDescribeInvocation)
			throws InvalidInvocationException {
		performBaseChecks(toDescribeInvocation, cubeManager);
		performCustomChecks(toDescribeInvocation);
		Table targetTable=cubeManager.getTable(toDescribeInvocation.getTargetTableId());
		return String.format("Denormalize %s by %s.", 
			OperationHelper.retrieveColumnLabel(
					targetTable.getColumnById(OperationHelper.getParameter(VALUE_COLUMN, toDescribeInvocation).getColumnId())),
					OperationHelper.retrieveColumnLabel(
							targetTable.getColumnById(OperationHelper.getParameter(ATTRIBUTE_COLUMN, toDescribeInvocation).getColumnId())));
	}
	
	private void performCustomChecks(OperationInvocation invocation) throws InvalidInvocationException{
		if(!OperationHelper.getParameter(ATTRIBUTE_COLUMN, invocation).getTableId().equals(invocation.getTargetTableId()))
			throw new InvalidInvocationException(invocation, "Attribute column must belong to target table");		
		if(!OperationHelper.getParameter(VALUE_COLUMN, invocation).getTableId().equals(invocation.getTargetTableId()))
			throw new InvalidInvocationException(invocation, "Value column must belong to target table");
		
		DataType quantityDataType=cubeManager.getTable(invocation.getTargetTableId()).
				getColumnById(OperationHelper.getParameter(VALUE_COLUMN, invocation).getColumnId()).getDataType();
		
		if(!(quantityDataType instanceof NumericType) && ! (quantityDataType instanceof IntegerType)){
			throw new InvalidInvocationException(invocation, "Unable to aggregate over "+quantityDataType.getName()+" data type");
		}
	}
	
	
	@Override
	protected OperationId getOperationId() {
		return OPERATION_ID;
	}
	

	@Override
	protected String getOperationDescription() {
		return "Denormalize a Table";
	}

	@Override
	protected String getOperationName() {
		return "Denormalization";
	}

	@Override
	protected List<Parameter> getParameters() {
		return params;
	}

	
}
