package org.gcube.data.analysis.tabulardata.operation.column.typechange;

import java.util.Collection;
import java.util.List;

import org.gcube.data.analysis.tabulardata.cube.CubeManager;
import org.gcube.data.analysis.tabulardata.cube.data.connection.DatabaseConnectionProvider;
import org.gcube.data.analysis.tabulardata.cube.tablemanagers.TableCreator;
import org.gcube.data.analysis.tabulardata.model.column.Column;
import org.gcube.data.analysis.tabulardata.model.column.ColumnType;
import org.gcube.data.analysis.tabulardata.model.column.type.IdColumnType;
import org.gcube.data.analysis.tabulardata.model.datatype.TextType;
import org.gcube.data.analysis.tabulardata.model.metadata.column.ColumnMetadata;
import org.gcube.data.analysis.tabulardata.model.table.Table;
import org.gcube.data.analysis.tabulardata.operation.worker.BaseWorker;
import org.gcube.data.analysis.tabulardata.operation.worker.OperationInvocation;
import org.gcube.data.analysis.tabulardata.operation.worker.exceptions.OperationException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.collect.Lists;

public abstract class ChangeColumnTypeTransformation extends BaseWorker {
	
	private static final Logger log = LoggerFactory.getLogger(ChangeColumnTypeTransformation.class);
	
	protected Table targetTable;

	protected Column targetColumn;

	protected CubeManager cubeManager;
	
	protected DatabaseConnectionProvider databaseConnectionProvider;
	
	public ChangeColumnTypeTransformation(OperationInvocation invocation, CubeManager cubeManager,
			DatabaseConnectionProvider databaseConnectionProvider) {
		super(invocation);
		this.cubeManager = cubeManager;
		this.databaseConnectionProvider = databaseConnectionProvider;
		this.targetTable = cubeManager.getTable(invocation.getTargetTableId());
		this.targetColumn = targetTable.getColumnById(invocation.getTargetColumnId());
	}

	@Override
	public void run() {
		try {
			inProgress(0.1f);
			Table newTable = createNewTable();
			inProgress(0.3f);
			String sqlCommand = generateSQLFillCommand(targetTable, newTable);
			inProgress(0.5f);
			executeSQLCommand(sqlCommand, databaseConnectionProvider);
			succeed(newTable);
		} catch (OperationException e) {
			fail(e);
		}
	}

	private Table createNewTable() {
		TableCreator tableCreator = cubeManager.createTable(targetTable.getTableType());
		log.debug("Column to remove: " + targetColumn );
		tableCreator.like(targetTable, false, Lists.newArrayList(targetColumn));
		Column modifiedColumn = new Column(new TextType(), getManagedColumnType());
		modifiedColumn.setAllMetadata(modifiedColumn.getAllMetadata());
		modifiedColumn.setLocalId(targetColumn.getLocalId());
		modifiedColumn.setName(targetColumn.getName());
		modifiedColumn.setAllMetadata(getMetadataToSet());
		tableCreator.addColumn(modifiedColumn);
		Table newTable = tableCreator.create();
		log.trace("Empty table created:\n" + newTable);
		return newTable;
	}
	
	protected abstract ColumnType getManagedColumnType();

	@SuppressWarnings("unchecked")
	private String generateSQLFillCommand(Table sourceTable, Table newTable) {
		StringBuilder sqlBuilder = new StringBuilder();
		List<Column> columnsToCopy = newTable.getColumnsExceptTypes(IdColumnType.class);
		String columnNamesSnippet = generateColumnNameSnippet(columnsToCopy);
		sqlBuilder.append(String.format("INSERT INTO %s (%s) ", newTable.getName(), columnNamesSnippet));
		sqlBuilder.append(String.format("SELECT %s FROM %s;", columnNamesSnippet, sourceTable.getName()));
		return sqlBuilder.toString();
	}

	private String generateColumnNameSnippet(List<Column> columnsToCopy) {
		StringBuilder stringBuilder = new StringBuilder();
		for (Column column : columnsToCopy) {
			stringBuilder.append(" " + column.getName());
			if (columnsToCopy.indexOf(column) != (columnsToCopy.size() - 1))
				stringBuilder.append(", ");
		}
		return stringBuilder.toString();
	}
	
	protected abstract Collection<ColumnMetadata> getMetadataToSet();
	
}
