package org.gcube.data.analysis.tabulardata.operation.column;

import java.sql.SQLException;

import org.gcube.data.analysis.tabulardata.cube.CubeManager;
import org.gcube.data.analysis.tabulardata.cube.data.connection.DatabaseConnectionProvider;
import org.gcube.data.analysis.tabulardata.cube.tablemanagers.TableCreator;
import org.gcube.data.analysis.tabulardata.model.column.Column;
import org.gcube.data.analysis.tabulardata.model.column.factories.TimeDimensionColumnFactory;
import org.gcube.data.analysis.tabulardata.model.table.Table;
import org.gcube.data.analysis.tabulardata.model.time.PeriodType;
import org.gcube.data.analysis.tabulardata.operation.OperationHelper;
import org.gcube.data.analysis.tabulardata.operation.SQLHelper;
import org.gcube.data.analysis.tabulardata.operation.invocation.OperationInvocation;
import org.gcube.data.analysis.tabulardata.operation.time.PeriodTypeHelper;
import org.gcube.data.analysis.tabulardata.operation.time.PeriodTypeHelperProvider;
import org.gcube.data.analysis.tabulardata.operation.validation.TimeDimensionColumnValidatorFactory;
import org.gcube.data.analysis.tabulardata.operation.worker.ImmutableWorkerResult;
import org.gcube.data.analysis.tabulardata.operation.worker.Worker;
import org.gcube.data.analysis.tabulardata.operation.worker.WorkerResult;
import org.gcube.data.analysis.tabulardata.operation.worker.exceptions.WorkerException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.collect.Lists;

public class ChangeToTimeDimensionColumn extends Worker {

	private static final Logger log = LoggerFactory.getLogger(ChangeToTimeDimensionColumn.class);

	private CubeManager cubeManager;

	private DatabaseConnectionProvider connectionProvider;


	private Table targetTable;

	private Column targetColumn;

	private PeriodType periodType;

	private Table newTable;

	private Table timeCodelist;

	private PeriodTypeHelperProvider periodTypeHelperProvider;

	private Column timeDimensionColumn;

	public ChangeToTimeDimensionColumn(OperationInvocation sourceInvocation, CubeManager cubeManager,
			DatabaseConnectionProvider connectionProvider,
			PeriodTypeHelperProvider periodTypeHelperProvider) {
		super(sourceInvocation);
		this.cubeManager = cubeManager;
		this.connectionProvider = connectionProvider;
		this.periodTypeHelperProvider = periodTypeHelperProvider;
	}

	@Override
	protected WorkerResult execute() throws WorkerException {
		retrieveParameters();
		updateProgress(0.1f);
		createTimeCodelist();
		updateProgress(0.3f);
		createNewTable();
		updateProgress(0.6f);
		linkNewTableToTimeCodelist();
		return new ImmutableWorkerResult(newTable, Lists.newArrayList(timeCodelist));
	}

	private void linkNewTableToTimeCodelist() throws WorkerException {
		PeriodTypeHelper helper = periodTypeHelperProvider.getHelper(periodType);
		String sql = helper.getUpdateDimensionColumnSQL(targetTable.getName(), targetColumn.getName(),
				newTable.getName(), timeDimensionColumn.getName(), timeCodelist.getName());
		try {
			SQLHelper.executeSQLCommand(sql, connectionProvider);
		} catch (SQLException e) {
			throw new WorkerException("Unable to link target table tuples to the newly created time codelist", e);
		}
	}

	private void createNewTable() {
		timeDimensionColumn = new TimeDimensionColumnFactory().create(periodType);
		TableCreator tableCreator = cubeManager.createTable(targetTable.getTableType());
		tableCreator.like(targetTable, true, Lists.newArrayList(targetColumn));
		tableCreator.addColumn(timeDimensionColumn);
		newTable = tableCreator.create();
	}

	private void createTimeCodelist() throws WorkerException {
		PeriodTypeHelper helper = periodTypeHelperProvider.getHelper(periodType);
		timeCodelist = helper.createTimeCodelist();
		String fillTimeCodelistSQL = helper.getFillTimeCodelistSQL(targetTable.getName(), targetColumn.getName(),
				timeCodelist.getName());
		try {
			SQLHelper.executeSQLCommand(fillTimeCodelistSQL, connectionProvider);
		} catch (SQLException e) {
			String msg = "An error occurred while filling the time dimension table, data is probably not suited for the operation";
			log.error(msg);
			throw new WorkerException(msg);
		}
	}


	private void retrieveParameters() {
		targetTable = cubeManager.getTable(getSourceInvocation().getTargetTableId());
		targetColumn = targetTable.getColumnById(getSourceInvocation().getTargetColumnId());
		String periodTypeName = OperationHelper.getParameter(
				TimeDimensionColumnValidatorFactory.PERIOD_FORMAT_PARAMETER, getSourceInvocation());
		periodType = PeriodType.fromName(periodTypeName);
	}

}
