package org.gcube.data.analysis.tabulardata.operation.validation;

import java.sql.SQLException;

import org.gcube.data.analysis.tabulardata.cube.CubeManager;
import org.gcube.data.analysis.tabulardata.cube.data.connection.DatabaseConnectionProvider;
import org.gcube.data.analysis.tabulardata.cube.tablemanagers.TableMetaCreator;
import org.gcube.data.analysis.tabulardata.expression.Expression;
import org.gcube.data.analysis.tabulardata.expression.MalformedExpressionException;
import org.gcube.data.analysis.tabulardata.expression.TableReferenceReplacer;
import org.gcube.data.analysis.tabulardata.expression.evaluator.EvaluatorException;
import org.gcube.data.analysis.tabulardata.expression.evaluator.description.DescriptionExpressionEvaluatorFactory;
import org.gcube.data.analysis.tabulardata.expression.evaluator.sql.SQLExpressionEvaluatorFactory;
import org.gcube.data.analysis.tabulardata.model.column.Column;
import org.gcube.data.analysis.tabulardata.model.column.ColumnReference;
import org.gcube.data.analysis.tabulardata.model.column.factories.ValidationColumnFactory;
import org.gcube.data.analysis.tabulardata.model.metadata.column.DataValidationMetadata;
import org.gcube.data.analysis.tabulardata.model.metadata.column.ValidationReferencesMetadata;
import org.gcube.data.analysis.tabulardata.model.metadata.common.ImmutableLocalizedText;
import org.gcube.data.analysis.tabulardata.model.table.Table;
import org.gcube.data.analysis.tabulardata.operation.SQLHelper;
import org.gcube.data.analysis.tabulardata.operation.ValidationHelper;
import org.gcube.data.analysis.tabulardata.operation.invocation.OperationInvocation;
import org.gcube.data.analysis.tabulardata.operation.worker.ImmutableWorkerResult;
import org.gcube.data.analysis.tabulardata.operation.worker.Worker;
import org.gcube.data.analysis.tabulardata.operation.worker.WorkerResult;
import org.gcube.data.analysis.tabulardata.operation.worker.exceptions.WorkerException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ValidateDataWithExpression extends Worker {

	private static final Logger log = LoggerFactory.getLogger(ValidateDataWithExpression.class);

	private CubeManager cubeManager;

	private DatabaseConnectionProvider connectionProvider;

	private Table targetTable;

	private Expression validationExpression;

	private String expressionDescription = null;

	private Column validationColumn;

	private SQLExpressionEvaluatorFactory sqlEvaluatorFactory;

	private DescriptionExpressionEvaluatorFactory descriptionEvaluatorFactory;

	private Table resultTable;

	

	public ValidateDataWithExpression(OperationInvocation invocation, CubeManager cubeManager,
			DatabaseConnectionProvider connectionProvider, SQLExpressionEvaluatorFactory sqlEvaluatorFactory, DescriptionExpressionEvaluatorFactory descriptionEvaluatorFactory) {
		super(invocation);
		this.cubeManager = cubeManager;
		this.connectionProvider = connectionProvider;
		targetTable = cubeManager.getTable(invocation.getTargetTableId());
		validationExpression = (Expression) invocation.getParameterInstances().get(
				ValidateDataWithExpressionFactory.EXPRESSION_PARAMETER.getIdentifier());
		
		if (invocation.getParameterInstances().containsKey(ValidateDataWithExpressionFactory.DESCRIPTION_PARAMETER))
			expressionDescription = (String) invocation.getParameterInstances().get(
					ValidateDataWithExpressionFactory.DESCRIPTION_PARAMETER.getIdentifier());
		
		
		this.sqlEvaluatorFactory = sqlEvaluatorFactory;
		this.descriptionEvaluatorFactory = descriptionEvaluatorFactory;
	}

	@Override
	protected WorkerResult execute() throws WorkerException {
		updateProgress(0.1f);
		addValidationColumn();
		updateProgress(0.4f);
		fillValidationColumn();
		updateProgress(0.7f);
		evaluateValidityAndUpdateTableMeta();
		return new ImmutableWorkerResult(resultTable);
	}

	private void evaluateValidityAndUpdateTableMeta() throws WorkerException {
		boolean validationResult = ValidationHelper.evaluateValidationColumnValidity(connectionProvider, targetTable.getName(), validationColumn.getName()); 
		TableMetaCreator tmc = cubeManager.modifyTableMeta(targetTable.getId());
		DataValidationMetadata validationMeta = new DataValidationMetadata(new ImmutableLocalizedText(createValidationDescription()),validationResult);
		tmc.setColumnMetadata(validationColumn.getLocalId(), validationMeta);		
		resultTable = tmc.create();
	}

	private String createValidationDescription() {
		return descriptionEvaluatorFactory.getEvaluator(validationExpression).evaluate();
	}

	private void fillValidationColumn() throws WorkerException {
		try {
			SQLHelper.executeSQLBatchCommands(connectionProvider, generateSetAllFalseSqlCommand(),
					generateValidationSqlCommand());
		} catch (SQLException e) {
			e.getNextException().printStackTrace();
			throw new WorkerException("Error occurred while executing SQL command", e);
		}
	}

	private String generateSetAllFalseSqlCommand() {
		return String.format("UPDATE %s SET %s = false;", targetTable.getName(), validationColumn.getName());
	}

	private String generateValidationSqlCommand() throws WorkerException {
		try {
			//Change table references in given expression
			//			log.debug("Passed expression is "+sqlEvaluatorFactory.getEvaluator(validationExpression).evaluate());
			//			Expression toUseExpression =new TableReferenceReplacer(validationExpression).replaceTableId(targetTable.getId(), tempTable.getId()).getExpression();
			validationExpression.validate();
			return String.format("UPDATE %s SET %s = true WHERE %s;", targetTable.getName(), validationColumn.getName(),
					sqlEvaluatorFactory.getEvaluator(validationExpression).evaluate());
		} catch (EvaluatorException e) {
			throw new WorkerException("Unable to evaluate epression", e);
		} catch (MalformedExpressionException e) {
			throw new WorkerException("Unable to evaluate epression", e);
		}
	}

	private void addValidationColumn() throws WorkerException {	
		String validationDescription = "Expression based validation";
		if (expressionDescription!=null)
			validationDescription = expressionDescription;
		
		DataValidationMetadata validationMeta = new DataValidationMetadata(new ImmutableLocalizedText(validationDescription),false);
		validationColumn = new ValidationColumnFactory().create(new ImmutableLocalizedText(validationDescription),validationMeta);
		targetTable=cubeManager.addValidations(targetTable.getId(),validationColumn);	
		log.debug("Added validation column:\n" + targetTable);
		
		//Check expression 
		try{
		TableReferenceReplacer replacer=new TableReferenceReplacer(validationExpression);
		TableMetaCreator creator=cubeManager.modifyTableMeta(targetTable.getId());
		for(ColumnReference refColumn:replacer.getReferences(targetTable.getId())){
			Column column =targetTable.getColumnById(refColumn.getColumnId());
			ValidationReferencesMetadata referencesMeta;
			if (column.contains(ValidationReferencesMetadata.class)){
				referencesMeta = column.getMetadata(ValidationReferencesMetadata.class);
				referencesMeta.add(validationColumn.getLocalId());
			} else referencesMeta = new ValidationReferencesMetadata(validationColumn.getLocalId());
			
			creator.setColumnMetadata(refColumn.getColumnId(), referencesMeta);
		}
		creator.create();		
		}catch(MalformedExpressionException e){
			throw new WorkerException("Unable to get expression related columns",e);
		}
	}

}
