package org.gcube.data.analysis.tabulardata.operation.sdmx.dataset;

import java.net.URI;
import java.net.URISyntaxException;
import java.util.Iterator;
import java.util.List;

import org.gcube.data.analysis.tabulardata.metadata.NoSuchMetadataException;
import org.gcube.data.analysis.tabulardata.model.column.Column;
import org.gcube.data.analysis.tabulardata.model.column.ColumnLocalId;
import org.gcube.data.analysis.tabulardata.model.column.type.AttributeColumnType;
import org.gcube.data.analysis.tabulardata.model.column.type.DimensionColumnType;
import org.gcube.data.analysis.tabulardata.model.column.type.MeasureColumnType;
import org.gcube.data.analysis.tabulardata.model.column.type.TimeDimensionColumnType;
import org.gcube.data.analysis.tabulardata.model.metadata.common.LocalizedText;
import org.gcube.data.analysis.tabulardata.model.metadata.common.NamesMetadata;
import org.gcube.data.analysis.tabulardata.model.resources.InternalURI;
import org.gcube.data.analysis.tabulardata.model.resources.ResourceType;
import org.gcube.data.analysis.tabulardata.model.table.Table;
import org.gcube.data.analysis.tabulardata.operation.invocation.OperationInvocation;
import org.gcube.data.analysis.tabulardata.operation.sdmx.dataset.ds.DataSourceConfigurationBean;
import org.gcube.data.analysis.tabulardata.operation.worker.exceptions.WorkerException;
import org.gcube.data.analysis.tabulardata.operation.worker.results.ResourcesResult;
import org.gcube.data.analysis.tabulardata.operation.worker.results.resources.ImmutableURIResult;
import org.gcube.data.analysis.tabulardata.operation.worker.types.ResourceCreatorWorker;
import org.gcube.datapublishing.sdmx.api.registry.SDMXRegistryClient;
import org.gcube.datapublishing.sdmx.impl.exceptions.SDMXRegistryClientException;
import org.sdmxsource.sdmx.api.constants.ATTRIBUTE_ATTACHMENT_LEVEL;
import org.sdmxsource.sdmx.api.constants.SDMX_STRUCTURE_TYPE;
import org.sdmxsource.sdmx.api.model.beans.reference.StructureReferenceBean;
import org.sdmxsource.sdmx.api.model.mutable.base.ComponentMutableBean;
import org.sdmxsource.sdmx.api.model.mutable.base.TextTypeWrapperMutableBean;
import org.sdmxsource.sdmx.api.model.mutable.conceptscheme.ConceptMutableBean;
import org.sdmxsource.sdmx.api.model.mutable.conceptscheme.ConceptSchemeMutableBean;
import org.sdmxsource.sdmx.api.model.mutable.datastructure.AttributeMutableBean;
import org.sdmxsource.sdmx.api.model.mutable.datastructure.DataStructureMutableBean;
import org.sdmxsource.sdmx.api.model.mutable.datastructure.DataflowMutableBean;
import org.sdmxsource.sdmx.api.model.mutable.datastructure.DimensionListMutableBean;
import org.sdmxsource.sdmx.api.model.mutable.datastructure.DimensionMutableBean;
import org.sdmxsource.sdmx.api.model.mutable.datastructure.PrimaryMeasureMutableBean;
import org.sdmxsource.sdmx.sdmxbeans.model.mutable.base.TextTypeWrapperMutableBeanImpl;
import org.sdmxsource.sdmx.sdmxbeans.model.mutable.conceptscheme.ConceptMutableBeanImpl;
import org.sdmxsource.sdmx.sdmxbeans.model.mutable.conceptscheme.ConceptSchemeMutableBeanImpl;
import org.sdmxsource.sdmx.sdmxbeans.model.mutable.datastructure.AttributeMutableBeanImpl;
import org.sdmxsource.sdmx.sdmxbeans.model.mutable.datastructure.DataStructureMutableBeanImpl;
import org.sdmxsource.sdmx.sdmxbeans.model.mutable.datastructure.DimensionListMutableBeanImpl;
import org.sdmxsource.sdmx.sdmxbeans.model.mutable.datastructure.DimensionMutableBeanImpl;
import org.sdmxsource.sdmx.sdmxbeans.model.mutable.datastructure.PrimaryMeasureMutableBeanImpl;
import org.sdmxsource.sdmx.sdmxbeans.model.mutable.metadatastructure.DataflowMutableBeanImpl;
import org.sdmxsource.sdmx.util.beans.reference.StructureReferenceBeanImpl;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.collect.Lists;

public class SDMXDataSetExporter extends ResourceCreatorWorker {

	private static Logger log = LoggerFactory.getLogger(SDMXDataSetExporter.class);

	private Table table;

	private OperationInvocation invocation;

	private Column primaryMeasure;

	private String registryUrl;

	private String targetAgency;

	private String targetId;

	private String targetVersion;
	
	private List<LocalizedText> tableNamesMetadata;

	private final String ATTRIBUTE_ASSIGNMENT_STATUS_MANDATORY = "Mandatory";
	
	private static String errorMessage = "Unable to complete export procedure";;

	
	
	public SDMXDataSetExporter(Table table, OperationInvocation invocation) {
		super(invocation);
		this.table = table;
		this.invocation = invocation;
		

	}
	
	@Override
	protected ResourcesResult execute() throws WorkerException {
		try {
			DataSourceConfigurationBean datasourceConfigurationBean = new DataSourceConfigurationBean();
			datasourceConfigurationBean.setTable_id(this.table.getName());
			retrieveParameters(datasourceConfigurationBean);
			loadMetadata ();
			updateProgress(0.1f,"Creating beans");
			DataStructureMutableBean dataStructure = createDataStructureBean(datasourceConfigurationBean);
			ConceptSchemeMutableBean conceptScheme = createConceptSchemeBean();
			DataflowMutableBean dataFlow = createDataFlowBean(dataStructure);
			updateProgress(0.2f,"Populating data structure");
			populateDataStructure(dataStructure, conceptScheme,datasourceConfigurationBean);
			updateProgress(0.6f,"Publishing");
			publishData(dataStructure, conceptScheme,dataFlow);
			updateProgress(0.8f,"Finalizing");

			return new ResourcesResult(new ImmutableURIResult(new InternalURI(new URI(registryUrl)), "Codelist SDMX export" , 
					String.format("%scodelist/%s/%s/%s/", registryUrl, targetAgency, targetId, targetVersion), ResourceType.SDMX));
		} catch (RuntimeException e) {
			log.error(errorMessage, e);
			throw new WorkerException(errorMessage, e);
		} catch (URISyntaxException e) {
			throw new WorkerException(String.format("exported url %s not valid",registryUrl),e);
		}
	}


	private void retrieveParameters(DataSourceConfigurationBean datasourceConfigurationBean) {
		registryUrl = (String) invocation.getParameterInstances().get(DatasetWorkerUtils.REGISTRY_BASE_URL);
		targetAgency = (String) invocation.getParameterInstances().get(DatasetWorkerUtils.AGENCY);
		targetId = (String) invocation.getParameterInstances().get(DatasetWorkerUtils.ID);
		targetVersion = (String) invocation.getParameterInstances().get(DatasetWorkerUtils.VERSION);
		String observationValue = (String) this.invocation.getParameterInstances().get(DatasetWorkerUtils.OBS_VALUE_COLUMN);
		primaryMeasure = this.table.getColumnById(new ColumnLocalId(observationValue));
		datasourceConfigurationBean.setObservationValue(observationValue);
	}


	/**
	 * 
	 * @param dsd
	 * @param concepts
	 * @param dataFlow
	 * @throws WorkerException
	 */
	private void publishData(DataStructureMutableBean dsd, ConceptSchemeMutableBean concepts,DataflowMutableBean dataFlow) throws WorkerException {
		String url = (String) invocation.getParameterInstances().get(DatasetWorkerUtils.REGISTRY_BASE_URL);
		SDMXRegistryClient registryClient = DatasetWorkerUtils.initSDMXClient(url);

		try {
			log.debug("Publishing concepts...");
			registryClient.publish(concepts.getImmutableInstance());
			log.debug("Concepts published");
		} catch (SDMXRegistryClientException e) {
			throw new WorkerException("Unable to publish concepts on registry.", e);
		}
		
		
		try {
		
			log.debug("Publishing dsd...");
			registryClient.publish(dsd.getImmutableInstance());
			
			log.debug("DSD published");
		} catch (SDMXRegistryClientException e) {
			throw new WorkerException("Unable to publish dsd on registry.", e);
		}
		
		try {
			
			log.debug("Publishing data flow...");
			registryClient.publish(dataFlow.getImmutableInstance());
			
			log.debug("Data flow published");
		} catch (SDMXRegistryClientException e) {
			throw new WorkerException("Unable to publish dsd on registry.", e);
		}
	}
	




	@SuppressWarnings ("unchecked")
	private void populateDataStructure (DataStructureMutableBean dataStructure,ConceptSchemeMutableBean conceptScheme, DataSourceConfigurationBean dataSourceConfigurationBean) throws WorkerException 
	{

		List<Column> measureColumns = this.table.getColumnsByType(MeasureColumnType.class);
		List<Column> dimensionColumns = this.table.getColumnsByType(DimensionColumnType.class);
		List<Column> attributeColumns = this.table.getColumnsByType(AttributeColumnType.class);
		Column timeDimensionColumn = this.table.getColumnsByType(TimeDimensionColumnType.class).get(0);
		dataSourceConfigurationBean.setObservationTime(timeDimensionColumn.getLocalId().getValue());
		
		log.debug("Columns loaded");
		boolean found = false;
		Iterator<Column> measureColumnsIterator = measureColumns.iterator();
		log.debug("Looking for primary measure column");
		
		while (measureColumnsIterator.hasNext() && !found)
		{
			Column column = measureColumnsIterator.next();
			
			if (column.getLocalId().getValue().equals(this.primaryMeasure.getLocalId().getValue()))
			{
				measureColumns.remove(column);
				found = true;
			}
		}

		ConceptMutableBean primaryMeasureConcept = createConceptBean(this.primaryMeasure);

		conceptScheme.addItem(primaryMeasureConcept);
		StructureReferenceBean conceptReferenceBean = new StructureReferenceBeanImpl (primaryMeasureConcept.getParentAgency(),conceptScheme.getId(),
				this.targetVersion,SDMX_STRUCTURE_TYPE.CONCEPT,primaryMeasureConcept.getId());
		
		PrimaryMeasureMutableBean primaryMeasureBean = new PrimaryMeasureMutableBeanImpl();
	
		primaryMeasureBean.setConceptRef(conceptReferenceBean);
		primaryMeasureBean.setId("OBS_VALUE");
		
		dataStructure.setPrimaryMeasure(primaryMeasureBean);
		DimensionListMutableBean dimensionListBean = new DimensionListMutableBeanImpl();
		addMeasureDimensions(dimensionListBean, measureColumns,conceptScheme,dataSourceConfigurationBean);
		addGenericDimensions(dimensionListBean, dimensionColumns,conceptScheme,dataSourceConfigurationBean);
		DimensionMutableBean timeDimensionBean = new DimensionMutableBeanImpl();
		timeDimensionBean.setId("TIME_PERIOD");
		ConceptMutableBean timeDimensionConcept = createConceptBean(timeDimensionColumn);
		conceptScheme.addItem(timeDimensionConcept);
		timeDimensionBean.setConceptRef(getConceptReference(conceptScheme, timeDimensionConcept));
		timeDimensionBean.setTimeDimension(true);
		dimensionListBean.addDimension(timeDimensionBean);
		dataStructure.setDimensionList(dimensionListBean);
		addAttributes(dataStructure, attributeColumns,conceptScheme,dataSourceConfigurationBean);


	}
	
	/**
	 * 
	 * @param dimensionListBean
	 * @param dimensionColumns
	 * @param concepts
	 */
	private void addGenericDimensions (DimensionListMutableBean dimensionListBean, List<Column> dimensionColumns,ConceptSchemeMutableBean concepts,DataSourceConfigurationBean dataSourceConfigurationBean)
	{
		log.debug("Adding dimension list bean");
		for (Column column : dimensionColumns)
		{
			ConceptMutableBean columnConcept = createConceptBean(column);
			concepts.addItem(columnConcept);
			DimensionMutableBean dimensionBean = new DimensionMutableBeanImpl();
			setBeanData(column, dimensionBean, getConceptReference(concepts, columnConcept));
			dimensionListBean.addDimension(dimensionBean);
			dataSourceConfigurationBean.addDimension(columnConcept.getId(), column.getLocalId().getValue());
		}	
	}
	
	/**
	 * 
	 * @param dataStructure
	 * @param attributeColumns
	 * @param concepts
	 */
	private void addAttributes (DataStructureMutableBean dataStructure, List<Column> attributeColumns,ConceptSchemeMutableBean concepts,DataSourceConfigurationBean dataSourceConfigurationBean)
	{
		log.debug("Adding attribute list bean");
		for (Column column : attributeColumns)
		{
			ConceptMutableBean columnConcept = createConceptBean(column);
			concepts.addItem(columnConcept);
			AttributeMutableBean attributeBean = new AttributeMutableBeanImpl();
			attributeBean.setAttachmentLevel(ATTRIBUTE_ATTACHMENT_LEVEL.OBSERVATION);
			attributeBean.setAssignmentStatus(ATTRIBUTE_ASSIGNMENT_STATUS_MANDATORY);
			setBeanData(column, attributeBean, getConceptReference(concepts, columnConcept));
			dataStructure.addAttribute(attributeBean);
			dataSourceConfigurationBean.addAttributes(columnConcept.getId(), column.getLocalId().getValue());
		}	
	}
	
	/**
	 * 
	 * @param dimensionListBean
	 * @param measureColumns
	 * @param concepts
	 */
	private void addMeasureDimensions (DimensionListMutableBean dimensionListBean, List<Column> measureColumns,ConceptSchemeMutableBean concepts, DataSourceConfigurationBean dataSourceConfigurationBean )
	{
		log.debug("Adding measure dimension list bean");
		for (Column column : measureColumns)
		{
			ConceptMutableBean columnConcept = createConceptBean(column);
			concepts.addItem(columnConcept);
			DimensionMutableBean dimensionBean = new DimensionMutableBeanImpl();
			dimensionBean.setMeasureDimension(true);
			setBeanData(column, dimensionBean, getConceptReference(concepts, columnConcept));
			dimensionListBean.addDimension(dimensionBean);
			dataSourceConfigurationBean.addDimension(columnConcept.getId(), column.getLocalId().getValue());
		}	
	}
	
	/**
	 * 
	 * @param conceptScheme
	 * @param concept
	 * @return
	 */
	private StructureReferenceBean getConceptReference (ConceptSchemeMutableBean conceptScheme, ConceptMutableBean concept)
	{

		return new StructureReferenceBeanImpl (concept.getParentAgency(),conceptScheme.getId(),
				this.targetVersion,SDMX_STRUCTURE_TYPE.CONCEPT,concept.getId());
	}
	
	
	/**
	 * 
	 * @param column
	 * @param componentBean
	 * @param conceptRef
	 */
	private void setBeanData (Column column,ComponentMutableBean componentBean,StructureReferenceBean conceptRef)
	{
		componentBean.setId(column.getName()+"_DSD");
		componentBean.setConceptRef(conceptRef);
	}

	/**
	 * 
	 * @param column
	 * @return
	 */
	private ConceptMutableBean createConceptBean (Column column)
	{
		log.debug("Generating concept mutable bean for column "+column.getName());
		ConceptMutableBean concept = new ConceptMutableBeanImpl();
		concept.setId(column.getName()+"_concept");
		concept.setParentAgency(this.targetAgency);
		
		try {
			List<LocalizedText> conceptNames = column.getMetadata(NamesMetadata.class).getTexts();
			log.debug("Concept names size "+conceptNames.size());
			concept.setNames(getNamesMetadata(conceptNames, null, null));
		} catch (NoSuchMetadataException e) {
			// TODO-LF: This exception should not occur
		}
		
		return concept;
	}
	
	/**
	 * 
	 * @return
	 */
	private DataStructureMutableBean createDataStructureBean(DataSourceConfigurationBean datasourceConfigurationBean) {
		DataStructureMutableBean dataStructure = new DataStructureMutableBeanImpl();
		dataStructure.setAgencyId(this.targetAgency);
		dataStructure.setVersion(this.targetVersion);
		String dsdId = this.targetId+"_DSD";
		dataStructure.setId(dsdId);
		dataStructure.setNames(getNamesMetadata(this.tableNamesMetadata,targetId+" Data Structure Definition", "en"));
		datasourceConfigurationBean.setDsdId(dsdId);
		return dataStructure;
	}
	
	/**
	 * 
	 * @param dataStructure
	 * @return
	 */
	private DataflowMutableBean createDataFlowBean (DataStructureMutableBean dataStructure)
	{
		DataflowMutableBean dataFlow = new DataflowMutableBeanImpl();
		log.debug("Populating data flow bean");
		dataFlow.setAgencyId(this.targetAgency);
		dataFlow.setDataStructureRef(new StructureReferenceBeanImpl (dataStructure.getAgencyId(),dataStructure.getId(), this.targetVersion,SDMX_STRUCTURE_TYPE.DSD));
		dataFlow.setId(this.targetId+"_dataFlow");
		dataFlow.setNames(getNamesMetadata(this.tableNamesMetadata,targetId+" Data Flow", "en"));
		return dataFlow;
	}
	
	/**
	 * 
	 * @return
	 */
	private ConceptSchemeMutableBean createConceptSchemeBean() {
		ConceptSchemeMutableBean conceptScheme = new ConceptSchemeMutableBeanImpl();
		String conceptID = this.table.getName()+"_concepts";
		conceptScheme.setId(conceptID);
		conceptScheme.setAgencyId(this.targetAgency);
		conceptScheme.setVersion(this.targetVersion);
		conceptScheme.setNames(getNamesMetadata(this.tableNamesMetadata,conceptID+ " Concepts", "en"));
		return conceptScheme;
	}


	/**
	 * 
	 */
	private void loadMetadata ()
	{
		try
		{
			this.tableNamesMetadata = this.table.getMetadata(NamesMetadata.class).getTexts();
		} catch (NoSuchMetadataException e)
		{
			this.tableNamesMetadata = Lists.newArrayList();
		}

		
	}



	/**
	 * 
	 * @param metadataValues
	 * @param defaultValue
	 * @param defaultLocale
	 * @return
	 */
	private List<TextTypeWrapperMutableBean> getNamesMetadata (List<LocalizedText> metadataValues,String defaultValue, String defaultLocale)
	{
		List<TextTypeWrapperMutableBean> response = Lists.newArrayList();
		
		if (metadataValues.size() == 0 && defaultValue != null) 
		{
			log.warn("Names Metadata: using default value "+defaultValue);
			response.add(new TextTypeWrapperMutableBeanImpl(defaultLocale, defaultValue));
		}
		else
		{
			for (LocalizedText text : metadataValues) 
			{
				log.debug("Adding metadata value "+text.getValue()+" "+text.getLocale());
				response.add(new TextTypeWrapperMutableBeanImpl(text.getLocale(), text.getValue()));
			}
		}
		
		return response;

	}
	
	

}