package org.gcube.data.analysis.tabulardata.operation.sdmx.datastructuredefinition.impl;

import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import org.apache.commons.dbutils.DbUtils;
import org.gcube.data.analysis.tabulardata.cube.CubeManager;
import org.gcube.data.analysis.tabulardata.cube.data.connection.DatabaseConnectionProvider;
import org.gcube.data.analysis.tabulardata.model.column.Column;
import org.gcube.data.analysis.tabulardata.model.metadata.common.LocalizedText;
import org.gcube.data.analysis.tabulardata.model.table.Table;
import org.gcube.data.analysis.tabulardata.operation.invocation.OperationInvocation;
import org.gcube.data.analysis.tabulardata.operation.sdmx.datastructuredefinition.SDMXDataStructureDefinitionAbstractExporter;
import org.gcube.data.analysis.tabulardata.operation.sdmx.datastructuredefinition.beans.DataStructureBean;
import org.gcube.data.analysis.tabulardata.operation.sdmx.datastructuredefinition.beans.ExcelDataStructureBean;
import org.gcube.data.analysis.tabulardata.operation.sdmx.excel.ExcelGenerator;
import org.gcube.data.analysis.tabulardata.operation.sdmx.excel.impl.ExcelGeneratorFromTable;
import org.gcube.data.analysis.tabulardata.operation.sdmx.template.TemplateWorkerUtils;
import org.sdmxsource.sdmx.api.model.beans.codelist.CodelistBean;
import org.sdmxsource.sdmx.api.model.mutable.conceptscheme.ConceptMutableBean;
import org.sdmxsource.sdmx.api.model.mutable.datastructure.DataStructureMutableBean;
import org.sdmxsource.sdmx.sdmxbeans.model.mutable.datastructure.DataStructureMutableBeanImpl;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SDMXDataStructureDefinitionExcelExporter extends SDMXDataStructureDefinitionAbstractExporter {

	private Logger logger;
	
	public SDMXDataStructureDefinitionExcelExporter(Table table, DatabaseConnectionProvider connectionProvider, OperationInvocation invocation,CubeManager cubeManager) {
		super (table,connectionProvider,invocation,cubeManager);
		this.logger = LoggerFactory.getLogger(this.getClass());
	}
	
	
	@Override
	protected void extraOperation(DataStructureBean dataStructure) {
		updateProgress(0.8f, "Generating excel");
		this.logger.debug("Generating excel");
		ExcelGenerator generator = new ExcelGeneratorFromTable(((ExcelDataStructureBean) dataStructure).getTableBean());
		generator.generateExcel(((ExcelDataStructureBean) dataStructure).getExcelName(), TemplateWorkerUtils.DEFAULT_EXCEL_FOLDER);
		this.logger.debug("Operation completed");

	}

	@Override
	protected void registerTimeDimensionColumn(Column column, ConceptMutableBean concept,
			DataStructureBean dataStructure) {
		this.logger.debug("Adding time dimension to excel table");
		((ExcelDataStructureBean)dataStructure).setTimeDimension(column, concept);
		this.logger.debug("Time dimension added");

	}

	@Override
	protected void registerDimensionColumn(Column column, ConceptMutableBean concept, CodelistBean immutableCodelist,
			DataStructureBean dataStructure) {
		this.logger.debug("Adding generic dimension to excel table");
		((ExcelDataStructureBean)dataStructure).addDimensionColumn(column, concept, immutableCodelist);
		this.logger.debug("Generic dimension added");


	}

	@Override
	protected void registerAttributeColumn(Column column, ConceptMutableBean concept, CodelistBean immutableCodelist,
			DataStructureBean dataStructure) {
		this.logger.debug("Adding attribute to excel table");
		((ExcelDataStructureBean)dataStructure).addAttributeColumn(column, concept, immutableCodelist);
		this.logger.debug("Attribute dimension added");

	}

	@Override
	protected void registerMeasureColumn(Column column, ConceptMutableBean concept, DataStructureBean dataStructure) {
		this.logger.debug("Adding measure to excel table");
		((ExcelDataStructureBean)dataStructure).addMeasureColumn(column, concept);
		this.logger.debug("Measure added");

	}

	@Override
	protected DataStructureBean createDataStructureBean(Table table, String targetAgency, String targetVersion,
			String targetId, List<LocalizedText> tableNamesMetadata) {
		DataStructureMutableBean dataStructureDefinition = new DataStructureMutableBeanImpl();
		dataStructureDefinition.setAgencyId(targetAgency);
		dataStructureDefinition.setVersion(targetVersion);
		String dsdId = targetId+"_DSD";
		dataStructureDefinition.setId(dsdId);
		dataStructureDefinition.setNames(getNamesMetadata(tableNamesMetadata,targetId+" Data Structure Definition", "en"));
		ExcelDataStructureBean response = new ExcelDataStructureBean(table);
		response.setDsd(dataStructureDefinition);
		response.setExcelName(targetId+"_"+targetVersion);
		return response;
	}


	@Override
	protected void registerPrimaryMeasure(Column column, ConceptMutableBean concept, DataStructureBean dataStructure) {
		this.logger.debug("Adding primary measure to excel table");
		((ExcelDataStructureBean)dataStructure).setPrimaryMeasure(column, concept);
		this.logger.debug("Primary measure added");
		
	}

	private String buildSqlQuery(Table table,List<Column> columns) {
		StringBuilder sql = new StringBuilder();
		sql.append("SELECT ");
		for (Column column : columns) {
			sql.append(column.getName() + " , ");
		}	
		sql.delete(sql.length() - 2, sql.length()); // Delete comma
		sql.append(" FROM " + table.getName() + ";");
		return sql.toString();
	}



	private Map<String, List<String>> getData(Table table, List<Column> columns,DatabaseConnectionProvider connectionProvider) throws SQLException{
		this.logger.debug("Downloading data");
		String query = buildSqlQuery(table,columns);
		this.logger.debug("Query "+query);
		Connection conn = null;
		Statement stmt = null;
		ResultSet rs = null;
		try {
			conn = connectionProvider.getConnection();
			stmt = conn.createStatement();
			rs = stmt.executeQuery(query);
			
			HashMap<String, List<String>> response = new HashMap<>();
			
			while (rs.next()) {

				for (Column column : columns) 
				{
					String columnName = column.getName();
					this.logger.debug("Adding data for column "+columnName);
					List<String> dataList = response.get(columnName);
						
					if (dataList == null)
					{
						dataList = new LinkedList<>();
						response.put(columnName, dataList);
					}
					
					dataList.add(rs.getString(columnName));

				}

			}

			return response;
			
		} catch (SQLException e) 
		{

			this.logger.error("Unable to execute database query.", e);
			if (e.getNextException() != null)
				this.logger.error("Inner Exception: ", e.getNextException());
			throw e;
		} finally 
		{
			DbUtils.closeQuietly(rs);
			DbUtils.closeQuietly(stmt);
			DbUtils.closeQuietly(conn);
		}
	}


	@Override
	protected void registerData(DataStructureBean dataStructure, Table table, List<Column> columns, DatabaseConnectionProvider connectionProvider) {
		this.logger.debug("Registering data for excel");
		
		try
		{
			Map<String, List<String>> data = getData(table, columns, connectionProvider);
			
			Iterator<String> columnNames = data.keySet().iterator();
			
			while (columnNames.hasNext())
			{
				String columnName = columnNames.next();
				this.logger.debug("Adding data for column "+columnName);
				((ExcelDataStructureBean)dataStructure).addData(columnName, data.get(columnName));
				this.logger.debug("Data added");
			}
			

			
		} catch (SQLException e)
		{
			this.logger.error("Unable to register excel data",e);
		}
		
	}
	
}
