package org.gcube.data.analysis.tabulardata.statistical;

import java.io.File;
import java.nio.charset.Charset;
import java.util.HashMap;
import java.util.UUID;

import org.gcube.contentmanagement.blobstorage.service.IClient;
import org.gcube.data.analysis.statisticalmanager.exception.ResourceNotFoundException;
import org.gcube.data.analysis.statisticalmanager.proxies.StatisticalManagerDataSpace;
import org.gcube.data.analysis.tabulardata.cube.CubeManager;
import org.gcube.data.analysis.tabulardata.model.table.Table;
import org.gcube.data.analysis.tabulardata.operation.csv.Constants;
import org.gcube.data.analysis.tabulardata.operation.csv.importer.CSVImportFactory;
import org.gcube.data.analysis.tabulardata.operation.invocation.OperationInvocation;
import org.gcube.data.analysis.tabulardata.operation.worker.ImmutableWorkerResult;
import org.gcube.data.analysis.tabulardata.operation.worker.Worker;
import org.gcube.data.analysis.tabulardata.operation.worker.WorkerResult;
import org.gcube.data.analysis.tabulardata.operation.worker.WorkerStatus;
import org.gcube.data.analysis.tabulardata.operation.worker.WorkerWrapper;
import org.gcube.data.analysis.tabulardata.operation.worker.exceptions.InvalidInvocationException;
import org.gcube.data.analysis.tabulardata.operation.worker.exceptions.WorkerException;

public class ImportFromStatisticalOperation extends Worker{

	
	private CubeManager cubeManager;
	private StatisticalManagerDataSpace statisticalDataSpace;
	private CSVImportFactory csvImportFactory;
	private IClient storageClient;
	
	
	
	
	public ImportFromStatisticalOperation(OperationInvocation sourceInvocation,
			CubeManager cubeManager,
			StatisticalManagerDataSpace statisticalDataSpace,
			CSVImportFactory csvImportFactory, IClient storageClient) {
		super(sourceInvocation);
		this.cubeManager = cubeManager;
		this.statisticalDataSpace = statisticalDataSpace;
		this.csvImportFactory = csvImportFactory;
		this.storageClient = storageClient;
	}

	
	private String resourceId;
	private Table importedTable;
	private File csvFile;
	
	@Override
	protected WorkerResult execute() throws WorkerException {
		resourceId=(String) getSourceInvocation().getParameterInstances().get(ImportFromStatisticalOperationFactory.RESOURCE_ID.getIdentifier());
		updateProgress(0.1f);
		importSMTable();
		updateProgress(0.5f);
		importedTable=importCSV(csvFile);
		updateProgress(0.9f);
		return new ImmutableWorkerResult(importedTable);
	}
	
	
	
	private void importSMTable() throws WorkerException {
		try {
			csvFile = statisticalDataSpace.exportTable(resourceId);			
		} catch (ResourceNotFoundException e) {
			throw new WorkerException(
					"Unable to import result table from dataspace ", e);
		}
	}

	private Table importCSV(File csv) throws WorkerException {
		Character delimiter=',';
		Boolean hasHeader=true;
		String fileUri = sendToStorage(csv);

		HashMap<String, Object> parameters = new HashMap<String, Object>();
		parameters.put(Constants.HASHEADER, hasHeader);
		parameters.put(Constants.URL, fileUri);
		parameters.put(Constants.SEPARATOR, delimiter+"");
		parameters.put(Constants.ENCODING, Charset.defaultCharset().toString());

		WorkerWrapper wrapper = new WorkerWrapper(csvImportFactory);
		try{
			WorkerStatus status=wrapper.execute(null, null, parameters);
			if(status.equals(WorkerStatus.SUCCEDED)){
				return wrapper.getResult().getResultTable();
			}else throw new WorkerException("Failed to import file from Statistical Manager");
		}catch(InvalidInvocationException e){
			throw new WorkerException("Unable to import SM Table from File "+csv.getAbsolutePath(),e);
		}
	}

	private String sendToStorage(File csv) {
		String rPath="/csv/"+UUID.randomUUID().toString();
		String remoteId=storageClient.put(true).LFile(csv.getAbsolutePath()).RFile(rPath);
		return storageClient.getUrl().RFile(rPath);
	}
	
}
