package org.gcube.data.publishing.gis.geoserver;

import java.io.IOException;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;

import org.gcube.common.core.utils.logging.GCUBELog;
import org.gcube.common.geoserverinterface.GeoCaller;
import org.gcube.common.geoserverinterface.GeonetworkCommonResourceInterface.GeonetworkCategory;
import org.gcube.common.geoserverinterface.bean.BoundsRest;
import org.gcube.common.geoserverinterface.bean.FeatureTypeRest;
import org.gcube.common.geoserverinterface.engine.MakeStyle;
import org.gcube.data.publishing.gis.geoserver.db.DBUtils;
import org.gcube.data.publishing.gis.geoserver.db.Table;
import org.gcube.data.publishing.gis.geoserver.model.DBDescriptor;
import org.gcube.data.publishing.gis.geoserver.model.GeoServerDescriptor;
import org.gcube.data.publishing.gis.publisher.plugin.fwk.CSquareConverter;
import org.gcube.data.publishing.gis.publisher.plugin.fwk.GISInteraction;
import org.gcube.data.publishing.gis.publisher.plugin.fwk.model.CSquarePoint;
import org.gcube.data.publishing.gis.publisher.plugin.fwk.model.GISData;
import org.gcube.data.publishing.gis.publisher.plugin.fwk.model.GISFile;
import org.gcube.data.publishing.gis.publisher.plugin.fwk.model.GeometryPoint;
import org.gcube.data.publishing.gis.publisher.plugin.fwk.model.StreamedGISData;
import org.gcube.data.publishing.gis.publisher.plugin.fwk.model.faults.BadRequestException;
import org.gcube.data.publishing.gis.publisher.plugin.fwk.model.faults.DataParsingException;
import org.gcube.data.publishing.gis.publisher.plugin.fwk.model.faults.NotSupportedDataTypeException;
import org.gcube.data.publishing.gis.publisher.plugin.fwk.model.faults.UnreachableDestinationException;
import org.gcube.data.publishing.gis.publisher.plugin.fwk.model.requests.GISRequestConfiguration;
import org.gcube.data.publishing.gis.publisher.plugin.fwk.model.requests.LayerGenerationRequest;
import org.gcube.data.publishing.gis.publisher.plugin.fwk.model.requests.StyleGenerationRequest;
import org.gcube.data.publishing.gis.publisher.plugin.fwk.model.requests.WMSGroupGenerationRequest;
import org.gcube.data.publishing.gis.publisher.plugin.fwk.model.response.GeneratedLayerReport;
import org.gcube.data.publishing.gis.publisher.plugin.fwk.model.response.GeneratedStyleReport;
import org.gcube.data.publishing.gis.publisher.plugin.fwk.model.response.PublishingReport;
import org.gcube.data.publishing.gis.publisher.plugin.fwk.model.response.PublishingStatus;
import org.gcube.data.streams.Stream;

public class GeoServerPlugin implements GISInteraction {

	private static final String crs="GEOGCS[\"WGS 84\", DATUM[\"World Geodetic System 1984\", SPHEROID[\"WGS 84\", 6378137.0, 298.257223563, AUTHORITY[\"EPSG\",\"7030\"]],"+ 
	"AUTHORITY[\"EPSG\",\"6326\"]], PRIMEM[\"Greenwich\", 0.0, AUTHORITY[\"EPSG\",\"8901\"]],  UNIT[\"degree\", 0.017453292519943295],"+ 
	"AXIS[\"Geodetic longitude\", EAST],  AXIS[\"Geodetic latitude\", NORTH],  AUTHORITY[\"EPSG\",\"4326\"]]";
	
	private static GCUBELog logger= new GCUBELog(GeoServerPlugin.class);
	
	
	private CSquareConverter converter; 
	private GeoCaller caller;
	private DBDescriptor postgis;
	private GeoServerDescriptor geoServer;
	private PublishingReport currentActivityReport =new PublishingReport();
	
	

	public GeoServerPlugin(CSquareConverter converter,GeoCaller caller,DBDescriptor db, GeoServerDescriptor geoServer) {
		this.geoServer=geoServer;
		this.converter=converter;
		this.caller=caller;
		this.postgis=db;		
	}

	@Override
	public PublishingReport publishData(GISData toPublishData,
			GISRequestConfiguration config) throws DataParsingException,
			NotSupportedDataTypeException, UnreachableDestinationException {
		try{
			if(toPublishData instanceof GISFile) readFile((GISFile) toPublishData);
			else {
				Table table=readStream((StreamedGISData)toPublishData);
				logger.trace("Created table "+table);
				if(config.getToGenerateStyles()!=null)
					for(StyleGenerationRequest styleReq:config.getToGenerateStyles()){
						if(styleReq!=null)
							try {
								currentActivityReport.getGeneratedStyles().put(styleReq.getNameStyle(),generateStyle(styleReq));
							} catch (Exception e) {
								logger.error("Unaebl to generate style "+styleReq,e);
								currentActivityReport.getErrorMessages().put(styleReq.getNameStyle(), e.getMessage());
							}
					}
				if(config.getToGenerateLayers()!=null)
					for(LayerGenerationRequest layerRequest:config.getToGenerateLayers()){
						if(layerRequest!=null)
							try {
								currentActivityReport.getGeneratedLayers().put(layerRequest.getName(),generateLayer(table, layerRequest));
							} catch (Exception e) {
								logger.error("Unaebl to generate layer "+layerRequest,e);
								currentActivityReport.getErrorMessages().put(layerRequest.getName(), e.getMessage());
							}
					}
				if(config.getToGenerateWMSGroups()!=null)
					for(WMSGroupGenerationRequest groupRequest : config.getToGenerateWMSGroups())
						if(groupRequest!=null)
							currentActivityReport.getErrorMessages().put(groupRequest.getToGenerateGroupName(), "WMS Group are not supported by this plugin");
						
				
				
				currentActivityReport.setStatus(PublishingStatus.COMPLETED);
			}
			return currentActivityReport;
		}catch (DataParsingException e){
			throw e;
		}catch (IOException e){
			throw new DataParsingException();
		} 
	}


	private void readFile(GISFile toRead)throws IOException{
		logger.trace("Received GIS File "+toRead);		
		logger.trace("File size : "+toRead.getFile().getTotalSpace());

	}

	private Table readStream(StreamedGISData toRead) throws NotSupportedDataTypeException, DataParsingException, UnreachableDestinationException{
		logger.trace("Reading streaming data : "+toRead);
		if(toRead.getStreamedType().isAssignableFrom(CSquarePoint.class)) {
			// CSQUARE POINTS NEED TO BE CONVERTED			
			logger.trace("Requesting csquare conversion to Geometries..");
			StreamedGISData<CSquarePoint> toConvert=(StreamedGISData<CSquarePoint>)toRead;
			return readStream(new StreamedGISData<GeometryPoint>(converter.convert(toConvert.getStream()), GeometryPoint.class));

		}else if(toRead.getStreamedType().isAssignableFrom(GeometryPoint.class)){
			//GEOMETRIES ARE READY TO BE STORED IN POSTGIS
			Connection conn=null;
			try{
			logger.trace("Connecting to DB..");
			conn=DBUtils.getConnection(postgis.getEntryPoint(), postgis.getUser(), postgis.getPassword());

			Stream<GeometryPoint> stream=((StreamedGISData<GeometryPoint>)toRead).getStream();
			PreparedStatement ps=null;
			Table toFill=null;
			logger.trace("Reading stream..");
			boolean continueProcess =true; 
			while(stream.hasNext()&&continueProcess){
				GeometryPoint point=stream.next();
				try{
					if(toFill==null){
						toFill=DBUtils.createTable(conn,point);
						logger.debug("Created table "+toFill);
					}
					if(ps==null)ps=DBUtils.prepareStatementForInsert(conn,toFill,point);
					toFill.setRowCount(DBUtils.insertPoint(toFill,ps, point)+toFill.getRowCount());
				}catch(Exception e){					
					if(toFill==null||ps==null){
						logger.error("Unexpected Exception, going to stop process ",e);
						continueProcess=false;
					}logger.debug("Exception while inserting point ",e);
				}
			}
			if(continueProcess)conn.commit();
			return toFill;
			}catch (Exception e) {
				throw new UnreachableDestinationException(e);
			}
		}else throw new NotSupportedDataTypeException("Data type was "+toRead.getStreamedType());


	}

	/**
	 * Generate the style with the given parameters and sends it to GeoServer
	 * 
	 * @param req
	 * @return
	 * @throws Exception
	 */

	private GeneratedStyleReport generateStyle(StyleGenerationRequest req)throws UnreachableDestinationException,BadRequestException,Exception{
		GeneratedStyleReport report=new GeneratedStyleReport();
		report.setToGenerateStyle(req.getNameStyle());
		String actualStyleName=Utils.getUUID();
		
		logger.trace("Generating style "+req);
		String style;
		if(req.getTypeValue()==Integer.class){
			switch(req.getClusterScaleType()){
			case logarithmic : style=MakeStyle.createStyleLog(actualStyleName, req.getAttributeName().toLowerCase(), req.getnClasses(), req.getC1(), req.getC2(), req.getTypeValue(), Integer.parseInt(req.getMax()), Integer.parseInt(req.getMin()));
			break;
			default 	: style=MakeStyle.createStyle(actualStyleName, req.getAttributeName().toLowerCase(), req.getnClasses(), req.getC1(), req.getC2(), req.getTypeValue(), Integer.parseInt(req.getMax()), Integer.parseInt(req.getMin()));
			break;

			}
		}
		else if(req.getTypeValue()==Float.class){
			switch(req.getClusterScaleType()){
			case logarithmic : style=MakeStyle.createStyleLog(actualStyleName, req.getAttributeName().toLowerCase(), req.getnClasses(), req.getC1(), req.getC2(), req.getTypeValue(), Float.parseFloat(req.getMax()), Float.parseFloat(req.getMin()));
			break;
			default : 	style=MakeStyle.createStyle(actualStyleName, req.getAttributeName().toLowerCase(), req.getnClasses(), req.getC1(), req.getC2(), req.getTypeValue(), Float.parseFloat(req.getMax()), Float.parseFloat(req.getMin()));
			break;
			}
		}
		else throw new BadRequestException("Invalid type class : "+req.getTypeValue());
		logger.trace("Submitting style "+req+" with name : "+actualStyleName);		
		if(!caller.sendStyleSDL(style))throw new UnreachableDestinationException("Caller returned false");
		report.setGeneratedStyle(actualStyleName);
		report.setStyleURI("N/A");
		return report;
	}

	private GeneratedLayerReport generateLayer(Table table,LayerGenerationRequest request)throws UnreachableDestinationException,BadRequestException,Exception{
		GeneratedLayerReport report=new GeneratedLayerReport();
		String featureTable=table.getTableName();
		String layerName=request.getName();
		Map<String,String> meta=request.getMetaData();
		if(meta==null)meta=new HashMap<String, String>();
		logger.debug("Generating "+request);
		try{
			FeatureTypeRest featureTypeRest=new FeatureTypeRest();		
			featureTypeRest.setDatastore(geoServer.getDatastore());
			featureTypeRest.setEnabled(true);
			featureTypeRest.setLatLonBoundingBox(new BoundsRest(-180.0,180.0,-85.5,90.0,"EPSG:4326"));
			featureTypeRest.setNativeBoundingBox(new BoundsRest(-180.0,180.0,-85.5,90.0,"EPSG:4326"));
			featureTypeRest.setName(featureTable);
			featureTypeRest.setNativeName(featureTable);
			featureTypeRest.setProjectionPolicy("FORCE_DECLARED");
			featureTypeRest.setSrs("EPSG:4326");
			featureTypeRest.setNativeCRS(crs);
			featureTypeRest.setTitle(layerName);
			featureTypeRest.setWorkspace(geoServer.getWorkspace());
			StringBuilder description=new StringBuilder();
			
			meta.put("LAYER_NAME", featureTypeRest.getName());
			
			if(meta!=null)
				for(Entry<String,String> entry:meta.entrySet())
					description.append(entry.getKey()+" : "+entry.getValue()+" | ");
			logger.debug("Invoking Caller for registering layer : ");
			logger.debug("featureTypeRest.getNativeName : "+featureTypeRest.getNativeName());
			logger.debug("featureTypeRest.getTitle : "+featureTypeRest.getTitle());
			if (caller.addFeatureType(featureTypeRest,GeonetworkCategory.DATASETS,description.toString(),"")){
				logger.debug("Add feature type returned true .. waiting for GS "+Constants.get().getGSWaitTime()+" ms");
				try {
					Thread.sleep(Constants.get().getGSWaitTime());
				} catch (InterruptedException e) {}		
				
				//Replace selected styles with actual generated ones
				ArrayList<String> styles=new ArrayList<String>();
				for(String s:request.getToAssociateStyles()){
					if(currentActivityReport.getGeneratedStyles().containsKey(s)) styles.add(currentActivityReport.getGeneratedStyles().get(s).getGeneratedStyle());
					else if(currentActivityReport.getErrorMessages().containsKey(s))logger.trace("Skipped style "+s+", previous error was "+currentActivityReport.getErrorMessages().get(s));
					else styles.add(s);
				}
				logger.debug("To associate styles : "+styles+", default : "+styles.get(request.getDefaultStyleIndex()));
				if(caller.setLayer(featureTypeRest, styles.get(request.getDefaultStyleIndex()), styles)){
					report.setAssociatedStyles(styles);
					report.setGeneratedLayerName(featureTable);
					report.setToGenerateLayerName(request.getName());
					report.setLayerUri(caller.getCurrentWmsGeoserver()+"/wms/"+featureTable);
					return report;
				}else throw new BadRequestException("Unable to set Layer "+request);				
				
			}else	throw new UnreachableDestinationException("Add feature type returned false, layer has not been created.");				
			
			}catch(Exception e){
				logger.debug("Create layer threw an exception ",e);
				throw e;
			}
	}

	
}
