package org.gcube.data.spd.ncbi;

import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.gcube.common.core.resources.GCUBERuntimeResource;
import org.gcube.common.core.resources.runtime.AccessPoint;
import org.gcube.common.core.utils.logging.GCUBELog;
import org.gcube.data.spd.ncbi.capabilities.ClassificationCapabilityImpl;
import org.gcube.data.spd.ncbi.capabilities.NamesMappingImpl;
import org.gcube.data.spd.ncbi.connection.ConnectionPool;
import org.gcube.data.spd.ncbi.connection.ConnectionPoolException;
import org.gcube.data.spd.plugin.fwk.AbstractPlugin;
import org.gcube.data.spd.plugin.fwk.Capabilities;
import org.gcube.data.spd.plugin.fwk.Property;
import org.gcube.data.spd.plugin.fwk.SearchTypes;
import org.gcube.data.spd.plugin.fwk.capabilities.ClassificationCapability;
import org.gcube.data.spd.plugin.fwk.capabilities.NamesMappingCapability;
import org.gcube.data.spd.plugin.fwk.model.DataProvider;
import org.gcube.data.spd.plugin.fwk.model.DataSet;
import org.gcube.data.spd.plugin.fwk.model.ResultItem;
import org.gcube.data.spd.plugin.fwk.model.Taxon;
import org.gcube.data.spd.plugin.fwk.util.ElementProperty;
import org.gcube.data.spd.plugin.fwk.util.RepositoryInfo;
import org.gcube.data.spd.plugin.fwk.writers.ObjectWriter;


public class NcbiPlugin extends AbstractPlugin{

	static GCUBELog logger = new GCUBELog(NcbiPlugin.class);

	//POSTGRESQL CONNECTION

	public static String dbDriver = "org.postgresql.Driver";   
	//		public static String urlDump = "ftp://ftp.ncbi.nih.gov/pub/taxonomy/taxdump.tar.gz";
	public static String sqlfile = "/test.sql";

	public static String jdbc;
	public static String username;
	public static String password;
	public static String urlDump;
	

	//	public static Hashtable<String, String> tables = new Hashtable<String, String>();

	public final static String credits = "This information object has been generated via the Species Product Discovery service on XDATEX by interfacing with The National Center for Biotechnology Information (http://www.ncbi.nlm.nih.gov/)";
	public final static String citation = "Accessed through: The National Center for Biotechnology Information at http://www.ncbi.nlm.nih.gov/ on ";


	final static String zipFileName = "taxdump.tar.gz";
	final static String fileDump = "/ncbi_db.sql";
	final static String names[] = {"nodes", "delnodes", "merged", "citations", "names", "division", "gencode"};
	final static String tables[] = {"nodes", "delnodes", "merged", "citations", "names", "division", "gencode", "citation", "updates"};
	@Override
	public void initialize(GCUBERuntimeResource res) throws Exception {
		setUseCache(true);
		for (AccessPoint ap:res.getAccessPoints())
		{		
			if (ap.getEntryname().equals("ftp")) {
				urlDump = ap.getEndpoint();
			}
			else if (ap.getEntryname().equals("jdbc")) {
				jdbc = ap.getEndpoint();	
				username = ap.getUsername();
				password = ap.getPassword();			
			}
		}

		//		Boolean flag = false;
		//		for(String s : names){
		//			if (!(Utils.SQLTableExists(s)))
		//				flag = true;
		//			break;
		//		}
		//
		//		if (flag)
		//			Utils.createDB();
		//
		//		long update = Utils.lastupdate();
		//		new UpdateThread(update);

	}

	@Override
	public void update(GCUBERuntimeResource resource) throws Exception {
		for (AccessPoint ap:resource.getAccessPoints())
		{		
			if (ap.getEntryname().equals("ftp")) {
				urlDump = ap.getEndpoint();
			}
			else if (ap.getEntryname().equals("jdbc")) {
				jdbc = ap.getEndpoint();	
				username = ap.getUsername();
				password = ap.getPassword();			
			}
		}
		super.update(resource);
	}

	@Override
	public String getDescription() {
		return ("NCBI Plugin");
	}

	@Override
	public String getRepositoryName() {
		return ("NCBI");
	}

	@SuppressWarnings("serial")
	@Override
	public Set<SearchTypes> getSupportedSearch() {
		return new HashSet<SearchTypes>(){{add(SearchTypes.ScientificName); add(SearchTypes.CommonName);}};
	}

	@SuppressWarnings("serial")
	@Override
	public Set<Capabilities> getSupportedCapabilities() {
		return new HashSet<Capabilities>(){{add(Capabilities.Classification);add(Capabilities.NamesMapping);}};
	}

	@Override
	public ClassificationCapability getClassificationInterface() {
		return new ClassificationCapabilityImpl();
	}

	@Override
	public NamesMappingCapability getNamesMappingInterface() {
		return new NamesMappingImpl();
	}

	@Override
	public void searchByCommonName(String commonName, ObjectWriter<ResultItem> writer, Property... arg2) {

		logger.trace("searchByCommonName for "+commonName+" in NCBI...");
		ResultSet rs = null;
		ResultSet rs2 = null;
		ConnectionPool pool = null;
		Connection con = null;

		try {

			pool = ConnectionPool.getConnectionPool();
			con = pool.getConnection();
			Statement statement = con.createStatement();

			rs = getListCommonNames(commonName);

			if (rs!=null){
				while(rs.next()) {	
					String id = rs.getString(1);
					//				logger.trace("select b.name_txt, a.rank, a.parent_tax_id from nodes as a NATURAL JOIN names as b where b.name_class = 'scientific name' and a.tax_id =" + id );
					rs2 = statement.executeQuery("select b.name_txt, a.rank, a.parent_tax_id from nodes as a NATURAL JOIN names as b where b.name_class = 'scientific name' and a.tax_id =" + id );

					if(rs2.next()) {	

						String ScName = rs2.getString(1);
						String rank = rs2.getString(2);
						String idParent = rs2.getString(3);

						ResultItem item = new ResultItem(id, ScName);

						item.setCommonNames(Utils.getCommonNames(id));

						Calendar now = Calendar.getInstance();
						SimpleDateFormat format = new SimpleDateFormat("yyyy-MM-dd");
						item.setCredits(credits.replace("XDATEX", format.format(now.getTime())));
						item.setCitation(citation.replace("XDATEX", format.format(now.getTime())));

						if (rank.equals("no rank"))
							item.setRank(null);
						else					
							item.setRank(rank);

						List<String> citation = Utils.getCitation(idParent);
						for (String c: citation){
							ElementProperty property = new ElementProperty("Comments and References", c);
							item.addProperty(property);
						}

						//rs.setCitation(Utils.getCitation(idParent));

						DataSet dataSet = new DataSet("GenBank");
						dataSet.setName("GenBank taxonomy database");
						DataProvider dp = new DataProvider("GenBank");
						dp.setName("GenBank taxonomy database");
						dataSet.setDataProvider(dp);
						item.setDataSet(dataSet);

						if (!(idParent.equals("1")))	
							item.setParent(findTax(idParent));	            	      	

						if (writer.isAlive())
							writer.write(item);
						else
							break;
					}
				}
			}
			logger.trace("searchByCommonName finished for "+commonName+" in NCBI");
		}
		catch (SQLException sqlExcept)
		{
			logger.error("sql Error", sqlExcept);
		} catch (ConnectionPoolException e) {
			logger.error("ConnectionPoolException", e);
		} finally {
			writer.close();
			logger.trace("close");
			try {
				if (rs != null) 
					rs.close();
				if (rs2 != null) 
					rs2.close();
				if ((pool!=null) && (con!=null)){
					pool.releaseConnection(con);
				}
			} catch (SQLException ex) {
				logger.error("sql Error", ex);

			} catch (Exception e) {
				logger.error("General Error", e);
			}
		}
	}


	/**
	 * Get a list of common name id
	 */	
	private ResultSet getListCommonNames(String commonName) {

		ResultSet rs = null;
		ConnectionPool pool = null;
		Connection con = null;

		try {

			pool = ConnectionPool.getConnectionPool();
			con = pool.getConnection();
			Statement statement = con.createStatement();
			rs = statement.executeQuery("select distinct(tax_id) from names where name_class = 'common name' and UPPER(name_txt) like UPPER('%" + commonName + "%')");

		}
		catch (SQLException sqlExcept){
			logger.error("sql Error", sqlExcept);
		} catch (ConnectionPoolException e) {
			logger.error("ConnectionPoolException", e);
		} finally {
			if ((pool!=null) && (con!=null)){
				pool.releaseConnection(con);
			}

		}
		return rs;

	}

	@Override
	public void searchByScientificName(String scientificName, ObjectWriter<ResultItem> writer, Property... properties) {

		logger.trace("searchByScientificName for "+scientificName+" in NCBI...");

		ResultSet rs1 = null;
		ResultSet rs2 = null;

		ConnectionPool pool = null;
		Connection con = null;

		try{
			rs1 = getListScientificNameID(scientificName);
			if (rs1!=null){

				while(rs1.next()) {	
					String  id = rs1.getString(1);

					pool = ConnectionPool.getConnectionPool();
					con = pool.getConnection();
					Statement statement = con.createStatement();
					try {

						//				logger.trace("select b.name_txt, a.rank, a.parent_tax_id from nodes as a NATURAL JOIN names as b where b.name_class = 'scientific name' and a.tax_id =" + id );
						rs2 = statement.executeQuery("select b.name_txt, a.rank, a.parent_tax_id from nodes as a NATURAL JOIN names as b where b.name_class = 'scientific name' and a.tax_id =" + id );

						if(rs2.next()) {	

							String ScName = rs2.getString(1);
							String rank = rs2.getString(2);
							String idParent = rs2.getString(3);

							ResultItem rs = new ResultItem(id, ScName);

							rs.setCommonNames(Utils.getCommonNames(id));

							Calendar now = Calendar.getInstance();
							SimpleDateFormat format = new SimpleDateFormat("yyyy-MM-dd");
							rs.setCredits(credits.replace("XDATEX", format.format(now.getTime())));
							rs.setCitation(citation.replace("XDATEX", format.format(now.getTime())));

							rs.setAuthor(Utils.setAuthorship(id, ScName));

							if (rank.equals("no rank"))
								rs.setRank(null);
							else					
								rs.setRank(rank);

							List<String> citation = Utils.getCitation(idParent);
							for (String c: citation){
								ElementProperty property = new ElementProperty("Comments and References", c);
								rs.addProperty(property);
							}

							DataSet dataSet = new DataSet("GenBank");
							dataSet.setName("GenBank taxonomy database");
							DataProvider dp = new DataProvider("GenBank");
							dp.setName("GenBank taxonomy database");
							dataSet.setDataProvider(dp);
							rs.setDataSet(dataSet);

							if (!(idParent.equals("1")))	
								rs.setParent(findTax(idParent));	            	      	

							if (writer.isAlive())
								writer.write(rs);
							else
								break;
						}

						logger.trace("searchByCommonName finished for "+scientificName+" in NCBI");

					}
					catch (SQLException sqlExcept){
						logger.error("sql Error", sqlExcept);
					} finally {
						if ((pool!=null) && (con!=null)){
							pool.releaseConnection(con);
						}
						try {				
							if (rs2 != null) 
								rs2.close();
						} catch (SQLException ex) {
							logger.error("sql Error", ex);
						} catch (Exception e) {
							logger.error("General Error", e);
						}
					}
				}
			}
		} catch (ConnectionPoolException e) {
			logger.error("ConnectionPoolException", e);
		} catch (SQLException e) {
			logger.error("SQLException", e);
		}finally{
			writer.close();

			try {
				if (rs1 != null)
					rs1.close();
			} catch (SQLException e) {
				logger.error("sql Error", e);
			}
		}

	}

	private ResultSet getListScientificNameID(String scientificName) {
		ResultSet rs = null;
		ConnectionPool pool = null;
		Connection con = null;
		try {

			pool = ConnectionPool.getConnectionPool();
			con = pool.getConnection();
			Statement statement = con.createStatement();
			rs = statement.executeQuery("select distinct(tax_id) from names where name_class = 'scientific name' and UPPER(name_txt) like UPPER('%" + scientificName + "%')");

		}
		catch (SQLException sqlExcept){
			logger.error("sql Error", sqlExcept);
		} catch (ConnectionPoolException e) {
			logger.error("ConnectionPoolException", e);
		} finally {

			if ((pool!=null) && (con!=null)){
				pool.releaseConnection(con);
			}
		}
		return rs;	
	}


	@Override
	public RepositoryInfo getRepositoryInfo() {
		RepositoryInfo info = new RepositoryInfo("http://static.pubmed.gov/portal/portal3rc.fcgi/3417851/img/3242381", 
				"http://www.ncbi.nlm.nih.gov/", 
				"The National Center for Biotechnology Information advances science and health by providing access to biomedical and genomic information.The NCBI Taxonomy contains the names and phylogenetic lineages of more than 160,000 organisms that have molecular data in the NCBI databases. New taxa are added to the Taxonomy database as data are deposited for them.");
		return info;
	}


	//find taxonomy 
	public static Taxon findTax(String id_record) {		
		String id = null;
		try{
			id = Utils.getOriginalId(id_record);
		}
		catch (ArrayIndexOutOfBoundsException e){
			id = id_record;
			//			logger.error("id is already a original one", e);
		}
		Taxon t = null;

		ConnectionPool pool = null;
		Connection con = null;
		try {
			pool = ConnectionPool.getConnectionPool();
			con = pool.getConnection();
			Statement statement = con.createStatement();
			ResultSet rs = statement.executeQuery("select a.parent_tax_id, b.name_txt, a.rank from nodes as a NATURAL JOIN names as b where a.tax_id = " + id + " and b.name_class = 'scientific name'");
			if (rs!=null){
				if(rs.next())
				{	        	
					String id_parent = rs.getString(1);
					String rank = rs.getString(3);
					t = new Taxon(id+"");
					t.setRank(rank);
					t.setScientificName(rs.getString(2));
					if  ((id_parent != "1") && !(rank.equals("kingdom")))		
						t.setParent(findTax(id_parent));	
				}
				rs.close();
			}
		}
		catch (SQLException sqlExcept) {
			logger.error("sql Error",sqlExcept);
		} catch (ConnectionPoolException e) {
			logger.error("ConnectionPoolException",e);
		}finally {
			if ((pool!=null) && (con!=null)){
				pool.releaseConnection(con);
			}
		}   
		return t;		
	}

}
