package org.gcube.data.transfer.library.transferers;

import java.io.File;
import java.net.URL;

import lombok.extern.slf4j.Slf4j;

import org.gcube.data.transfer.library.TransferResult;
import org.gcube.data.transfer.library.client.Client;
import org.gcube.data.transfer.library.faults.FailedTransferException;
import org.gcube.data.transfer.library.faults.InitializationException;
import org.gcube.data.transfer.library.faults.InvalidSourceException;
import org.gcube.data.transfer.library.faults.SourceNotSetException;
import org.gcube.data.transfer.library.model.LocalSource;
import org.gcube.data.transfer.library.model.Source;
import org.gcube.data.transfer.library.model.StorageSource;
import org.gcube.data.transfer.library.model.URLSource;
import org.gcube.data.transfer.model.TransferCapabilities;
import org.gcube.data.transfer.model.TransferRequest;
import org.gcube.data.transfer.model.TransferTicket;
import org.gcube.data.transfer.model.TransferTicket.Status;

@Slf4j
public abstract class Transferer {

	protected Client client;

	protected Transferer(Client client){
		this.client=client;
	}

	
	
	
	protected Source source=null;
	protected boolean prepared=false;

	public Transferer localFile(File f) throws InvalidSourceException{
		source=new LocalSource(f);
		return this;
	}

	public Transferer localFile(String path) throws InvalidSourceException{
		try{
			File f=new File(path);
			return localFile(f);
		}catch(NullPointerException e){
			throw new InvalidSourceException("Path is null",e);
		}

	}

	public Transferer storageFileId(String fileId) throws InvalidSourceException{
		source=new StorageSource(fileId);
		return this;
	}

	public Transferer fromURL(URL sourceURL) throws InvalidSourceException{		
		source=new URLSource(sourceURL);
		return this;
	}


	public TransferResult transfer() throws SourceNotSetException, InvalidSourceException, FailedTransferException, InitializationException{
		try{
			checkSource();
			prepare();
			TransferRequest request=prepareRequest();
			log.debug("Request is {}, sending it to {}",request,client.getEndpoint());
			TransferResult result=doTheTransfer(request);
			return result;
		}finally{
			clean();
		}
	}

	protected TransferResult doTheTransfer(TransferRequest request) throws FailedTransferException{
		TransferTicket submissionResponse= client.submit(request);
		boolean continuePolling=true;
		TransferTicket ticket=null;
		do{
			ticket=client.getTransferStatus(submissionResponse.getId());
			System.out.println("Status : "+ticket);
			continuePolling=ticket.getStatus().equals(Status.PENDING)||ticket.getStatus().equals(Status.TRANSFERRING)||ticket.getStatus().equals(Status.WAITING);
			try{
				Thread.sleep(500);
			}catch(InterruptedException e){}
		}while(continuePolling);
		if(ticket.getStatus().equals(Status.ERROR)) throw new FailedTransferException("Remote Message : "+ticket.getMessage());
		if(ticket.getStatus().equals(Status.STOPPED)) throw new FailedTransferException("Stopped transfer : "+ticket.getMessage());
		long elapsedTime=System.currentTimeMillis()-ticket.getSubmissionTime().value.getTimeInMillis();
		return new TransferResult(source, client.getEndpoint(), elapsedTime, ticket.getTransferredBytes(), ticket.getDestinationFileName());
	}

	protected void checkSource() throws SourceNotSetException, InvalidSourceException{
		if(source==null) throw new SourceNotSetException();
		source.validate();
	}

	protected abstract TransferRequest prepareRequest() throws InitializationException;


	protected void prepare() throws InitializationException{
		prepared=true;
	}
	protected void clean(){
		
	}

	public TransferCapabilities getDestinationCapabilities(){
		return client.getCapabilties();
	}
	
	
}
