package org.gcube.datatransfer.resolver.http;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URL;
import java.net.URLConnection;
import java.net.URLStreamHandler;

import javax.servlet.ServletConfig;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.commons.io.IOUtils;
import org.gcube.contentmanager.storageclient.model.protocol.smp.Handler;
import org.gcube.contentmanager.storageclient.model.protocol.smp.SMPURLConnection;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;



/**
 * 
 * @author Andrea Manzi(CERN)
 * updated by Francesco Mangiacrapa
 * 
 *
 */
public class HttpResolver extends HttpServlet {
	
	protected static final String SMP_URI = "smp-uri";
	protected static final String VALIDATION = "validation";
	protected static final String CONTENT_TYPE = "contentType";
	protected static final String FILE_NAME = "fileName";

	private static final long serialVersionUID = 1L;

	/** The logger. */
	private static final Logger logger = LoggerFactory.getLogger(HttpResolver.class);

	public void init(ServletConfig conf) throws ServletException {
		Handler.activateProtocol();
		super.init(conf);

	}

	protected void doGet(HttpServletRequest request, HttpServletResponse response) throws IOException {
		
		String uri =null;
		String fileName =null;
		String contentType =null;
		boolean validatingURI = false;

//		logger.info("The http session id is: " + request.getSession().getId());

		uri = request.getParameter(SMP_URI);

		if (uri == null || uri.equals("")) {
			logger.debug("URI not found");
			response.sendError(404);
			return;
		}
		
		fileName  = request.getParameter(FILE_NAME);
		
		
		if (fileName == null || fileName.equals("")) {
			logger.debug("fileName not found");
			fileName = null;
		}

		contentType  = request.getParameter(CONTENT_TYPE);
		
		if (contentType == null || contentType.equals("")) {
			logger.debug("contentType not found");
			contentType = null;
		}
		
		String validation = request.getParameter(VALIDATION);
		validatingURI = Boolean.parseBoolean(validation);
		logger.debug("validation? "+validatingURI);
		
		//we should not unescape the filename with spaces
		
		int index= uri.indexOf("?");
		if ( index!= -1)
		{
			String firsPart = uri.substring(0, index);
			String secondPart=  uri.substring( index+1);
			secondPart.replace(" ","+");//the char + is removed when the servlet is doing unescaping of the query paramenters, we just put it back
			uri= firsPart+"?"+secondPart;
		}
		else uri = uri.replace(" ","+");//the char + is removed when the servlet is doing unescaping of the query paramenters, we just put it back

		
		logger.debug("URI = "+ uri);
		InputStream in = null;
		try {

			OutputStream out = response.getOutputStream();
			
			if (fileName != null)
				response.addHeader("content-disposition", "attachment; filename=" +fileName);
			else
				response.addHeader("content-disposition", "attachment; filename=fromStorageManager");
			
			if (contentType!= null)
				response.setContentType(contentType);
			else	
				response.setContentType("unknown/unknown");
			
			
			URL url = new URL(null, uri, new URLStreamHandler() {
				
				@Override
				protected URLConnection openConnection(URL u) throws IOException {
					return new SMPURLConnection(u);
				}
			});
		
			URLConnection uc = null;

			try {
				uc = ( URLConnection ) url.openConnection ( );
				in = uc.getInputStream();
			}
			catch(Exception e){
				response.sendError(404);
				logger.error("URLConnection Exception:", e);
				return;
			}
			
			//CASE InputStream NULL
			if(in==null){
				logger.warn("Input stream is null, sending status error 404");
				sendErrorQuietly(response, 404);
				return;
			}

			//CASE VALIDATION
			if(validatingURI){
				byte[] bytes = new byte[1]; //1B
				int c = in.read(bytes);
				logger.info(c+" byte read from InputStream");
				if(c>0){
					logger.info("at least 1 byte read, returning status 200");
					IOUtils.closeQuietly(in);
					response.setStatus(200);
					return;
				}	
			}
	
			/*
			IOUtils.copy(in, out);
		
			out.flush();
			out.close();
			in.close();
			*/
			
			//CHANGED BY FRANCESCO M.
			try {
				
				IOUtils.copy(in, out);
				
			} catch (IOException e){
				logger.warn("IOException class name: "+e.getClass().getSimpleName());
				if (e.getClass().getSimpleName().equals("ClientAbortException"))
					logger.warn("Skipping ClientAbortException: "+e.getMessage());
				else
					throw e; //Sending Exceptions
				
			}	catch (NullPointerException e) {
				logger.warn("NullPointerException during copy, skipping printStrackTrace");
				sendErrorQuietly(response, 404);
				
			} finally {
				IOUtils.closeQuietly(in);
				IOUtils.closeQuietly(out);
			}
			
		} catch (Exception e) {
			logger.error("Exception:", e);
//			response.sendError(404);
			IOUtils.closeQuietly(in);
			sendErrorQuietly(response, 404);
			return;
		}

	}
	
	protected void sendErrorQuietly(HttpServletResponse response, int code){
		
		if(response!=null){
			try {
				response.sendError(code);
				logger.info("Response sent error: "+code);
			} catch (IOException ioe) {
				 // ignore
			}
		}
	}
	
	protected void doPost(HttpServletRequest request,
			HttpServletResponse response) throws IOException {
		doGet(request,response);
	}
	

}