/**
 * 
 */
package org.gcube.dataanalysis.copernicus.cmems.importer.api;

import java.text.ParseException;
import java.util.Calendar;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.TreeSet;

import javax.xml.bind.annotation.XmlElement;
import javax.xml.bind.annotation.XmlElementWrapper;
import javax.xml.bind.annotation.XmlRootElement;

import org.apache.commons.codec.digest.DigestUtils;
import org.apache.commons.lang.builder.ToStringBuilder;
import org.gcube.dataanalysis.datasetimporter.util.TimeUtil;

/**
 * @author Paolo Fabriani
 *
 */
@XmlRootElement
public class ImportOptions {

    /**
     * In days
     */
    private Integer backTime;

    /**
     * The name of the dataset
     */
    private String dataset;

    /**
     * In days
     */
    private String importSchedule;

    /**
     * The endpoint of the Motu server.
     */
    private String motu;

    /**
     * The name of the product.
     */
    private String product;

    /**
     * Can be day, month, year
     */
    private ChunkTimespan chunkSpan;

    /**
     * Time window
     */
    private Calendar tLo, tHi;

    /**
     * The variables to import
     */
    private Collection<String> variables;

    /**
     * Longitude range
     */
    private Double xLo, xHi;

    /**
     * Latitude range
     */
    private Double yLo, yHi;

    /**
     * Depth range
     */
    private Double zLo, zHi;

    public ImportOptions() {
    }

    public ImportOptions(Map<String, String> p) throws ParseException {
        this();
        if (p.get("chunkSpan") != null) {
            switch (p.get("chunkSpan")) {
            case "day":
                this.setChunkSpan(ChunkTimespan.DAY);
                break;
            case "month":
                this.setChunkSpan(ChunkTimespan.MONTH);
                break;
            case "year":
                this.setChunkSpan(ChunkTimespan.YEAR);
                break;
            }
        }
        if (p.get("dataset") != null) {
            this.setDataset(p.get("dataset"));
        }
        if (p.get("motu") != null) {
            this.setMotu(p.get("motu"));
        }
        if (p.get("product") != null) {
            this.setProduct(p.get("product"));
        }
        if (p.get("backTime") != null) {
            this.setBackTime(Integer.parseInt(p.get("backTime")));
        }
        if (p.get("importSchedule") != null) {
            this.setImportSchedule(p.get("importSchedule"));
        }
        if (p.get("tLo") != null) {
            this.settLo(TimeUtil.toCalendar(p.get("tLo")));
        }
        if (p.get("tHi") != null) {
            this.settHi(TimeUtil.toCalendar(p.get("tHi")));
        }
        if (p.get("xLo") != null) {
            this.setxLo(Double.parseDouble(p.get("xLo")));
        }
        if (p.get("xHi") != null) {
            this.setxHi(Double.parseDouble(p.get("xHi")));
        }
        if (p.get("yLo") != null) {
            this.setyLo(Double.parseDouble(p.get("yLo")));
        }
        if (p.get("yHi") != null) {
            this.setyHi(Double.parseDouble(p.get("yHi")));
        }
        if (p.get("zLo") != null) {
            this.setzLo(Double.parseDouble(p.get("zLo")));
        }
        if (p.get("zHi") != null) {
            this.setzHi(Double.parseDouble(p.get("zHi")));
        }
        if (p.get("variables") != null) {
            String[] vars = p.get("variables").split(",");
            for (String v : vars) {
                this.addVariable(v.trim());
            }
        }
    }

    public Integer getBackTime() {
        return backTime;
    }

    public void setBackTime(int backTime) {
        this.backTime = backTime;
    }

    public String getDataset() {
        return dataset;
    }

    public void setDataset(String dataset) {
        this.dataset = dataset;
    }

    public String getMotu() {
        return motu;
    }

    public void setMotu(String motu) {
        this.motu = motu;
    }

    public String getProduct() {
        return product;
    }

    public void setProduct(String product) {
        this.product = product;
    }

    public ChunkTimespan getChunkSpan() {
        return chunkSpan;
    }

    public void setChunkSpan(ChunkTimespan chunkSpan) {
        this.chunkSpan = chunkSpan;
    }

    public Calendar gettLo() {
        return tLo;
    }

    public void settLo(Calendar tLo) {
        this.tLo = tLo;
    }

    public Calendar gettHi() {
        return tHi;
    }

    public void settHi(Calendar tHi) {
        this.tHi = tHi;
    }

    @XmlElement(name="variable")
    @XmlElementWrapper(name="variables")
    public Collection<String> getVariables() {
        return variables;
    }

    public void addVariable(String variable) {
        if (this.variables == null)
            this.variables = new TreeSet<>();
        this.variables.add(variable);
    }

    public void setVariables(Collection<String> variables) {
        this.variables = variables;
    }

    public Double getxLo() {
        return xLo;
    }

    public void setxLo(Double xLo) {
        this.xLo = xLo;
    }

    public Double getxHi() {
        return xHi;
    }

    public void setxHi(Double xHi) {
        this.xHi = xHi;
    }

    public Double getyLo() {
        return yLo;
    }

    public void setyLo(Double yLo) {
        this.yLo = yLo;
    }

    public Double getyHi() {
        return yHi;
    }

    public void setyHi(Double yHi) {
        this.yHi = yHi;
    }

    public Double getzLo() {
        return zLo;
    }

    public void setzLo(Double zLo) {
        this.zLo = zLo;
    }

    public Double getzHi() {
        return zHi;
    }

    public void setzHi(Double zHi) {
        this.zHi = zHi;
    }

    public Map<String, String> toMap() {
        Map<String, String> out = new HashMap<>();
        if (this.getChunkSpan() != null) {
            out.put("chunkSpan", this.getChunkSpan().toString());
        }
        if (this.getDataset() != null) {
            out.put("dataset", this.getDataset());
        }
        if (this.getMotu() != null) {
            out.put("motu", this.getMotu());
        }
        if (this.getProduct() != null) {
            out.put("product", this.getProduct());
        }
        if (this.getBackTime() != null) {
            out.put("backTime", this.getBackTime().toString());
        }
        if (this.getImportSchedule() != null) {
            out.put("importSchedule", this.getImportSchedule().toString());
        }
        if (this.gettHi() != null) {
            out.put("tHi", TimeUtil.toString(this.gettHi()));
        }
        if (this.gettLo() != null) {
            out.put("tLo", TimeUtil.toString(this.gettLo()));
        }
        if (this.getxHi() != null) {
            out.put("xHi", this.getxHi().toString());
        }
        if (this.getxLo() != null) {
            out.put("xLo", this.getxLo().toString());
        }
        if (this.getyHi() != null) {
            out.put("yHi", this.getyHi().toString());
        }
        if (this.getyLo() != null) {
            out.put("yLo", this.getyLo().toString());
        }
        if (this.getzHi() != null) {
            out.put("zHi", this.getzHi().toString());
        }
        if (this.getzLo() != null) {
            out.put("zLo", this.getzLo().toString());
        }
        if (this.getVariables() != null) {
            out.put("variables", "" + String.join(", ", this.getVariables()));
        }
        return out;
    }

    public String getHash() {

        String S = "|";

        /*
         * Note: motu server, backtime and importFrequency does not change the
         * content of the dataset. So they're not used to distinguish two
         * imported datasets.
         */
        String text = S;

        text += (this.getProduct() == null ? "PR" : this.getProduct()) + S;
        text += (this.getDataset() == null ? "DS" : this.getDataset()) + S;

        text += (this.getChunkSpan() == null ? "CS" : this.getChunkSpan()) + S;

        text += (this.getxLo() == null ? "XL" : this.getxLo()) + S;
        text += (this.getxHi() == null ? "XH" : this.getxHi()) + S;

        text += (this.getyLo() == null ? "YL" : this.getyLo()) + S;
        text += (this.getyHi() == null ? "YH" : this.getyHi()) + S;

        text += (this.getzLo() == null ? "ZL" : this.getzLo()) + S;
        text += (this.getzHi() == null ? "ZH" : this.getzHi()) + S;

        text += (this.gettLo() == null ? "TL"
                : TimeUtil.toString(this.gettLo())) + S;
        text += (this.gettHi() == null ? "TH"
                : TimeUtil.toString(this.gettHi())) + S;

        if (this.getVariables() == null || this.getVariables().isEmpty()) {
            text += "AV" + S;
        } else {
            text += String.join(S, this.getVariables()) + S;
        }
        // System.out.println(text);
        return DigestUtils.md5Hex(text);
    }

    public String getImportSchedule() {
        return importSchedule;
    }

    public void setImportSchedule(String importSchedule) {
        this.importSchedule = importSchedule;
    }

    public void setBackTime(Integer backTime) {
        this.backTime = backTime;
    }

    public String toString() {
        return ToStringBuilder.reflectionToString(this);
    }

}
