/**
 * 
 */
package org.gcube.dataanalysis.copernicus.cmems.importer.client;

import java.net.URL;
import java.util.Arrays;
import java.util.List;

import javax.ws.rs.client.Client;
import javax.ws.rs.client.ClientBuilder;
import javax.ws.rs.client.Entity;
import javax.ws.rs.client.Invocation;
import javax.ws.rs.client.WebTarget;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;

import org.gcube.common.authorization.library.provider.SecurityTokenProvider;
import org.gcube.dataanalysis.copernicus.cmems.importer.api.ImportOptions;
import org.gcube.dataanalysis.copernicus.cmems.importer.task.Execution;
import org.gcube.dataanalysis.copernicus.cmems.importer.task.ExecutionReport;
import org.gcube.dataanalysis.copernicus.cmems.importer.task.ImportTask;
import org.gcube.dataanalysis.copernicus.cmems.model.CmemsProduct;

/**
 * A client to browse scheduled import tasks, executions and reports.
 * 
 * @author Paolo Fabriani
 *
 */
public class TaskManagerClient {

    private URL endpoint;

    private static final String TASKS = "tasks";
    private static final String EXECUTIONS = "executions";
    private static final String REPORTS = "reports";
    
    public TaskManagerClient(URL endpoint) {
        this.endpoint = endpoint;
    }

    public List<ImportTask> getTasks(String user, String scope, String token) {
        // TODO: add query parameters: user, scope, token
        Invocation.Builder ib = this.getBuilderForPath(TASKS);
        return Arrays.asList(ib.get(ImportTask[].class));
    }
    
    public ImportTask getTask(String taskId) {
        Invocation.Builder ib = this.getBuilderForPath(TASKS, taskId);
        return ib.get(ImportTask.class);
    }

    public List<Execution> getTaskExecutions(String taskId) {
        Invocation.Builder ib = this.getBuilderForPath(TASKS, taskId, EXECUTIONS);
        return Arrays.asList(ib.get(Execution[].class));
    }

    public Execution getTaskExecution(String taskId, String executionId) {
        Invocation.Builder ib = this.getBuilderForPath(TASKS, taskId, EXECUTIONS, executionId);
        return ib.get(Execution.class);
    }

    public List<ExecutionReport> getExecutionReports(String taskId, String executionId) {
        Invocation.Builder ib = this.getBuilderForPath(TASKS, taskId, EXECUTIONS, executionId, REPORTS);
        return Arrays.asList(ib.get(ExecutionReport[].class));
    }

    public ExecutionReport getExecutionReport(String taskId, String executionId, String reportId) {
        Invocation.Builder ib = this.getBuilderForPath(TASKS, taskId, EXECUTIONS, executionId, REPORTS, reportId);
        return ib.get(ExecutionReport.class);
    }

    public void addTask(ImportOptions options) throws Exception {
        Invocation.Builder ib = this.getBuilderForPath(TASKS);
        Response response = ib.post(Entity.entity(options, MediaType.APPLICATION_XML));
        if(response.getStatus()!=200) {
            throw new Exception(response.getStatus() + " " + response.getEntity());
        }
    }

    public void addExecution(String taskId, Execution execution) throws Exception {
        Invocation.Builder ib = this.getBuilderForPath(TASKS, taskId, EXECUTIONS);
        Response response = ib.post(Entity.entity(execution, MediaType.APPLICATION_XML));
        if(response.getStatus()!=200) {
            throw new Exception(response.getStatus() + " " + response.getEntity());
        }
    }

    public void updateExecution(String taskId, Execution execution) throws Exception {
        Invocation.Builder ib = this.getBuilderForPath(TASKS, taskId, EXECUTIONS, execution.getId());
        Response response = ib.put(Entity.entity(execution, MediaType.APPLICATION_XML));
        if(response.getStatus()!=200) {
            throw new Exception(response.getStatus() + " " + response.getEntity());
        }
    }

    public void addReport(String taskId, String executionId, ExecutionReport report) throws Exception {
        Invocation.Builder ib = this.getBuilderForPath(TASKS, taskId, EXECUTIONS, executionId, REPORTS);
        Response response =  ib.post(Entity.entity(report, MediaType.APPLICATION_XML));
        if(response.getStatus()!=200) {
            throw new Exception(response.getStatus() + " " + response.getEntity());
        }
    }

    public void updateReport(String taskId, String executionId, ExecutionReport report) throws Exception {
        Invocation.Builder ib = this.getBuilderForPath(TASKS, taskId, EXECUTIONS, executionId, REPORTS, report.getName());
        Response response = ib.put(Entity.entity(report, MediaType.APPLICATION_XML));
        if(response.getStatus()!=200) {
            throw new Exception(response.getStatus() + " " + response.getEntity());
        }
    }
    
    private Invocation.Builder getBuilderForPath(String ... coords) {
        Client client = ClientBuilder.newClient();
        WebTarget target = client.target(this.endpoint.toString());
        for(String c:coords) {
            target = target.path(c);
        }
        String gcubeToken = SecurityTokenProvider.instance.get();
        return target.request(MediaType.APPLICATION_XML).header("gcube-token", gcubeToken);
    }

}
