/**
 * 
 */
package org.gcube.dataanalysis.copernicus.cmems.importer.client;

import java.util.Arrays;
import java.util.Collection;
import java.util.Vector;

import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.CommandLineParser;
import org.apache.commons.cli.DefaultParser;
import org.apache.commons.cli.HelpFormatter;
import org.apache.commons.cli.Option;
import org.apache.commons.cli.Options;
import org.apache.commons.cli.ParseException;
import org.gcube.dataanalysis.copernicus.cmems.client.CmemsClient;
import org.gcube.dataanalysis.copernicus.cmems.client.SearchOptions;
import org.gcube.dataanalysis.copernicus.cmems.importer.api.ChunkTimespan;
import org.gcube.dataanalysis.copernicus.cmems.importer.api.ImportOptions;
import org.gcube.dataanalysis.copernicus.cmems.model.CmemsProduct;
import org.gcube.dataanalysis.copernicus.motu.client.DownloadRequest;
import org.gcube.dataanalysis.copernicus.motu.client.MotuClient;
import org.gcube.dataanalysis.copernicus.motu.model.Axis;
import org.gcube.dataanalysis.copernicus.motu.model.ProductMetadataInfo;
import org.gcube.dataanalysis.copernicus.motu.model.Variable;
import org.gcube.dataanalysis.datasetimporter.util.TimeUtil;

/**
 * @author Paolo Fabriani
 *
 */
public class CmemsImporterClientCLI {

    private static Options searchOptions;
    
    private static Options productOptions;
    
    private static Options scheduleOptions;
    
    private static Options unscheduleOptions;
    
    private static Options listOptions;

    static {
        
        Option help = new Option("h", "help", false, "print this message");

        Option containsKey = Option.builder("k").longOpt("key").hasArgs()
                .desc("a key to appear in the dataset metadata").build();

        Option doesNotContainKey = Option.builder("K").longOpt("nokey")
                .hasArgs().desc("a key to not appear in the dataset metadata")
                .build();

        Option regdomain = Option.builder("r").longOpt("regional-domain")
                .hasArg().desc("regional domain").build();

        Option ignoreCase = Option.builder("i").longOpt("ignore-case")
                .hasArg(false).desc("ignore case").build();

        Option count = Option.builder("c").longOpt("count-matches")
                .hasArg(false).desc("count the matching datasets").build();

        Option variable = Option.builder("v").hasArg().longOpt("variable")
                .desc("variable").build();

        Option from = Option.builder("f").hasArg().longOpt("from")
                .desc("from").build();

        Option to = Option.builder("t").hasArg().longOpt("to")
                .desc("to").build();

        Option wholeTimeRange = Option.builder("w").hasArg(false)
                .longOpt("whole-time-range")
                .desc("if set the search results will only show products containing the whole selected time range")
                .build();
        
        // search options
        searchOptions = new Options();
        searchOptions.addOption(containsKey);
        searchOptions.addOption(doesNotContainKey);
        searchOptions.addOption(ignoreCase);
        searchOptions.addOption(count);
        searchOptions.addOption(regdomain);
        searchOptions.addOption(variable);
        searchOptions.addOption(from);
        searchOptions.addOption(to);
        searchOptions.addOption(wholeTimeRange);
        searchOptions.addOption(help);
        
        Option datasetName = Option.builder("d").hasArgs().desc("The name of the dataset").longOpt("dataset").build();
        Option productName = Option.builder("p").hasArgs().desc("The name of the product").longOpt("product").build();
        Option motuServer = Option.builder("m").hasArgs().desc("The endpoint of the motu server").longOpt("motu-server").build();
        
        // product options
        productOptions = new Options();
        productOptions.addOption(help);
        productOptions.addOption(datasetName);
        productOptions.addOption(productName);
        productOptions.addOption(motuServer);
        
        // schedule options
        Option check = Option.builder("c").longOpt("check").hasArg(false).desc("Get most relevant info before scheduling a job, without actually submitting it").build();
        Option tlo = Option.builder().longOpt("tlo").hasArg().desc("Low time").build();
        Option thi = Option.builder().longOpt("thi").hasArg().desc("High time").build();
        Option xlo = Option.builder().longOpt("xlo").hasArg().desc("Low longitude").build();
        Option xhi = Option.builder().longOpt("xhi").hasArg().desc("High longitude").build();
        Option ylo = Option.builder().longOpt("ylo").hasArg().desc("Low latitude").build();
        Option yhi = Option.builder().longOpt("yhi").hasArg().desc("High latitude").build();
        Option frequency = Option.builder("f").hasArg().longOpt("import-frequency").build();
        Option back = Option.builder("b").hasArg().longOpt("back-time").build();
        Option chunkSpan = Option.builder("s").hasArg().longOpt("chunk-span").build();
        Option postProcess = Option.builder("p").longOpt("post-process").hasArg().build();
        
        scheduleOptions = new Options();
        scheduleOptions.addOption(check);
        scheduleOptions.addOption(tlo);
        scheduleOptions.addOption(thi);
        scheduleOptions.addOption(xlo);
        scheduleOptions.addOption(xhi);
        scheduleOptions.addOption(ylo);
        scheduleOptions.addOption(yhi);
        scheduleOptions.addOption(frequency);
        scheduleOptions.addOption(back);
        scheduleOptions.addOption(chunkSpan);
        scheduleOptions.addOption(postProcess);
        scheduleOptions.addOption(datasetName);
        scheduleOptions.addOption(productName);
        scheduleOptions.addOption(motuServer);
        scheduleOptions.addOption(variable);
        scheduleOptions.addOption(help);
     
        unscheduleOptions = new Options();
        Option jobId = Option.builder("j").hasArg().longOpt("job-id").build();
        unscheduleOptions.addOption(jobId);
        unscheduleOptions.addOption(help);
        
        listOptions = new Options();
        listOptions.addOption(help);
        
    }

    public CmemsImporterClientCLI() {

    }
    
    private static void unschedule(CommandLine line) throws Exception {
        if (line.hasOption("h")) {
            HelpFormatter formatter = new HelpFormatter();
            formatter.printHelp("gcube-cmems", unscheduleOptions);
        } else {
            String jobId = null;
            if(line.hasOption("j"))
                jobId = line.getOptionValue("j");
            else
                throw new Exception("missing mandatory jobId");
            CmemsImporterClient client = new CmemsImporterClient();
            client.unscheduleTask(jobId);
        }
    }

    private static void list(CommandLine line) throws Exception {
        if (line.hasOption("h")) {
            HelpFormatter formatter = new HelpFormatter();
            formatter.printHelp("gcube-cmems", listOptions);
        } else {
            CmemsImporterClient client = new CmemsImporterClient();
            client.listTasks();
        }
    }

    private static void schedule(CommandLine line) throws Exception {
        if (line.hasOption("h")) {
            HelpFormatter formatter = new HelpFormatter();
            formatter.printHelp("gcube-cmems", scheduleOptions);
        } else {
            ImportOptions options = new ImportOptions();
            if(line.hasOption("tlo"))
                options.settLo(TimeUtil.toCalendar(line.getOptionValue("tlo")));
            if(line.hasOption("thi"))
                options.settHi(TimeUtil.toCalendar(line.getOptionValue("thi")));
            if(line.hasOption("xlo"))
                options.setxLo(Double.parseDouble(line.getOptionValue("xlo")));
            if(line.hasOption("xhi"))
                options.setxHi(Double.parseDouble(line.getOptionValue("xhi")));
            if(line.hasOption("ylo"))
                options.setyLo(Double.parseDouble(line.getOptionValue("ylo")));
            if(line.hasOption("yhi"))
                options.setyHi(Double.parseDouble(line.getOptionValue("yhi")));
            if(line.hasOption("m"))
                options.setMotu(line.getOptionValue("m"));
            if(line.hasOption("p"))
                options.setProduct(line.getOptionValue("p"));
            if(line.hasOption("d"))
                options.setDataset(line.getOptionValue("d"));
            if(line.hasOption("f"))
                options.setImportSchedule(line.getOptionValue("f"));
            if(line.hasOption("b"))
                options.setBackTime(Integer.parseInt(line.getOptionValue("b")));
            if(line.hasOption("s")) {
                String option = line.getOptionValue("s");
                if("year".equals(option))
                    options.setChunkSpan(ChunkTimespan.YEAR);
                else if("month".equals(option))
                    options.setChunkSpan(ChunkTimespan.MONTH);
                else if("day".equals(option))
                    options.setChunkSpan(ChunkTimespan.DAY);
            }
            CmemsImporterClient client = new CmemsImporterClient();
            client.scheduleJob(options);
        }        
    }

    private static void product(CommandLine line) throws Exception {
        if (line.hasOption("h")) {
            HelpFormatter formatter = new HelpFormatter();
            formatter.printHelp("gcube-cmems", productOptions);
        } else {

            CmemsProduct p = null;
            
            String dataset = line.getOptionValue("d");
            String product = line.getOptionValue("p");
            String motuServer = line.getOptionValue("m");

            if(motuServer==null || product==null) {
                SearchOptions so = new SearchOptions();
                so.addMatchedKey(dataset);
                if(product!=null && !product.isEmpty())
                    so.addMatchedKey(product);
                Collection<CmemsProduct> products = new CmemsClient().searchProducts(so);
//                System.out.println("found " + products.size() + " products");
                if(products.size()==1) {
                    p = products.iterator().next();
                    if(motuServer==null)
                        motuServer = p.getMotuServer();
                    if(product==null)
                        product = p.getExternalShortname();
                } else {
                    System.err.println("found zero or more than one product... please refine your search");
                    System.exit(1);
                }
            }
            
            DownloadRequest request = new DownloadRequest();
            request.setService(product+"-TDS");
            request.setProduct(dataset);
            MotuClient mc = new MotuClient(motuServer);
            mc.setUsername(System.getenv().get("CMEMS_USERNAME"));
            mc.setPassword(System.getenv().get("CMEMS_PASSWORD"));
            ProductMetadataInfo pmi = null;
            try {
                pmi = mc.describeProduct(request);
            } catch(Exception e) {
                e.printStackTrace();
                request.setService(product);
                pmi = mc.describeProduct(request);
            }
            System.out.println("--- Product info --------------------------------------------------");
            System.out.print(printProduct(p, 1));
            System.out.println("--- Dataset info --------------------------------------------------");
            System.out.print(printProduct(pmi, 1));

        }
    }
    
    private static void search(CommandLine line) throws Exception {
        if (line.hasOption("h")) {
            HelpFormatter formatter = new HelpFormatter();
            formatter.printHelp("gcube-cmems", searchOptions);
        } else {
            SearchOptions options = new SearchOptions();
            if(line.hasOption("i"))
                options.setIgnoreCase(true);
            if(line.hasOption("k"))
                for(String k:line.getOptionValues("k"))
                    options.addMatchedKey(k);
            if(line.hasOption("K"))
                for(String k:line.getOptionValues("K"))
                    options.addUnMatchedKey(k);
            if(line.hasOption("r"))
                for(String r:line.getOptionValues("r"))
                    options.addRegionalDomain(r);
            if(line.hasOption("v"))
                for(String v:line.getOptionValues("v"))
                    options.addVariable(v);
            if(line.hasOption("f"))
                options.setFrom(TimeUtil.toCalendar(line.getOptionValue("f")));
            if(line.hasOption("t"))
                options.setTo(TimeUtil.toCalendar(line.getOptionValue("t")));
            if(line.hasOption("w"))
                options.setWholeTimeRange(true);
            Collection<CmemsProduct> products = new CmemsClient().searchProducts(options);
            int i=1;
            for(CmemsProduct p:products) {
                System.out.println("--- " + i + " ---------------------------------");
                System.out.print(printProduct(p, 2));
                i++;
            }
        }
    }

    private static String printProduct(ProductMetadataInfo dataset, int verbosity) {
        StringBuffer out = new StringBuffer();
        
        // id
        out.append(format("Id", dataset.getId()));

        // name
        out.append(format("Name", dataset.getTitle()));
        
        // variables
        Collection<String> vars = new Vector<>();
        for(Variable v:dataset.getVariables()) {
            String s = v.getName()+": " + v.getDescription() + ", " + v.getStandardName() + " ("+v.getUnits()+")";
            vars.add(s);
        }
        out.append(format("Variables", vars.toArray(new String[vars.size()]), true));
        
        // time resolution
        out.append(format("Time resolution", dataset.getTimeResolution() + " hours"));
        
        // time coverage
        String tc = String.format("from %s to %s (%s steps)",
                TimeUtil.toString(dataset.getFirstAvailableTimeCode()),
                TimeUtil.toString(dataset.getLastAvailableTimeCode()),
                dataset.getAvailableTimeCodes().size());
        out.append(format("Time coverage", tc));

        // depths
        String dpt = String.format("from %f to %f (%s levels)",
                dataset.getFirstAvailableDepth(),
                dataset.getLastAvailableDepth(),
                dataset.getAvailableDepths().size());
        out.append(format("Depths", dpt));
        
        // axis
        Collection<String> axes = new Vector<>();
        for(Axis a:dataset.getDataGeospatialCoverage()) {
            String s = String.format("%s (%s) - from %s to %s", a.getName(), a.getUnits(), a.getLower(), a.getUpper());
            axes.add(s);
        }
        out.append(format("Axes", axes.toArray(new String[axes.size()]), true));
        
        return out.toString();
        
    }
    
    private static String printProduct(CmemsProduct product, int verbosity) {
        StringBuffer out = new StringBuffer();
        String fmt = "%20s: %s\n";
        out.append(String.format(fmt, "Id", product.getId()));
        out.append(String.format(fmt, "Name", product.getExternalShortname()));
        
        // geographical coverage
        String topLeft = "["+product.getWest()+", "+product.getNorth()+"]";
        String bottomRight = "["+product.getEast()+", "+product.getSouth()+"]";
        out.append(String.format(fmt,  "Geograph. coverage", topLeft + " to " + bottomRight));

        // area
        out.append(String.format(fmt, "Area", product.getGeographicalArea()));

        // observation/models
        out.append(String.format(fmt,  "Observation/models", product.getMissionType()));

        // product type
        out.append(String.format(fmt,  "Product type", String.join(", ", product.getTemporalScale())));

        // processing level
        out.append(String.format(fmt, "Processing lev", product.getProcessingLevel()));

        // data assimilation
        out.append(format("Data assimilation", product.getModelAssimilation()));

        // variables
//        out.append(format("Variables", product.getOceanKeys(), true));

        // spatial resolution
        out.append(String.format(fmt, "Spatial resolution", product.getSpatialResolutionColumn()+" "+product.getSpatialResolutionColumnUnit()+" x "+product.getSpatialResolutionRow()+" "+product.getSpatialResolutionRowUnit()));

        // vertical coverage
        String vc = String.format("from %s to %s (%s levels)", product.getVerticalMin(), product.getVerticalMax(), product.getVerticalLevels());
        out.append(String.format(fmt, "Vertical coverage", vc));

        // coordinate reference system
        out.append(String.format(fmt, "Coord. Ref. Sys", product.getCoordRefSys()));

        // feature type
        out.append(String.format(fmt, "Feature Type", product.getFeatureType()));

        // temporal coverage
        String from = "";
        String to = "";
        try {
            from = TimeUtil.toString(product.getTemporalBegin());
        } catch(java.text.ParseException e) {
            e.printStackTrace();
        }
        try {
            if(product.getTemporalEnd()==null)
                to = "present";
            else
                to = TimeUtil.toString(product.getTemporalEnd());
        } catch(java.text.ParseException e) {
            e.printStackTrace();
        }
        String tc = String.format("from %s to %s", from, to);
        out.append(String.format(fmt, "Temporal coverage", tc));

        // temporal resolution
        out.append(String.format(fmt, "Temporal resolution", product.getTemporalResolution()));

        // update frequency
        out.append(String.format(fmt, "Update frequency", product.getUpdateFrequency()));

        // production unit
        out.append(String.format(fmt, "Production unit", product.getProductionUnit()));

        if(verbosity>1) {
            
            // subsetter server
            out.append(format("Subsetter server", product.getMotuServer()));
    
            // dataset services
//            out.append(format("Datasets (Subsetter)", product.getTdsDatasets(), true));
//            out.append(format("Datasets (WMS)", product.getWmsDatasets(), true));
//            out.append(format("Datasets (FTP)", product.getFtpDatasets(), true));
//            out.append(format("Datasets (DGF)", product.getDgfDatasets(), true));
        }
        
        out.append(String.format(fmt, "Description", product.getShortDescription()));
        out.append(String.format(fmt, "Keywords", product.getAllKeywords()));
        
        // abstract
//        out.append(String.format(fmt, "Abstract", product.getAbstract()));
        return out.toString();
    }
    
    private static String format(String key, String[] values, boolean multiline) {
        String fmt = "%20s: %s\n";
        if(values.length==0) {
            return String.format(fmt, key, "---");
        } else {
            String js = String.format(",\n%22s", "");
            return String.format(fmt, key, String.join(js, values));
        }
    }
    
    private static String format(String key, String value) {
        String fmt = "%20s: %s\n";
        if(value==null || value.trim().isEmpty()) {
            return String.format(fmt, key, "---");
        } else {
            return String.format(fmt, key, value.trim());
        }
    }

    private static void innerMain(String[] args) throws Exception {

        // TODO: setup security
        
        String action = null;
        if(args.length>0) {
            // get the first argument as action
            action = args[0];
            
            // leave all other arguments as actions
            args = Arrays.copyOfRange(args, 1, args.length);
        } else {
            System.err.println("no action specified");
            System.exit(1);
        }
        
        System.out.println("action is " + action);
        
        try {
            // create the parser
            CommandLineParser parser = new DefaultParser();
            switch (action) {
            case "search":
                search(parser.parse(searchOptions, args));
                break;
            case "product":
                product(parser.parse(productOptions, args));
                break;
            case "list-jobs":
                list(parser.parse(listOptions, args));
                break;
            case "schedule-job":
                schedule(parser.parse(scheduleOptions, args));
                break;
            case "unschedule-job":
                unschedule(parser.parse(unscheduleOptions, args));
                break;
            case "job-status":
                break;
            default:
                throw new ParseException("Unrecognized action: " + action);
            }
        } catch (ParseException exp) {
            System.err.println("Parsing failed.  Reason: " + exp.getMessage());
        }
    }

    public static void main(String[] args) throws Exception {
        String ags = "search -k GLOBAL_ANALYSIS_FORECAST_PHY_001_024";
        ags = "search -k salinity -k mediterranean -i";
        ags = "search -r black -i";
        ags = "search -v velocity -i";
        ags = "search -k MEDSEA_REANALYSIS_PHYS_006_004";
        ags = "search -r mediterranean -v temperature --from 2010-01-01 --to 2017-12-31 --whole-time-range --ignore-case";
        ags = "product -m http://cmems-med-mfc.eu/motu-web/Motu -d sv03-med-ingv-cur-rean-d -p MEDSEA_REANALYSIS_PHYS_006_004";
        ags = "product -d sv03-med-ingv-cur-rean-d -p MEDSEA_REANALYSIS_PHYS_006_004";
        ags = "product -d sv03-med-ingv-cur-rean-d";
        ags = "product -d bs-ulg-bio-an-fc-d";
        ags = "product -d bs-ulg-nut-an-fc-d";
        ags = "product -d CORIOLIS-GLOBAL-CORA03.2-OBS_FULL_TIME_SERIE";
        ags = "product -d KNMI-GLO-WIND_L3-REP-OBS_QUIKSCAT_SEAWINDS_25_ASC";
        ags = "product -d KNMI-GLO-WIND_L3-REP-OBS_ERS-1_SCAT_25_DES";
        ags = "unschedule-job -j 293847-2349s7df-asdf-234";
//        innerMain(ags.split(" "));
        innerMain(args);
    }    
    
}
