
/**
 * 
 */
package org.gcube.dataanalysis.copernicus.cmems.importer.seplugin;

import java.io.File;
import java.net.URL;
import java.text.ParseException;
import java.util.Calendar;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Vector;

import org.apache.commons.io.FileUtils;
import org.gcube.dataanalysis.copernicus.cmems.importer.api.ImportOptions;
import org.gcube.dataanalysis.copernicus.cmems.importer.seplugin.thredds.ThreddsClient;
import org.gcube.dataanalysis.copernicus.cmems.importer.seplugin.thredds.ThreddsDataset;
import org.gcube.dataanalysis.copernicus.cmems.importer.seplugin.thredds.ThreddsDatasetChunk;
import org.gcube.dataanalysis.copernicus.cmems.importer.task.ExecutionReport;
import org.gcube.dataanalysis.copernicus.motu.client.DownloadRequest;
import org.gcube.dataanalysis.copernicus.motu.client.MotuClient;
import org.gcube.dataanalysis.datasetimporter.exception.ServiceUnreachableException;
import org.gcube.dataanalysis.datasetimporter.util.ConfigurationUtil;
import org.gcube.dataanalysis.datasetimporter.util.ISClient;
import org.gcube.vremanagement.executor.plugin.Plugin;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.MDC;

/**
 * @author Paolo Fabriani
 *
 */
public class CmemsImporterPlugin extends Plugin<CmemsImporterPluginDeclaration> {

    /**
     * A logger for this class.
     */
    private static Logger logger = LoggerFactory.getLogger(CmemsImporterPlugin.class);

    /**
     * Coordinates for upload to THREDDS
     */
    private static final String THREDDS_PERSISTENCY_ID = "thredds";
    private static final String THREDDS_CATALOG_PATH = "public/netcdf/CMEMS";
    
    /**
     * Where downloaded chunks are placed in the local file system.
     */
    private static final String DOWNLOAD_DIR = "/tmp";
    
    /**
     * Where execution logs are placed.
     */
    private static final String LOG_DIR = "/tmp/cmems/logs";
    
    /**
     * An helper class to keep task/execution/reports updated
     */
    private ExecutionTracker taskTracker;

    /**
     * The name of the report, as published on the task/report service
     */
    private static final String PUBLISHED_REPORT_NAME = "import.log";

    
    /**
     * @param pluginDeclaration
     * @throws Exception
     */

    public CmemsImporterPlugin(CmemsImporterPluginDeclaration pluginDeclaration) throws Exception {
        super(pluginDeclaration);
    }

    @Override
    public void launch(Map<String, Object> inputs) throws Exception {
        
        String taskId = this.getUUID().toString();
        String executionId = Calendar.getInstance().getTimeInMillis()+"";

        // initialize the logging context
        MDC.put("executionId", taskId+"-"+executionId);
        
        // where execution logs are placed
        File logFile = new File(LOG_DIR+"/"+this.getUUID().toString()+"-"+executionId+".log");
        
        // prepare the ExecutionReport
        ExecutionReport report = new ExecutionReport();
        report.setName(PUBLISHED_REPORT_NAME);

        // build an helper to track this execution
        this.taskTracker = new ExecutionTracker(taskId, executionId);

        // process parameters and create importOptions
        ImportOptions options = this.getImportOptionsFromLaunchParameters(inputs);

        // initialize the task
        this.taskTracker.checkTaskExists();
//        this.taskTracker.taskInit(options);
        
        // record the task execution
        this.taskTracker.executionInit();

        // the total number of steps (1) is fake, just to change the status from
        // 'init' to 'running' with 0% progress.
        this.taskTracker.executionProgress(0, 1);

        // 1. build a THREDDS client
        ThreddsClient thredds = this.buildThreddsClient();

        // 2. get the current status on THREDDS
        ThreddsDataset publishedDataset = null;
        if(thredds.containsDataset(options)) {
            publishedDataset = thredds.getDataset(options);
        } else {
            publishedDataset = new ThreddsDataset();
            publishedDataset.setOptions(options);
            logger.info("No data yet published for this imported dataset");
        }
                
        // compute the desired state
        ThreddsDataset expectedDataset = new ThreddsDataset(options);
        
        // compute the work to be done (diff)
        Collection<ThreddsDatasetChunk> missingChunks = this.getMissingChunks(publishedDataset, expectedDataset);

        int totalSteps = 2 + missingChunks.size()*3;
        int stepsCompleted = 2;
        
        if(missingChunks.size()==0) {
            // nothing to do
            logger.info("The dataset on THREDDS is up-to-date. Nothing to do.");

        } else {

            // 3. build a motu client
            MotuClient motuClient = this.buildMotuClient(options.getMotu());
            
            // 4. Download and publish each missing chunk. Update the ncml
            for(ThreddsDatasetChunk chunk:missingChunks) {
                
                // build a request for the chunk
                DownloadRequest request = getRequestForMotu(options, chunk.getChunkStart(), chunk.getChunkEnd());

                // 4.1 download the chunk
                File f = motuClient.downloadProduct(request, new File(DOWNLOAD_DIR+"/"+chunk.getFileName()));
                logger.debug("downloaded " + f.getAbsolutePath());
                this.taskTracker.executionProgress(stepsCompleted++, totalSteps);
    
                // 4.2 upload it to thredds
                logger.info("uploading nc file to thredds");
                thredds.upload(f);
                this.taskTracker.executionProgress(stepsCompleted++, totalSteps);
                
                // remove .nc file locally
                logger.debug("removing local file " + f.getAbsolutePath());
                f.delete();
                
                // and add it to the published dataset
                publishedDataset.addChunk(chunk);
                
                // generate new ncml file and store it to the file system
                String ncml = publishedDataset.generateNCML();
                File ncmlFile = new File(DOWNLOAD_DIR + "/" + options.getHash()+".ncml");
                FileUtils.writeStringToFile(ncmlFile, ncml, "UTF-8");
    
                // 4.3 publish the ncml to thredds
                logger.info("uploading ncml file to thredds");
                thredds.upload(ncmlFile);
                this.taskTracker.executionProgress(stepsCompleted++, totalSteps);
                
                // remove nmcmlFile locally
                logger.debug("removing " + ncmlFile.getAbsolutePath());
                ncmlFile.delete();
                
                // publishing logs
                logger.info("Uploading reports.");
                report.setText(FileUtils.readFileToString(logFile, "UTF-8"));
                this.taskTracker.executionReport(report);

            }
            
        }

        // publishing logs
        logger.info("Uploading reports.");
        report.setText(FileUtils.readFileToString(logFile, "UTF-8"));
        this.taskTracker.executionReport(report);

        // remove logs locally
        logger.info("Remove logs locally");
        logFile.delete();

        // mark execution complete
        this.taskTracker.executionComplete();
        
    }
    
    @Override
    protected void onStop() throws Exception {
        
        logger.debug("onStop()");
        Thread.currentThread().interrupt();
        
    }

    /**
     * Compute the set of chunks to be processed (downloaded from motu and uploaded to thredds).
     * @param published
     * @param expected
     * @return
     */
    private Collection<ThreddsDatasetChunk> getMissingChunks(ThreddsDataset published, ThreddsDataset expected) {
        Collection<ThreddsDatasetChunk> missingChunks = new Vector<>();
        if(published==null || published.size()==0) {
            logger.debug("Adding all chunks");
            missingChunks = expected.getChunks();
        } else {
            for(ThreddsDatasetChunk c:expected.getChunks()) {
                if(!published.contains(c) || c.needsUpdate()) {
                    logger.debug("Adding chunk " + c.getFileName());
                    missingChunks.add(c);
                }
            }
        }
        return missingChunks;                
     }

    /**
     * Given a chunk, build the request to be submitted to Motu for download.
     * @param chunk
     * @return
     */
    private DownloadRequest getRequestForMotu(ImportOptions options, Calendar from, Calendar to) {
        DownloadRequest request = new DownloadRequest();
        request.setService(options.getProduct());
        request.setProduct(options.getDataset());
        request.setxRange(options.getxLo(), options.getxHi());
        request.setyRange(options.getyLo(), options.getyHi());
        request.setzRange(options.getzLo(), options.getzHi());
        request.settRange(from, to);
        request.setScriptVersion("1.4.00-20170410143941999");
        request.setMode("status");
        request.setOutput("netcdf");
        if(options.getVariables()!=null) {
            for(String v:options.getVariables()) {
                request.addVariable(v);
            }
        }
        return request;
    }
    
    /**
     * Build a THREDDS client.
     * @return
     * @throws ServiceUnreachableException
     */
    private ThreddsClient buildThreddsClient() throws ServiceUnreachableException {
        URL endpoint = new ISClient().getThreddsEndpoint();
        ThreddsClient client = new ThreddsClient(endpoint, THREDDS_PERSISTENCY_ID, THREDDS_CATALOG_PATH);
        client.checkThreddsIsReachable();
        return client;
    }

    /**
     * Build a Motu client
     * @return
     * @throws Exception
     */
    private MotuClient buildMotuClient(String motu) throws Exception {
        // build a configuration manager
        ConfigurationUtil cu = new ConfigurationUtil();
        String userHome = System.getenv("HOME");
        cu.setLocalConfigurationFile(userHome+"/.cmems");
        // build a client
        MotuClient mc = new MotuClient(motu);
        mc.setUsername(cu.getProperty("CMEMS_USERNAME"));
        mc.setPassword(cu.getProperty("CMEMS_PASSWORD"));
        return mc;
    }

    private ImportOptions getImportOptionsFromLaunchParameters(Map<String, Object> inputs) throws ParseException {
        // convert the <String,Object> map to <String, String>
        Map<String, String> map = new HashMap<String, String>();
        for(Entry<String, Object> entry: inputs.entrySet()) {
            String key = entry.getKey();
            String value = entry.getValue().toString();
            map.put(key, value);
        }
        // build ImportOptions from the map
        return new ImportOptions(map);
    }

}
