/**
 * 
 */
package org.gcube.dataanalysis.copernicus.cmems.importer.service.service;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.StringReader;
import java.io.StringWriter;
import java.nio.file.FileAlreadyExistsException;
import java.text.ParseException;
import java.util.SortedSet;
import java.util.TreeSet;

import javax.xml.bind.JAXBContext;
import javax.xml.bind.JAXBException;
import javax.xml.bind.Marshaller;
import javax.xml.bind.Unmarshaller;

import org.apache.commons.io.FileUtils;
import org.gcube.dataanalysis.copernicus.cmems.importer.api.ImportOptions;
import org.gcube.dataanalysis.copernicus.cmems.importer.service.exception.NotThereException;
import org.gcube.dataanalysis.copernicus.cmems.importer.task.Execution;
import org.gcube.dataanalysis.copernicus.cmems.importer.task.ExecutionReport;
import org.gcube.dataanalysis.copernicus.cmems.importer.task.ImportTask;
import org.gcube.dataanalysis.copernicus.cmems.importer.task.SubmissionInfo;

/**
 * @author Paolo Fabriani
 *
 */
public class TaskStoreHelper {

    private String root;
    
    public static final String IMPORT_PARAMETERS_FILE_NAME = "import-parameters.xml";
    public static final String SUBMISSION_INFO_FILE_NAME = "submission-info.xml";
    public static final String EXECUTIONS_DIR_NAME = "executions";
    public static final String EXECUTION_INFO_FILE_NAME = "execution-info.xml";
    public static final String REPORTS_DIR_NAME = "reports";
    
    public TaskStoreHelper(String root) {
        this.root = root;
    }
    
    /**
     * Return the root of the task store.
     * @return
     */
    public File getRoot() {
        return new File(this.root);
    }
    
    public String[] getAllTasks() {
        return this.getRoot().list();
    }
    
    /**
     * Return the root of the task.
     * @param taskId
     * @return
     */
    public File getTaskRoot(String taskId) {
        return new File(this.getRoot(), taskId);
    }

    /**
     * Return a file containing import parameters.
     * @param taskId
     * @return
     */
    private File getTaskParameters(String taskId) {
        return new File(this.getTaskRoot(taskId), IMPORT_PARAMETERS_FILE_NAME);
    }
    
    /**
     * Return the file containing the submission context.
     * @param taskId
     * @return
     */
    private File getTaskContext(String taskId) {
        return new File(this.getTaskRoot(taskId), SUBMISSION_INFO_FILE_NAME);
    }
    
    /**
     * Return the root of task executions
     * @param taskId
     * @return
     */
    public File getExecutionsRoot(String taskId) {
        return new File(this.getTaskRoot(taskId), EXECUTIONS_DIR_NAME);
    }

    /**
     * Return all ids of the executions for a task.
     * @param taskId
     * @return
     */
    public String[] getAllExecutions(String taskId) {
        return this.getExecutionsRoot(taskId).list();
    }

    /**
     * Return the last execution of the task. Since the id of the execution is
     * its start timestamp, it's implemented by returning the greatest execution
     * id.
     */
    public String getLastExecution(String taskId) {
        String[] allExecutions = this.getAllExecutions(taskId);
        if(allExecutions==null || allExecutions.length==0)
            return null;
        SortedSet<Long> ids = new TreeSet<>();
        for(String s:allExecutions) {
            try {
                ids.add(Long.parseLong(s));
            } catch(NumberFormatException e) {
                e.printStackTrace();
            }
        }
        if(ids.isEmpty()) {
            return null;
        } else {
            return ids.last().toString();
        }
    }
    
    /**
     * Return the root of a given execution
     * @param taskId
     * @return
     */
    public File getExecutionRoot(String taskId, String executionId) {
        return new File(this.getExecutionsRoot(taskId), executionId);
    }

    /**
     * Return the file containing execution info
     * @param taskId
     * @return
     */
    public File getExecutionInfoFile(String taskId, String executionId) {
        return new File(this.getExecutionRoot(taskId, executionId), EXECUTION_INFO_FILE_NAME);
    }

    /**
     * Return the root execution reports
     * @param taskId
     * @return
     */
    public File getExecutionReportsRoot(String taskId, String executionId) {
        return new File(this.getExecutionRoot(taskId, executionId), REPORTS_DIR_NAME);
    }

    /**
     * Return all reports for a given execution.
     * @param taskId
     * @return
     */
    public String[] getAllExecutionReports(String taskId, String executionId) {
        return this.getExecutionReportsRoot(taskId, executionId).list();
    }

    /**
     * Return the root execution reports
     * @param taskId
     * @return
     */
    public File getExecutionReport(String taskId, String executionId, String reportName) {
        return new File(this.getExecutionReportsRoot(taskId, executionId), reportName);
    }

    /**
     * Deserialize the task with the given id. Executions are ignored here.
     * @param taskId
     * @return
     * @throws IOException
     */
    public ImportTask deserializeTask(String taskId) throws JAXBException, NotThereException, IOException, ParseException {
        ImportTask t = new ImportTask();
        t.setId(taskId);
        t.setImportParameters(this.deserializeOptions(taskId));
        t.setSubmissionInfo(this.deserializeContext(taskId));
        return t;
    }
    
    /**
     * Deserialize task import parameters.
     * @param taskId
     * @return
     * @throws IOException
     * @throws Exception
     */
    public ImportOptions deserializeOptions(String taskId) throws JAXBException, NotThereException, IOException, ParseException {
        
        // be sure the task exist
        this.ensureTask(taskId);
        
        // be sure the options file exists
        File optionsFile = ensureExist(this.getTaskParameters(taskId));
        
        // parse it
        String xml = FileUtils.readFileToString(optionsFile, "UTF-8");
        return (ImportOptions)fromXml(xml, ImportOptions.class);
        
    }

    /**
     * Deserialize task submission context.
     * @param taskId
     * @return
     * @throws Exception
     */
    public SubmissionInfo deserializeContext(String taskId) throws NotThereException, IOException, ParseException, JAXBException {
        
        // be sure the task exist
        this.ensureTask(taskId);

        // be sure the context exists
        File contextFile = ensureExist(this.getTaskContext(taskId));

        // parse it
        String xml = FileUtils.readFileToString(contextFile, "UTF-8");
        return (SubmissionInfo)fromXml(xml, SubmissionInfo.class);

    }

    /**
     * Deserialize an execution
     * @param taskId
     * @param executionId
     * @return
     * @throws Exception
     */
    public Execution deserializeExecution(String taskId, String executionId) throws NotThereException, FileNotFoundException, ParseException, IOException, JAXBException {
        
        File execFile = this.ensureExecutionInfo(taskId, executionId);
        
        // parse xml
        String xml = FileUtils.readFileToString(execFile, "UTF-8");
        return (Execution)fromXml(xml, Execution.class);
    }

    /**
     * Serialize a task to the given file.
     * @param taskDir
     * @param task
     * @throws IOException
     */
    public static void serializeTask(File taskDir, ImportTask task) throws IOException, JAXBException {
        taskDir.mkdirs();
        if(task.getSubmissionInfo()!=null) {
            serializeContext(new File(taskDir, SUBMISSION_INFO_FILE_NAME), task.getSubmissionInfo());
        }
        if(task.getImportParameters()!=null) {
            serializeImportParameters(new File(taskDir, IMPORT_PARAMETERS_FILE_NAME), task.getImportParameters());
        }
    }

    /**
     * Serialize a submission context to the given file.
     * @param file
     * @param submissionInfo
     * @throws IOException
     */
    public static void serializeContext(File file, SubmissionInfo submissionInfo) throws IOException, JAXBException {
        FileUtils.writeStringToFile(file, toXml(submissionInfo), "UTF-8");
    }

    /**
     * Serialize import parameters to the given file.
     * @param file
     * @param options
     * @throws IOException
     */
    public static void serializeImportParameters(File file, ImportOptions options) throws IOException, JAXBException {
        FileUtils.writeStringToFile(file, toXml(options), "UTF-8");
    }
    



    /**
     * Serialize execution info to the given file.
     * @param file
     * @param execution
     * @throws IOException
     */
    public static void serializeExecution(File file, Execution execution) throws IOException, JAXBException {
        FileUtils.writeStringToFile(file, toXml(execution), "UTF-8");
    }

    /**
     * Serialize the report to the given file
     * @param file
     * @param report
     * @throws Exception
     */
    public static void serializeReport(File file, ExecutionReport report) throws IOException {
        FileUtils.write(file, report.getText(), "UTF-8");
    }
    
    public static String getContent(File f) throws IOException {
        return FileUtils.readFileToString(f, "UTF-8");
    }

    public static void writeContent(File f, String content) throws IOException {
        FileUtils.write(f, content, "UTF-8");
    }

    public File ensureTask(String taskId) throws FileNotFoundException {
        return ensureExist(this.getTaskRoot(taskId));
    }

    public File ensureTaskMiss(String taskId) throws FileAlreadyExistsException {
        return ensureDontExist(this.getTaskRoot(taskId));
    }

    public File ensureExecutionInfo(String taskId, String executionId) throws FileNotFoundException {
        this.ensureTask(taskId);
        return ensureExist(this.getExecutionInfoFile(taskId, executionId));
    }

    public File ensureExecutionInfoMiss(String taskId, String executionId) throws FileNotFoundException, FileAlreadyExistsException {
        this.ensureTask(taskId);
        return ensureDontExist(this.getExecutionInfoFile(taskId, executionId));
    }

    public File ensureReport(String taskId, String executionId, String reportName) throws FileNotFoundException {
        this.ensureExecutionInfo(taskId, executionId);
        return ensureExist(this.getExecutionReport(taskId, executionId, reportName));
    }

    public File ensureReportMiss(String taskId, String executionId, String reportName) throws FileNotFoundException, FileAlreadyExistsException {
        this.ensureExecutionInfo(taskId, executionId);
        return ensureDontExist(this.getExecutionReport(taskId, executionId, reportName));
    }

    public static File ensureExist(File f) throws FileNotFoundException {
        if(!f.exists()) {
            throw new FileNotFoundException("no such file: " + f.getAbsolutePath());
        }
        return f;
    }

    public static File ensureDontExist(File f) throws FileAlreadyExistsException {
        if(f.exists()) {
            throw new FileAlreadyExistsException("file " + f.getAbsolutePath() + " already exists");
        }
        return f;
    }

    private static String toXml(Object options) throws JAXBException {
        JAXBContext jaxbContext = JAXBContext.newInstance(options.getClass());
        Marshaller jaxbMarshaller = jaxbContext.createMarshaller();
        jaxbMarshaller.setProperty(Marshaller.JAXB_FORMATTED_OUTPUT, true);
        StringWriter writer = new StringWriter();
        jaxbMarshaller.marshal(options, writer);
        return writer.toString();
    }
    
    private static Object fromXml(String xml, Class clazz) throws JAXBException {
        JAXBContext jaxbContext = JAXBContext.newInstance(clazz);
        Unmarshaller unmarshaller = jaxbContext.createUnmarshaller();
        StringReader reader = new StringReader(xml);
        return unmarshaller.unmarshal(reader);
    }
 
}
