/**
 * 
 */
package org.gcube.dataanalysis.copernicus.cmems;

import java.util.Calendar;
import java.util.Collection;
import java.util.TreeSet;
import java.util.Vector;

import org.apache.log4j.Logger;
import org.gcube.dataanalysis.copernicus.motu.model.MotuServer;
import org.gcube.dataanalysis.copernicus.motu.util.NetworkUtils;
import org.gcube.dataanalysis.copernicus.motu.util.TimeUtil;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;

/**
 * @author Paolo Fabriani
 *
 */
public class CmemsClient {

    /**
     * A Logger for this class
     */
    private static Logger LOGGER = Logger.getLogger(CmemsClient.class);

    /**
     * The cached list of products
     */
    private static Collection<CmemsProduct> cachedProducts = new Vector<>();

    /**
     * When the cache was last updted
     */
    private static Long cacheTimestamp;

    /**
     * For how long we'll keep the cache (in minutes)
     */
    private static final Long CACHE_TTL = 1L;

    /**
     * The endpoint of the service
     */
    private String endpoint;

    public CmemsClient() {
        this("http://cmems-resources.cls.fr/index.php?option=com_csw&task=advancedsearch");
    }

    public CmemsClient(String endpoint) {
        this.endpoint = endpoint;
    }

    /**
     * Download the set of CMEMS products.
     * @return a collection of all CMEMS products.
     * @throws Exception
     */
    private Collection<CmemsProduct> downloadProducts() throws Exception {
        String json = NetworkUtils.doGet(this.endpoint, null);
        ObjectMapper objectMapper = new ObjectMapper();
        JsonNode rootNode = objectMapper.readTree(json);
        JsonNode docsNode = rootNode.path("products").path("response")
                .path("docs");
        Collection<CmemsProduct> out = new Vector();
        for (CmemsProduct p : objectMapper.readValue(docsNode.toString(),
                CmemsProduct[].class)) {
            out.add(p);
        }
        return out;
    }

    /**
     * Retrieve the set of all CMEMS products, possibly taking them from the
     * cache.
     * @return
     * @throws Exception
     */
    public Collection<CmemsProduct> getAllProducts() throws Exception {
        synchronized (cachedProducts) {
            if (this.isCacheExpired()) {
                LOGGER.debug("getting a fresh copy of products");
                Collection<CmemsProduct> prods = this.downloadProducts();
                cachedProducts = prods;
                cacheTimestamp = Calendar.getInstance().getTimeInMillis();
            }
            return cachedProducts;
        }
    }

    /**
     * Check whether the cache has expired.
     * @return true if expired, false otherwise.
     */
    private Boolean isCacheExpired() {
        if (cacheTimestamp == null) {
            return true;
        } else if ((Calendar.getInstance().getTimeInMillis()
                - cacheTimestamp) > CACHE_TTL * 60 * 1000) {
            return true;
        } else {
            return false;
        }
    }

    /**
     * Return the list of Motu servers publishing CMEMS datasets.
     * @return
     * @throws Exception
     */
    public Collection<MotuServer> getMotuServers() throws Exception {
        Collection<String> urls = new TreeSet<>();
        for (CmemsProduct psi : this.getAllProducts()) {
            String tds = psi.getTds();
            if (tds != null && !tds.equalsIgnoreCase("undefined")) {
                String url = tds.substring(0, tds.indexOf("?"));
                urls.add(url);
            }
        }
        Collection<MotuServer> out = new Vector<>();
        for (String url : urls) {
            out.add(new MotuServer(url));
        }
        return out;
    }

    /**
     * Return the list of regional domains
     * @return
     * @throws Exception
     */
    public Collection<CodedString> getRegionalDomains() throws Exception {
        Collection<CodedString> out = new TreeSet<>();
        for (CmemsProduct product : this.getAllProducts()) {
            String[] parameters = product.getGeographicalArea().split(",");
            for (String par : parameters) {
                if (par != null && !par.trim().isEmpty()) {
                    out.add(new CodedString(par.trim()));
                }
            }
        }
        out.remove(new CodedString("undefined"));
        return out;
    }

    public Collection<CodedString> getOceanVariables() throws Exception {
        Collection<CodedString> out = new TreeSet<>();
        for (CmemsProduct product : this.getAllProducts()) {
            String[] parameters = product.getOceanVariables().split(",");
            for (String par : parameters) {
                if (par != null && !par.trim().isEmpty()) {
                    out.add(new CodedString(par.trim()));
                }
            }
        }
        out.remove(new CodedString("undefined"));
        return out;
    }

    public Collection<CodedString> getOceanKeys() throws Exception {
        Collection<CodedString> out = new TreeSet<>();
        for (CmemsProduct product : this.getAllProducts()) {
            String[] parameters = product.getOceanKeys().split(",");
            for (String par : parameters) {
                if (par != null && !par.trim().isEmpty()) {
                    out.add(new CodedString(par.trim()));
                }
            }
        }
        out.remove(new CodedString("undefined"));
        return out;
    }
    
    public Calendar getMinTime() throws Exception {
        Calendar out = Calendar.getInstance();
        for (CmemsProduct product : this.getAllProducts()) {
            Calendar c = TimeUtil.toCalendar(product.getTemporalBegin());
            if (c.before(out)) {
                out = c;
            }
        }
        return out;
    }

    public Calendar getMaxTime() throws Exception {
        Calendar out = Calendar.getInstance();
        out.setTimeInMillis(0);
        for (CmemsProduct product : this.getAllProducts()) {
            if(product.getTemporalEnd()!=null) {
                Calendar c = TimeUtil.toCalendar(product.getTemporalEnd());
                if (c.after(out)) {
                    out = c;
                }
            }
        }
        return out;
    }

}
