package org.gcube.dataanalysis.copernicus.motu.util;

import java.io.BufferedReader;
import java.io.DataOutputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.io.UnsupportedEncodingException;
import java.net.CookieHandler;
import java.net.CookieManager;
import java.net.CookiePolicy;
import java.net.HttpURLConnection;
import java.net.URL;
import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;
import java.util.Collection;
import java.util.Iterator;
import java.util.Map;

import org.apache.http.HttpStatus;
import org.apache.log4j.Logger;
import org.gcube.dataanalysis.copernicus.motu.client.CASClient;
import org.gcube.dataanalysis.copernicus.motu.client.MultiValueParameters;

public class NetworkUtils {

    /**
     * A logger for this class.
     */
    private static Logger logger = Logger.getLogger(NetworkUtils.class);

    static {
        CookieHandler
                .setDefault(new CookieManager(null, CookiePolicy.ACCEPT_ALL));
    }

    /**
     * Encode parameters for the query string.
     *
     * @param params parameters to be encoded
     * @return the encoded query string
     * @throws UnsupportedEncodingException
     */
    public static String getDataString(MultiValueParameters params)
            throws UnsupportedEncodingException {
        StringBuilder result = new StringBuilder();
        boolean first = true;
        if (params == null) {
            return result.toString();
        }
        for (String key : params.keySet()) {
            Collection<String> value = params.get(key);
            Iterator<String> iter = value.iterator();
            while (iter.hasNext()) {
                if (first) {
                    first = false;
                } else {
                    result.append("&");
                }
                result.append(URLEncoder.encode(key, "UTF-8"));
                result.append("=");
                result.append(URLEncoder.encode(iter.next(), "UTF-8"));
            }
        }
        return result.toString();
    }

    /**
     * Do an HTTP GET.
     *
     * @param url
     * @param casProxy
     * @return the response
     * @throws Exception
     */
    public static String doGet(String url, CASClient casProxy)
            throws Exception {

        logger.debug("GET-ing: " + url);

        // do the request
        URL u = new URL(url);
        HttpURLConnection con = (HttpURLConnection) u.openConnection();
        con.setRequestMethod("GET");

        // handle redirect, in particular to the login page
        int responseCode = con.getResponseCode();
        if (responseCode == HttpStatus.SC_MOVED_TEMPORARILY) {
            String location = con.getHeaderField("Location");
            logger.debug("I've been redirected to " + location);
            if (casProxy.isAuthnEndpoint(location)) {
                logger.debug("setting the CAS endpoint");
                casProxy.setEndpoint(location);
                String redirectUrl = casProxy.authenticateForURL(url);
                logger.info("redirecting to " + redirectUrl);
                return doGet(redirectUrl, casProxy);
            } else {
                return doGet(location, casProxy);
            }
        }

        // process the response
        BufferedReader in = new BufferedReader(
                new InputStreamReader(con.getInputStream()));
        StringBuffer response = new StringBuffer();
        String inputLine;
        while ((inputLine = in.readLine()) != null) {
            response.append(inputLine);
        }
        in.close();
        return response.toString();
    }

    /**
     * Do an HTTP POST
     *
     * @param url
     * @param parameters
     * @return
     * @throws Exception
     */
    public static String post(String url, MultiValueParameters parameters)
            throws Exception {

        logger.debug("POST-ing data to " + url);

        // prepare payload
        byte[] postData = getDataString(parameters)
                .getBytes(StandardCharsets.UTF_8);

        // do connect
        URL u = new URL(url);
        HttpURLConnection conn = (HttpURLConnection) u.openConnection();
        conn.setDoOutput(true);
        conn.setInstanceFollowRedirects(false);
        conn.setRequestMethod("POST");
        conn.setRequestProperty("Content-Type",
                "application/x-www-form-urlencoded");
        conn.setRequestProperty("charset", "utf-8");
        conn.setRequestProperty("Content-Length",
                new Integer(postData.length).toString());
        conn.setUseCaches(false);

        // do write data
        try (DataOutputStream wr = new DataOutputStream(
                conn.getOutputStream())) {
            wr.write(postData);
        }

        // process response
        Reader in = new BufferedReader(
                new InputStreamReader(conn.getInputStream(), "UTF-8"));
        StringBuilder sb = new StringBuilder();
        for (int c; (c = in.read()) >= 0;) {
            sb.append((char) c);
        }
        return sb.toString();
    }

}
