package org.gcube.dataanalysis.copernicus.motu.client;

import java.text.ParseException;
import java.util.Calendar;
import java.util.Collection;
import java.util.Vector;

import org.gcube.dataanalysis.copernicus.motu.model.ProductMetadataInfo;
import org.gcube.dataanalysis.copernicus.motu.model.ServiceMetadata;
import org.gcube.dataanalysis.copernicus.motu.util.TimeUtil;

/**
 * This class models parameters to be passed to a 'downloadProduct' or 'getSize'
 * request.
 *
 * @author Paolo Fabriani
 *
 */
public class DownloadRequest {

    /**
     * The service provided by the server.
     */
    private String service;

    /**
     * The product (i.e. the dataset) to get.
     */
    private String product;

    /**
     * The action to perform on the dataset (e.g. productDownload, getSize)
     */
    private String action;

    /**
     * The physical variables to be extracted from the product. When no variable
     * is set, all the variables of the dataset are extracted.
     */
    private Collection<String> variables;

    /**
     * Low longitude of a geographic extraction. Default value is -180
     */
    private Double xLo;

    /**
     * High longitude of a geographic extraction. Default value is 180
     */
    private Double xHi;

    /**
     * Low latitude of a geographic extraction. Default value is -90.
     */
    private Double yLo;

    /**
     * High latitude of a geographic extraction. Default value is -90.
     */
    private Double yHi;

    /**
     * Low vertical depth . Default value is 0.
     */
    private Double zLo;

    /**
     * High vertical depth. Default value is 180
     */
    private Double zHi;

    /**
     * Start date of a temporal extraction. If not set, the default value is the
     * first date/time available for the dataset.
     */
    private Calendar tLo;

    /**
     * End date of a temporal extraction. If not set, the default value is the
     * last date/time available for the dataset.
     */
    private Calendar tHi;

    /**
     * ???
     */
    private String scriptVersion;

    /**
     * How the request is performed. Currently, only 'status' mode is supported
     */
    private String mode;

    /**
     * The output format.
     */
    private String output;

    /**
     * Create an empty request.
     */
    public DownloadRequest() {
    }

    /**
     * Create a request, actually cloning the 'other' request.
     *
     * @param other
     *            the source DownloadRequest
     */
    public DownloadRequest(final DownloadRequest other) {
        this();
        this.setService(other.getService());
        this.setProduct(other.getProduct());
        this.setAction(other.getAction());
        this.setVariables(other.getVariables());
        this.setxRange(other.getxLo(), other.getxHi());
        this.setyRange(other.getyLo(), other.getyHi());
        this.setzRange(other.getzLo(), other.getzHi());
        this.settRange(other.gettLo(), other.gettHi());
        this.setScriptVersion(other.getScriptVersion());
        this.setMode(other.getMode());
        this.setOutput(other.getOutput());
    }

    public String getService() {
        return service;
    }

    public void setService(String service) {
        this.service = service;
    }
    
    public void setService(ServiceMetadata service) {
        this.service = service.getName();
    }

    public String getProduct() {
        return product;
    }

    public void setProduct(String product) {
        this.product = product;
    }
    
    public void setProduct(ProductMetadataInfo product) {
        this.product = product.getTitle();
    }

    public Double getxLo() {
        return xLo;
    }

    public void setxLo(Double xLo) {
        this.xLo = xLo;
    }

    public Double getxHi() {
        return xHi;
    }

    public void setxHi(Double xHi) {
        this.xHi = xHi;
    }

    public void setxRange(Double xLo, Double xHi) {
        this.setxLo(xLo);
        this.setxHi(xHi);
    }

    public Double getyLo() {
        return yLo;
    }

    public void setyLo(Double yLo) {
        this.yLo = yLo;
    }

    public Double getyHi() {
        return yHi;
    }

    public void setyHi(Double yHi) {
        this.yHi = yHi;
    }

    public void setyRange(Double yLo, Double yHi) {
        this.setyLo(yLo);
        this.setyHi(yHi);
    }

    public Double getzLo() {
        return zLo;
    }

    public void setzLo(Double zLo) {
        this.zLo = zLo;
    }

    public Double getzHi() {
        return zHi;
    }

    public void setzHi(Double zHi) {
        this.zHi = zHi;
    }

    public void setzRange(Double zLo, Double zHi) {
        this.setzLo(zLo);
        this.setzHi(zHi);
    }

    public Calendar gettLo() {
        return tLo;
    }

    public void settLo(Calendar tLo) {
        this.tLo = tLo;
    }

    public void settLo(String tLo) throws ParseException {
        this.settLo(TimeUtil.toCalendar(tLo));
    }

    public Calendar gettHi() {
        return tHi;
    }

    public void settHi(Calendar tHi) {
        this.tHi = tHi;
    }

    public void settHi(String tHi) throws ParseException {
        this.settHi(TimeUtil.toCalendar(tHi));
    }

    public void settRange(Calendar tLo, Calendar tHi) {
        this.settLo(tLo);
        this.settHi(tHi);
    }

    public void settRange(String tLo, String tHi) throws ParseException {
        this.settLo(tLo);
        this.settHi(tHi);
    }

    public String getScriptVersion() {
        return scriptVersion;
    }

    public void setScriptVersion(String scriptVersion) {
        this.scriptVersion = scriptVersion;
    }

    public String getMode() {
        return mode;
    }

    public void setMode(String mode) {
        this.mode = mode;
    }

    public String getOutput() {
        return output;
    }

    public void setOutput(String output) {
        this.output = output;
    }

    public String getAction() {
        return action;
    }

    public void setAction(String action) {
        this.action = action;
    }

    public Collection<String> getVariables() {
        if (this.variables == null) {
            this.variables = new Vector<>();
        }
        return variables;
    }
    
    public Boolean hasVariables() {
        return this.variables!=null && this.variables.size()>0;
    }
    
    public void setVariables(Collection<String> variables) {
        this.variables = variables;
    }

    public void addVariable(String variable) {
        if (this.variables == null) {
            this.variables = new Vector<>();
        }
        this.variables.add(variable);
    }

    public MultiValueParameters getParametersMap() {
        MultiValueParameters output = new MultiValueParameters();
        if (this.service != null) {
            output.put("service", this.service);
        }
        if (this.product != null) {
            output.put("product", this.product);
        }
        if (this.action != null) {
            output.put("action", this.action);
        }
        if (this.xLo != null) {
            output.put("x_lo", this.xLo.toString());
        }
        if (this.xHi != null) {
            output.put("x_hi", this.xHi.toString());
        }
        if (this.yLo != null) {
            output.put("y_lo", this.yLo.toString());
        }
        if (this.yHi != null) {
            output.put("y_hi", this.yHi.toString());
        }
        if (this.zLo != null) {
            output.put("z_lo", this.zLo.toString());
        }
        if (this.zHi != null) {
            output.put("z_hi", this.zHi.toString());
        }
        if (this.tLo != null) {
            output.put("t_lo", TimeUtil.toString(this.tLo));
        }
        if (this.tHi != null) {
            output.put("t_hi", TimeUtil.toString(this.tHi));
        }
        if (this.scriptVersion != null) {
            output.put("scriptVersion", this.scriptVersion);
        }
        if (this.mode != null) {
            output.put("mode", this.mode);
        }
        if (this.output != null) {
            output.put("output", this.output);
        }
        if (this.variables != null) {
            for (String variable : this.variables) {
                output.put("variable", variable);
            }
        }
        return output;
    }

}
