package org.gcube.dataanalysis.ensemble;

import java.io.File;
import java.io.IOException;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.UUID;

import org.apache.commons.io.FileUtils;
import org.gcube.contentmanagement.lexicalmatcher.utils.AnalysisLogger;
import org.gcube.dataanalysis.ecoengine.datatypes.PrimitiveType;
import org.gcube.dataanalysis.ecoengine.datatypes.StatisticalType;
import org.gcube.dataanalysis.ecoengine.datatypes.enumtypes.PrimitiveTypes;
import org.gcube.dataanalysis.ecoengine.interfaces.StandardLocalExternalAlgorithm;
import org.gcube.dataanalysis.ensemble.util.Workspace;

/**
 *
 * @author Paolo Fabriani (Engineering Ingegneria Informatica S.p.A.)
 *
 */
public class EnsembleAlgorithm extends StandardLocalExternalAlgorithm {

  // location of the algorithm package
  private static final String ENSEMBLE_MODEL_SOURCE_PKG = "/tmp/ensemble/script/MEE_NORTHSEA_260914.py";

  // the script to execute locally
  private static final String ENSEMBLE_MODEL_SCRIPT_NAME = "MEE_NORTHSEA_260914.py";
  
  // the local position of input files (relative to execution root)
  private static final String INPUT_LOCATION = ".";

  // the local position of output files (relative to execution root)
  private static final String OUTPUT_LOCATION = ".";

  // the local position of executables files (relative to execution root)
  private static final String BINARIES_LOCATION = ".";

  // input files 'internal' names
  public static final String IN_DAPSTOM_FILE = "?DAPSTOM File?";
  public static final String IN_NS_DIET_FILE = "?NS_DIET File?";
  public static final String IN_FISHING_PARAM_HIST_FILE = "?FISHING_PARAM_HIST_FILE File?";
  public static final String IN_FISHING_PARAM_MSY_FILE = "?FISHING_PARAM_MSY_FILE File?";
  public static final String IN_NS_IV_FILE = "?NS_IV_FILE File?";
  public static final String IN_NS_PARAMS_FILE = "?NS_PARAMS_FILE File?";

  // input files 'internal' names
  private static final String IN_DAPSTOM_FILE_DESCRIPTION = "?description for DAPSTOM File?";
  private static final String IN_NS_DIET_FILE_DESCRIPTION = "?description for NS_DIET File?";
  private static final String IN_FISHING_PARAM_HIST_FILE_DESCRIPTION = "?description for FISHING_PARAM_HIST_FILE File?";
  private static final String IN_FISHING_PARAM_MSY_FILE_DESCRIPTION = "?description for FISHING_PARAM_MSY_FILE File?";
  private static final String IN_NS_IV_FILE_DESCRIPTION = "?description for NS_IV_FILE File?";
  private static final String IN_NS_PARAMS_FILE_DESCRIPTION = "?description for NS_PARAMS_FILE File?";

  //  names of input files as hardcoded in the script
  private static final String DAPSTOM_FILE_HARDCODED_NAME = "DAPSTOM.txt";
  private static final String NS_DIET_FILE_HARDCODED_NAME = "NS_diet.csv.txt";
  private static final String FISHING_PARAM_HIST_FILE_HARDCODED_NAME = "Fishing_Params_300114_HIST.txt";
  private static final String FISHING_PARAM_MSY_FILE_HARDCODED_NAME = "Fishing_Params_300114_MSY.txt";
  private static final String NS_IV_FILE_HARDCODED_NAME = "NS_IV_combo1.dat";
  private static final String NS_PARAMS_FILE_HARDCODED_NAME = "NS_Params_270114.dat";

  // internal names of output files
  private static final String OUT_BIOMASS = "?Biomass?";
  private static final String OUT_SSB = "?SSB?";
  private static final String OUT_XTABLE = "?XTABLE?";
  private static final String OUT_EXECUTION_LOG = "?Execution log?";

  // name of output files as produced by the script
  private static final String OUT_BIOMASS_HARDCODED_NAME = "BIOMASS_0.txt";
  private static final String OUT_SSB_HARDCODED_NAME = "SSB_0.txt";
  private static final String OUT_XTABLE_HARDCODED_NAME = "XTABLE_0.txt";

  // where the execution log (stdout and stderr) is redirected
  private static final String EXECUTION_LOG = "execution.log";

  private Workspace workspace;

  private String executionId;
  
  @Override
  public void init() throws Exception {
    AnalysisLogger.getLogger().info("Ensemble Abstract Initialisation");
    AnalysisLogger.getLogger().info(this.getDescription());
    
    this.executionId = UUID.randomUUID().toString();
    
    this.workspace = new Workspace(this.getExecutionId());
    this.workspace.setExecutionsRoot(config.getConfigPath());
    this.workspace.setBinariesLocation(BINARIES_LOCATION);
    this.workspace.setInputLocation(INPUT_LOCATION);
    this.workspace.setOutputLocation(OUTPUT_LOCATION);
  }

  @Override
  public String getDescription() {
    String description = "?TODO?";
    return description;
  }

  @Override
  protected void setInputParameters() {
    // ask for diet file #1
    this.inputs.add(new PrimitiveType(File.class.getName(), null,
        PrimitiveTypes.FILE, IN_DAPSTOM_FILE, IN_DAPSTOM_FILE_DESCRIPTION));

    // ask for diet file #2
    this.inputs
        .add(new PrimitiveType(File.class.getName(), null, PrimitiveTypes.FILE,
            IN_NS_DIET_FILE, IN_NS_DIET_FILE_DESCRIPTION));

    // ask for fishing params (historic)
    this.inputs.add(new PrimitiveType(File.class.getName(), null,
        PrimitiveTypes.FILE, IN_FISHING_PARAM_HIST_FILE,
        IN_FISHING_PARAM_HIST_FILE_DESCRIPTION));

    // ask for fishing params (msy)
    this.inputs.add(new PrimitiveType(File.class.getName(), null,
        PrimitiveTypes.FILE, IN_FISHING_PARAM_MSY_FILE,
        IN_FISHING_PARAM_MSY_FILE_DESCRIPTION));

    // ask for ??? initial values
    this.inputs.add(new PrimitiveType(File.class.getName(), null,
        PrimitiveTypes.FILE, IN_NS_IV_FILE, IN_NS_IV_FILE_DESCRIPTION));

    // ask for params file
    this.inputs.add(new PrimitiveType(File.class.getName(), null,
        PrimitiveTypes.FILE, IN_NS_PARAMS_FILE,
        IN_NS_PARAMS_FILE_DESCRIPTION));
  }

  @Override
  protected void process() throws Exception {
    // make sure the working directory is ready and contains input files
    this.setupWorkingDirectory();

    // make input files available in the working directory
    this.prepareInput();

    // make sure Ensemble Model algorithm is available in the working directory
    this.downloadAndSetupEnsemble();

    // execute the algorithm
    String command = String.format("python %s", ENSEMBLE_MODEL_SCRIPT_NAME);
    this.workspace.exec(command, null, EXECUTION_LOG);

    // prepare the output
    this.prepareOutput();

    // cleanup
    this.workspace.destroy();
  }

  @Override
  public StatisticalType getOutput() {

    LinkedHashMap<String, StatisticalType> map = new LinkedHashMap<String, StatisticalType>();

    PrimitiveType logOutputFile = new PrimitiveType(File.class.getName(),
        new File(config.getPersistencePath(), EXECUTION_LOG),
        PrimitiveTypes.FILE, EXECUTION_LOG, OUT_EXECUTION_LOG);
    map.put(OUT_EXECUTION_LOG, logOutputFile);

    PrimitiveType biomassOutputFile = new PrimitiveType(File.class.getName(),
        new File(config.getPersistencePath(), OUT_BIOMASS_HARDCODED_NAME),
        PrimitiveTypes.FILE, OUT_BIOMASS_HARDCODED_NAME, OUT_BIOMASS);
    map.put(OUT_BIOMASS, biomassOutputFile);

    PrimitiveType ssbOutputFile = new PrimitiveType(File.class.getName(),
        new File(config.getPersistencePath(), OUT_SSB_HARDCODED_NAME),
        PrimitiveTypes.FILE, OUT_SSB_HARDCODED_NAME, OUT_SSB);
    map.put(OUT_BIOMASS, ssbOutputFile);

    PrimitiveType xtableOutputFile = new PrimitiveType(File.class.getName(),
        new File(config.getPersistencePath(), OUT_XTABLE_HARDCODED_NAME),
        PrimitiveTypes.FILE, OUT_XTABLE_HARDCODED_NAME, OUT_XTABLE);
    map.put(OUT_XTABLE, xtableOutputFile);

    PrimitiveType output = new PrimitiveType(LinkedHashMap.class.getName(), map,
        PrimitiveTypes.MAP, "ResultsMap", "Results Map");
    return output;
  }

  @Override
  public void shutdown() {
    AnalysisLogger.getLogger().info("Shutdown");
  }

  private String getExecutionId() {
    return executionId;
  }

  /**
   * The goal of this method is to make input files available in the working
   * directory.
   */
  protected void setupWorkingDirectory() throws Exception {
    AnalysisLogger.getLogger().debug(
        "setting up working directory '" + this.workspace.getRoot() + "'");
    workspace.ensureStructureExists();
  }

  private void prepareInput() throws IOException {
    AnalysisLogger.getLogger().debug("Copying input files...");
    FileUtils.copyFile(new File(config.getParam(IN_DAPSTOM_FILE)),
        new File(workspace.getInputLocation(), DAPSTOM_FILE_HARDCODED_NAME));
    FileUtils.copyFile(new File(config.getParam(IN_NS_DIET_FILE)),
        new File(workspace.getInputLocation(), NS_DIET_FILE_HARDCODED_NAME));
    FileUtils.copyFile(new File(config.getParam(IN_FISHING_PARAM_HIST_FILE)),
        new File(workspace.getInputLocation(), FISHING_PARAM_HIST_FILE_HARDCODED_NAME));
    FileUtils.copyFile(new File(config.getParam(IN_FISHING_PARAM_MSY_FILE)),
        new File(workspace.getInputLocation(), FISHING_PARAM_MSY_FILE_HARDCODED_NAME));
    FileUtils.copyFile(new File(config.getParam(IN_NS_IV_FILE)),
        new File(workspace.getInputLocation(), NS_IV_FILE_HARDCODED_NAME));
    FileUtils.copyFile(new File(config.getParam(IN_NS_PARAMS_FILE)),
        new File(workspace.getInputLocation(), NS_PARAMS_FILE_HARDCODED_NAME));
    AnalysisLogger.getLogger().debug("Input files copied to input location.");
  }

  /**
   * The goal of this method is to make input files available in the working.
   * directory.
   */
  protected void downloadAndSetupEnsemble() throws Exception {
    AnalysisLogger.getLogger().debug("Downloading Ensemble package");
    FileUtils.copyFileToDirectory(new File(ENSEMBLE_MODEL_SOURCE_PKG), workspace.getRoot());
  }

  /**
   * Move output files to the persistence location.
   */
  protected void prepareOutput() throws IOException {
    FileUtils.copyFileToDirectory(new File(workspace.getOutputLocation(), EXECUTION_LOG),
        new File(config.getPersistencePath()));
    FileUtils.copyFileToDirectory(new File(workspace.getOutputLocation(), OUT_BIOMASS_HARDCODED_NAME),
        new File(config.getPersistencePath()));
    FileUtils.copyFileToDirectory(new File(workspace.getOutputLocation(), OUT_SSB_HARDCODED_NAME),
        new File(config.getPersistencePath()));
    FileUtils.copyFileToDirectory(new File(workspace.getOutputLocation(), OUT_XTABLE_HARDCODED_NAME),
        new File(config.getPersistencePath()));
  }

  /**
   * The goal of this method is to make input files available in the working
   * directory.
   */
  protected void removeWorkingDirectory() throws Exception {
    AnalysisLogger.getLogger().debug("Removing working directory");
  }

}