package org.gcube.dbinterface.persistence;

import java.lang.reflect.Constructor;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.TreeMap;
import org.gcube.common.dbinterface.ColumnDefinition;
import org.gcube.common.dbinterface.Condition;
import org.gcube.common.dbinterface.TableAlreadyExistsException;
import org.gcube.common.dbinterface.attributes.SimpleAttribute;
import org.gcube.common.dbinterface.conditions.OperatorCondition;
import org.gcube.common.dbinterface.preparedstatement.PreparedInsert;
import org.gcube.common.dbinterface.queries.Cast;
import org.gcube.common.dbinterface.queries.GetMetadata;
import org.gcube.common.dbinterface.queries.Select;
import org.gcube.common.dbinterface.queries.update.CreateTable;
import org.gcube.common.dbinterface.queries.update.Delete;
import org.gcube.common.dbinterface.queries.update.Update;
import org.gcube.common.dbinterface.queries.update.batch.BatchUpdater;
import org.gcube.common.dbinterface.registry.Connection;
import org.gcube.common.dbinterface.registry.DBInterface;
import org.gcube.common.dbinterface.registry.Shortcuts;
import org.gcube.common.dbinterface.tables.Table;
import org.gcube.common.dbinterface.tables.TableInfo;
import org.gcube.common.dbinterface.types.Type;
import org.gcube.common.dbinterface.types.Type.InnerType;
import org.gcube.dbinterface.persistence.annotations.AnnotationNotDefinedException;
import org.gcube.dbinterface.persistence.annotations.FieldDefinition;
import org.gcube.dbinterface.persistence.annotations.TableRootDefinition;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.thoughtworks.xstream.XStream;
import static org.gcube.common.dbinterface.utils.Conditions.*;
import static org.gcube.common.dbinterface.utils.Attributes.*;

/**
 * 
 * @author Lucio Lelii
 *
 * @param <T>
 */
public class ObjectPersistency<T> {

	@SuppressWarnings("rawtypes")
	public static HashMap<String, ObjectPersistency> persistencyMapping = new HashMap<String, ObjectPersistency>();
	
	private static final Logger logger = LoggerFactory.getLogger(ObjectPersistency.class);
	
	private DBInterface dbInterface;
	
	private Shortcuts sc ;
	
	private GetMetadata metadata;
	
	private static final String FIELD_PREFIX="ifield";
	//private static final String IDENTIFIERS_REF_FIELD_PREFIX="ref";
	private Class<T> _clazz;
	private Table table;
	
	private List<PersistencyCallback<T>> callbacks = new ArrayList<PersistencyCallback<T>>();
	
	public void addCallback(PersistencyCallback<T> obj){
		this.callbacks.add(obj);
	}
	
	
	@SuppressWarnings("unchecked")
	public static <T> ObjectPersistency<T> get(Class<T> clazz, DBInterface dbInterface) throws Exception{
		//logger.debug("requesting table for class "+clazz.getName());
		if (!persistencyMapping.containsKey(clazz.getName()))
				persistencyMapping.put(clazz.getName(), new ObjectPersistency<T>(clazz, dbInterface));
		return persistencyMapping.get(clazz.getName());
	}
	
	private ObjectPersistency(Class<T> clazz, DBInterface dbInterface) throws Exception {
		this.dbInterface = dbInterface;
		sc = dbInterface.queries();
		
		this._clazz=clazz;
        if (!_clazz.isAnnotationPresent(TableRootDefinition.class)) throw new AnnotationNotDefinedException(); 
		String tableName = clazz.getSimpleName()+Math.abs(clazz.getName().hashCode());
		try{
			this.table =createTable(tableName);
		}catch (TableAlreadyExistsException e) {
			this.table= new Table(tableName);
			/*checkTableCompatibility(table);
			table.initializeFieldMapping();*/
		}
		metadata = dbInterface.getMetadata(table);
		
	}
	
	@SuppressWarnings("rawtypes")
	private FieldMappingPair retrieveColumnDefinition(Class _clazz, int fieldIndex) throws Exception{
		List<ColumnDefinition> cdList= new ArrayList<ColumnDefinition>();
		TreeMap<String, String> internalFieldMapping=new TreeMap<String, String>();
		
		logger.trace("retrieving column definition for --> table "+_clazz.getSimpleName());
		
		for (Field field:_clazz.getDeclaredFields()){
			if(field.isAnnotationPresent(FieldDefinition.class)){
				FieldDefinition fieldDefinition= field.getAnnotation(FieldDefinition.class);
				String fieldNameInTable= FIELD_PREFIX+fieldIndex;
							
				Type type = Type.getTypeByJavaClass(field.getType());
				if (type==null || (type.getInnerType()==InnerType.STRING && fieldDefinition.precision().length==0))
					type = new Type(InnerType.TEXT);
				else type.setPrecision(fieldDefinition.precision());
				logger.trace(field.getName()+ " --> the type "+field.getType()+" is converted in  "+type.getInnerType().name());

				ColumnDefinition cDef= sc.column(fieldNameInTable, type, fieldDefinition.specifications());
				cdList.add(cDef);
				internalFieldMapping.put(field.getName(), fieldNameInTable);
				fieldIndex++;
			}
		}
		if (_clazz.getSuperclass()!=null){
			if (_clazz.getSuperclass().equals(ObjectStateControl.class)){
				
				Type type= new Type(InnerType.INTEGER);
				type.setPrecision(10);
				ColumnDefinition cDef= sc.column("objectversion", type);
				//cDef.setSpecification(Specification.NOT_NULL);
				cdList.add(cDef);
				//internalFieldMapping.put("objectversion", "objectversion");
			} else {
				FieldMappingPair fmPair = retrieveColumnDefinition(_clazz.getSuperclass(), fieldIndex);
				cdList.addAll(fmPair.getColumnsDefinition());
				internalFieldMapping.putAll(fmPair.getFieldMapping());
			}
		}
		return new FieldMappingPair(internalFieldMapping, cdList);
	}
	
	private Table createTable(String tableName) throws Exception {
		Connection conn = dbInterface.getConnection();
		
		
		
		FieldMappingPair fmPair = retrieveColumnDefinition(this._clazz, 0);
		CreateTable create = dbInterface.builders().createTable().name(tableName)
				.columns(fmPair.getColumnsDefinition().toArray(new ColumnDefinition[fmPair.getColumnsDefinition().size()])).build();
		
		try{
			conn.executeUpdate(create);
			SystemTableInfo.getSystemInfo(dbInterface).addInfo(fmPair.getFieldMapping(), tableName);
		}catch (SQLException e) {
			logger.error("error executing update",e);
		}finally{
			dbInterface.release(conn);
		}
		return new Table(tableName)	;	
	}
	
	public void deleteByKey(Object key) throws Exception{
		Map<String, Type> mapping = getMapping();
		String primaryKey = null;
		Type type= null;
		for (Entry<String, Type> entry: mapping.entrySet())
			if (entry.getValue().isPrimaryKey()){
				primaryKey = entry.getKey();
				type = entry.getValue(); 
				break;
			}
		
		if (callbacks.size()>0){
			T obj= getByKey(key);
			for (PersistencyCallback<T> callback :callbacks)
					callback.onObjectDeleted(obj);
		}
		
		Cast cast = dbInterface.builders().cast().value(key.toString()).type(type).build();
		
		Delete delete= dbInterface.builders().delete().table(this.table.getTableName()).where(eq(attribute(primaryKey), cast)).build();
		
		Connection conn = dbInterface.getConnection();
		int res =0;
		try{
			res = conn.executeUpdate(delete);
		}catch (SQLException e) {
			logger.error("error executing update",e);
		}finally{
			dbInterface.release(conn);
		}
		if (res==0) throw new ObjectNotFoundException();
	}
	
	public Map<String, String> getInfo() throws Exception{
		return SystemTableInfo.getSystemInfo(dbInterface).retrieveInfo(this.table.getTableName());
	}
	
	public T getByKey(Object key) throws ObjectNotFoundException, Exception{
		Map<String, Type> mapping = getMapping();
		String primaryKey = null;
		Type type= null;
		for (Entry<String, Type> entry: mapping.entrySet())
			if (entry.getValue().isPrimaryKey()){
				primaryKey = entry.getKey();
				type = entry.getValue(); 
				break;
			}
		if(type==null) throw new Exception("no primary key found in "+this.table.getTableName());
		
		Cast cast = dbInterface.builders().cast().value(key.toString()).type(type).build();
		
		
		Select select= dbInterface.builders().select().from(this.table).where(eq(attribute(primaryKey), cast)).build();
		
		T toReturn = null;
		Connection connection = dbInterface.getConnection();
		try{
			ResultSet result =connection.executeQuery(select).asDefault();
			if (result.next()) toReturn = createObject(result);
			else throw new ObjectNotFoundException();
		}finally{
			dbInterface.release(connection);
		}
		
		return toReturn;
	}
	
	public void insert(T object) throws Exception{
		for (PersistencyCallback<T> callback: this.callbacks)
			callback.onBeforeStore(object);
		
		List<Object> oInsert = retrieveInsertField(object, _clazz);
		
		PreparedInsert preparedInsert = dbInterface.builders().preparedInsert().table(this.table.getTableName()).elements(oInsert.size()).build();
		
		BatchUpdater updater = dbInterface.builders().batchInsert().stopOnError().use(preparedInsert)
				.elements(Collections.singletonList(oInsert.toArray(new Object[oInsert.size()]))).build();
		
		logger.trace("inserting "+oInsert.toString());
		
		Connection conn = dbInterface.getConnection();
		try{
			conn.executeBatchUpdate(updater);
			logger.trace("insert executed");
		}catch (Exception e) {
			dbInterface.release(conn);
		}
			
		for (PersistencyCallback<T> callback: this.callbacks)
			callback.onObjectStored(object);
		
	}
	
	@SuppressWarnings({ "rawtypes", "unchecked" })
	private List<Object> retrieveInsertField(T object, Class clazz) throws Exception{
		List<Object> oInsert= new ArrayList<Object>();

		logger.trace("storing class "+clazz.getSimpleName());
		
		for (Field field:clazz.getDeclaredFields())
			if(field.isAnnotationPresent(FieldDefinition.class)){
				if (Type.getTypeByJavaClass(field.getType())==null){
					logger.trace(field.getName()+" - "+field.getType());
					field.setAccessible(true);
					oInsert.add(new XStream().toXML(field.get(object)));
				}else {
					logger.trace(field.getName()+" - "+field.getType());
					field.setAccessible(true);
					oInsert.add(field.get(object));
				}
			}
		
		if (clazz.getSuperclass()!=null){
			if (clazz.getSuperclass().equals(ObjectStateControl.class))
				oInsert.add(0);
			else if (clazz.getSuperclass().isAnnotationPresent(TableRootDefinition.class)){
				oInsert.addAll(retrieveInsertField(object, clazz.getSuperclass()));
			}
		}
		return oInsert;
	}
	
	public void drop() throws Exception{
		Connection conn = dbInterface.getConnection();
		try{
			conn.executeUpdate(sc.remove(this.table));
			SystemTableInfo.getSystemInfo(dbInterface).deleteInfo(this.table.getTableName());
		}finally{
			dbInterface.release(conn);
		}
	}
	
	public void deleteByValue(String fieldName,Object value) throws Exception{
				
		String internalFieldName= SystemTableInfo.getSystemInfo(dbInterface).retrieveInfo(this.table.getTableName()).get(fieldName);
		if (internalFieldName== null) throw new Exception("field "+fieldName+" non retrieved");
		
		if (callbacks.size()>0){
			Iterator<T> it= getObjectByField(fieldName, value).iterator();
			while (it.hasNext())
				for (PersistencyCallback<T> callback :callbacks)
					callback.onObjectDeleted(it.next());
		}
		
		Delete delete= dbInterface.builders().delete().table(this.table.getTable()).where(eq(attribute(internalFieldName), value )).build();
		Connection conn = dbInterface.getConnection();
		try{
			conn.executeUpdate(delete);
		}finally{
			dbInterface.release(conn);
		}
		
		
	}
	
	public List<T> getObjectByField(String fieldName, Object value) throws Exception{
		
		String internalFieldName= SystemTableInfo.getSystemInfo(dbInterface).retrieveInfo(this.table.getTableName()).get(fieldName);
		if (internalFieldName== null) throw new Exception("field "+fieldName+" non retrieved");
		
		Select select= dbInterface.builders().select().from(this.table).where(eq(attribute(internalFieldName), value)).build();
		
		Connection conn= dbInterface.getConnection();
		try{
			return toList(conn.executeQuery(select).asDefault());
		}finally{
			dbInterface.release(conn);
		}
	}
	
	/*public Iterator<T> getObjectByMultipleFieldValues(String fieldName, Object[] values) throws Exception{
		Select select= dbInterface.getImplementation(Select.class);
		String internalFieldName= SystemTableInfo.getSystemInfo().retrieveInfo(this.table.getTableName()).get(fieldName);
		if (internalFieldName== null) throw new Exception("field "+fieldName+" non retrieved");
		ArrayList<OperatorCondition<SimpleAttribute, Object>> conditions = new ArrayList<OperatorCondition<SimpleAttribute,Object>>();  
		for (Object value: values)
			conditions.add(new OperatorCondition<SimpleAttribute, Object>(new SimpleAttribute(internalFieldName),value,"="));
		select.setUseDistinct(true);
		select.setFilter(new ORCondition(conditions.toArray(new OperatorCondition[conditions.size()])));
		select.setTables(this.table);
		DBConnection session= dbInterface.getSession();
		return new InternalIterator(select.getResults(session), session);
	}*/
	
	public List<T> getObjectByFields(HashMap<String, Object> fieldValueMapping) throws Exception{
		if (fieldValueMapping.isEmpty()) throw new Exception("the filedValueMapping is empty");
		
		ArrayList<OperatorCondition<SimpleAttribute, Object>> conditions = new ArrayList<OperatorCondition<SimpleAttribute,Object>>();  
		for (Entry<String, Object> entry: fieldValueMapping.entrySet()){
			String internalFieldName= SystemTableInfo.getSystemInfo(dbInterface).retrieveInfo(this.table.getTableName()).get(entry.getKey());
			if (internalFieldName== null) throw new Exception("field "+entry.getKey()+" non retrieved");
			conditions.add(new OperatorCondition<SimpleAttribute, Object>(new SimpleAttribute(internalFieldName),entry.getValue(),"="));
		}
		Select select = dbInterface.builders().select().from(this.table).where(and(conditions.toArray(new OperatorCondition[conditions.size()]))).build();
		
		Connection session= dbInterface.getConnection();
		try{
			return toList(session.executeQuery(select).asDefault());
		}finally{
			dbInterface.release(session);
		}
	}
	
	public List<T> getAll() throws Exception{
		Select select = dbInterface.builders().select().from(this.table).build();
		Connection session= dbInterface.getConnection();
		try{
			logger.trace("executing query {}",select.getQuery());
			return toList(session.executeQuery(select).asDefault());
		}catch (Exception e) {
			logger.error("error getting elements",e);
			throw e;
		}finally{
			dbInterface.release(session);
		}
	}
		
	public Table getTable() {
		return table;
	}

	public boolean existsKey(Object key) throws Exception{
		Connection session= dbInterface.getConnection();
		try{
			Map<String, Type> mapping=new TableInfo(table.getTableName()).getMapping();
			String primaryKey = null;
			Type type= null;
			for (Entry<String, Type> entry: mapping.entrySet())
				if (entry.getValue().isPrimaryKey()){
					primaryKey = entry.getKey();
					type = entry.getValue(); 
					break;
				}
			if(type==null) throw new ObjectNotFoundException("no primary key found in "+this.table.getTableName());
			Cast cast = dbInterface.builders().cast().value(key.toString()).type(type).build();
			
			Select select= dbInterface.builders().select().from(this.table).where(eq(attribute(primaryKey), cast)).build();
			
			ResultSet result = session.executeQuery(select).asDefault();
			if (result.next()) return true;
			else return false;
		}catch (Exception e) {
			logger.error("errror retrieving key",e);
			throw e;
		}finally{
			if (session !=null) dbInterface.release(session);
		}
	}
		
	public boolean existEntryByFields(HashMap<String, Object> fieldValueMapping) throws Exception{
		if (fieldValueMapping.isEmpty()) throw new Exception("the filedValueMapping is empty");
		
		ArrayList<OperatorCondition<SimpleAttribute, Object>> conditions = new ArrayList<OperatorCondition<SimpleAttribute,Object>>();  
		for (Entry<String, Object> entry: fieldValueMapping.entrySet()){
			String internalFieldName= SystemTableInfo.getSystemInfo(dbInterface).retrieveInfo(this.table.getTableName()).get(entry.getKey());
			if (internalFieldName== null) throw new Exception("field "+entry.getKey()+" non retrieved");
			conditions.add(new OperatorCondition<SimpleAttribute, Object>(new SimpleAttribute(internalFieldName),entry.getValue(),"="));
		}
		
		Select select= dbInterface.builders().select().from(this.table).where(and(conditions.toArray(new OperatorCondition[conditions.size()]))).build();
		
		Connection session= dbInterface.getConnection();
		try{
			ResultSet rs = session.executeQuery(select).asDefault();
			return rs.next();
		}finally{
			dbInterface.release(session);
		}
		
	}
	
	/**
	 * 
	 * @param obj
	 * @throws ObjectStateChangedException
	 * @throws ObjectNotFoundException
	 * @throws Exception
	 */
	public void update(T obj) throws ObjectStateChangedException,ObjectNotFoundException, Exception{
		Map<String, Type> mapping = getMapping();
		String primaryKey = null;
		Type type= null;
		for (Entry<String, Type> entry: mapping.entrySet())
			if (entry.getValue().isPrimaryKey()){
				primaryKey = entry.getKey();
				type = entry.getValue(); 
				break;
			}
		if(type==null) throw new ObjectNotFoundException("no primary key found in "+this.table.getTableName());
		
		Object keyValue = getFieldValue(obj, SystemTableInfo.getSystemInfo(dbInterface).retrieveFieldName(this.table.getTableName(), primaryKey) , type);
		Cast cast= dbInterface.builders().cast().value(keyValue!=null?keyValue.toString():null).type(type).build();
			
		executeUpdate(obj, eq(attribute(primaryKey), cast));
	}	

	
	
	/**
	 * 
	 * @param obj
	 * @param fieldValueMapping
	 * @throws ObjectStateChangedException
	 * @throws ObjectNotFoundException
	 * @throws Exception
	 */
	public void updateByFields(T obj, HashMap<String, Object> fieldValueMapping) throws ObjectStateChangedException,ObjectNotFoundException, Exception{
		if (fieldValueMapping.isEmpty()) throw new Exception("the filedValueMapping is empty");
		ArrayList<OperatorCondition<SimpleAttribute, Object>> conditions = new ArrayList<OperatorCondition<SimpleAttribute,Object>>();  
		//Update update= dbInterface.getImplementation(Update.class);
		//update.setTable(table);
		for (Entry<String, Object> entry: fieldValueMapping.entrySet()){
			String internalFieldName= SystemTableInfo.getSystemInfo(dbInterface).retrieveInfo(this.table.getTableName()).get(entry.getKey());
			if (internalFieldName== null) throw new Exception("field "+entry.getKey()+" non retrieved");
			conditions.add(new OperatorCondition<SimpleAttribute, Object>(new SimpleAttribute(internalFieldName),entry.getValue(),"="));
		}
		executeUpdate(obj, and(conditions.toArray(new OperatorCondition[conditions.size()])));
	}
	
	private Object getFieldValue(T obj, String fieldName, Type internalType) throws Exception{
		Field field = retrieveField(fieldName, _clazz);
		field.setAccessible(true);
		Object fieldValue= null;
		if (internalType.getInnerType().getJavaClass().isPrimitive()){
			Method fieldMethod= Field.class.getDeclaredMethod(internalType.getInnerType().getReflectionMethodGet(), Object.class);
			fieldValue = fieldMethod.invoke(field, obj);
		}else if (field.getType().isEnum()){
			fieldValue = field.get(obj).toString();
		}
		else if(Type.getTypeByJavaClass(field.getType())==null) fieldValue = new XStream().toXML(field.get(obj));
		else fieldValue = field.get(obj);
		return fieldValue;
	}
	
	@SuppressWarnings("rawtypes")
	private void executeUpdate(T obj, Condition filter) throws ObjectStateChangedException,ObjectNotFoundException,Exception{
		Map<String, Type> mapping = getMapping();
		List<OperatorCondition> setters= new ArrayList<OperatorCondition>();
		for (Entry<String, String> entry:SystemTableInfo.getSystemInfo(dbInterface).retrieveInfo(this.table.getTableName()).entrySet()){
			Type internaltype=mapping.get(entry.getValue());
			if (!internaltype.isPrimaryKey()){
				Object fieldValue = getFieldValue(obj, entry.getKey(), internaltype);
				Cast cast= dbInterface.builders().cast().value(fieldValue!=null?fieldValue.toString():null).type(internaltype).build();
							
				setters.add(eq(attribute(entry.getValue()), cast));
			}
		}
		
		Connection session= dbInterface.getConnection();
		session.disableAutoCommit();
		
		//checking if the object is already stored, and adding control for version consistency
		try{
			Select select = dbInterface.builders().select().from(this.table).where(filter).build();
			
			if (!session.executeQuery(select).asDefault().next()){
				dbInterface.release(session);
				throw new ObjectNotFoundException();
			}
			
			if (isUnderVersionControl()){
				Field field=ObjectStateControl.class.getDeclaredField("objectversion");
				field.setAccessible(true);
				int versionValue = field.getInt(obj);
				Condition versionControl = new OperatorCondition<SimpleAttribute, Integer>(new SimpleAttribute("objectversion"), versionValue,"=");
				filter = and(filter, versionControl);
				setters.add(new OperatorCondition<SimpleAttribute, Integer>(new SimpleAttribute("objectversion"), versionValue+1,"="));
				field.setInt(obj, versionValue+1);
			}
			
			//executing the update
			Update update= dbInterface.builders().update().table(table).set(setters.toArray(new OperatorCondition[setters.size()])).where(filter).build();
			
			int result = session.executeUpdate(update);
			session.commit();
			
			if (result==0) throw new ObjectStateChangedException();
		}catch (Exception e) {
			throw e;
		}finally{	
			dbInterface.release(session);
		}
		if (obj!=null){
			for (PersistencyCallback<T> callback: this.callbacks)
				callback.onObjectUpdated(obj);
		}
	}
	
	@SuppressWarnings({ "rawtypes", "unchecked" })
	private Field retrieveField(String fieldName, Class clazz) {
		Field field = null;
		try{	
			field = clazz.getDeclaredField(fieldName);
		}catch (Exception e) {
			if (clazz.getSuperclass()!=null && clazz.getSuperclass().isAnnotationPresent(TableRootDefinition.class))
				return retrieveField(fieldName, clazz.getSuperclass());
		}
		return field;
		
			
		
	}
	
	@SuppressWarnings("rawtypes")
	private boolean isUnderVersionControl(){
		Class clazz = _clazz;
		while ((clazz=clazz.getSuperclass())!=null)
			if (clazz.equals(ObjectStateControl.class)) return true;
		return false;
	}
	
	
	private List<T> toList(ResultSet resultSet) throws Exception{
		ArrayList<T> returnList= new ArrayList<T>();
		while (resultSet.next())
			returnList.add(createObject(resultSet));
		resultSet.close();
		return returnList;
	}
	
	private Map<String, Type> getMapping() throws Exception{
		Connection conn = dbInterface.getConnection();
		try{
			return metadata.getMetadata(conn);
		}finally{
			dbInterface.release(conn);
		}
	}
		
	
	private T createObject(ResultSet result) throws Exception{
		Constructor<T> constr= _clazz.getDeclaredConstructor();
		logger.debug("createObjectInternal with class "+_clazz.getName());
		constr.setAccessible(true);
		T returnObj=constr.newInstance();
		
		Map<String, Type> mapping = getMapping();
		
		logger.trace("mapping is empty?"+(mapping.size()==0));
					
		Map<String, String> classTableMap= SystemTableInfo.getSystemInfo(dbInterface).retrieveInfo(this.table.getTableName());
				
		logger.trace("classTableMap is empty?"+(classTableMap.size()==0));
				
		logger.trace("class table map is null?"+(classTableMap==null)+" and mapping is null? "+(mapping==null));
		createObjectInternal(_clazz, mapping, returnObj, classTableMap, result);
				
		
		if (returnObj!=null){
			for (PersistencyCallback<T> callback: this.callbacks)
				callback.onObjectLoaded(returnObj);
		}
		return returnObj;
	}
	
	@SuppressWarnings({ "unchecked", "rawtypes" })
	private void createObjectInternal(Class clazz, Map<String, Type> mapping, T returnObj, Map<String, String> classTableMap,  ResultSet result) throws Exception{
		for (Field field: clazz.getDeclaredFields()){
			String internalFieldName= classTableMap.get(field.getName());
			logger.trace("retrieved field "+internalFieldName);
			if (internalFieldName!=null){
				field.setAccessible(true);
				logger.trace("fieldName is "+field.getName());
				logger.trace("mapping.get(internalFieldName) is null?"+(mapping.get(internalFieldName)==null));
				if (mapping.get(internalFieldName).getInnerType().getJavaClass().isPrimitive()){
					Method rsMethod= ResultSet.class.getDeclaredMethod(mapping.get(internalFieldName).getInnerType().getReflectionMethodGet(), String.class);
					Method fieldMethod= Field.class.getDeclaredMethod(mapping.get(internalFieldName).getInnerType().getReflectionMethodSet(), Object.class, mapping.get(internalFieldName).getInnerType().getJavaClass());
					fieldMethod.invoke(field, returnObj, rsMethod.invoke(result, internalFieldName));
				}else if (field.getType().isEnum()){
					String value = result.getString(internalFieldName);
					field.set(returnObj, Enum.valueOf(((Class<? extends Enum>)field.getType()), value));
				} else if (Type.getTypeByJavaClass(field.getType())==null)//not identified object
					field.set(returnObj,new XStream().fromXML(result.getString(internalFieldName)));
				else if (Type.getTypeByJavaClass(field.getType()).getInnerType()==InnerType.STRING)
					field.set(returnObj,result.getString(internalFieldName));
				else{
					logger.trace("method is "+mapping.get(internalFieldName).getInnerType().getReflectionMethodGet());
					Method rsMethod= ResultSet.class.getDeclaredMethod(mapping.get(internalFieldName).getInnerType().getReflectionMethodGet(), String.class);
					field.set(returnObj, rsMethod.invoke(result, internalFieldName));
				}
			}
		}
		if (clazz.getSuperclass()!=null){
			if (clazz.getSuperclass()!=null){
				if (clazz.getSuperclass().equals(ObjectStateControl.class)){
					Field superClassField=clazz.getSuperclass().getDeclaredField("objectversion");
					superClassField.setAccessible(true);
					superClassField.setInt(returnObj, result.getInt("objectversion"));
				}
				else
					createObjectInternal(clazz.getSuperclass(), mapping, returnObj, classTableMap, result);
			}
		}
	}
	
	
	private  class FieldMappingPair{
				
		private TreeMap<String, String> fieldMapping;
		private List<ColumnDefinition> columnsDefinition;
		
		public FieldMappingPair(TreeMap<String, String> fieldMapping,
				List<ColumnDefinition> columnsDefinition) {
			super();
			this.fieldMapping = fieldMapping;
			this.columnsDefinition = columnsDefinition;
		}
		/**
		 * @return the fieldMapping
		 */
		public TreeMap<String, String> getFieldMapping() {
			return fieldMapping;
		}
		/**
		 * @return the columnsDefinition
		 */
		public List<ColumnDefinition> getColumnsDefinition() {
			return columnsDefinition;
		}
		
	}

	
	
}
