package org.gcube.dbinterface.persistence;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Iterator;
import java.util.Map;
import java.util.Map.Entry;
import java.util.TreeMap;
import org.gcube.common.dbinterface.Specification;
import org.gcube.common.dbinterface.TableAlreadyExistsException;
import org.gcube.common.dbinterface.preparedstatement.PreparedInsert;
import org.gcube.common.dbinterface.queries.Select;
import org.gcube.common.dbinterface.queries.update.CreateTable;
import org.gcube.common.dbinterface.queries.update.Delete;
import org.gcube.common.dbinterface.queries.update.batch.BatchUpdater;
import org.gcube.common.dbinterface.registry.Connection;
import org.gcube.common.dbinterface.registry.DBInterface;
import org.gcube.common.dbinterface.registry.Shortcuts;
import org.gcube.common.dbinterface.tables.Table;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import static org.gcube.common.dbinterface.utils.Conditions.*;
import static org.gcube.common.dbinterface.utils.Attributes.*;
import static org.gcube.common.dbinterface.utils.Types.*;

public class SystemTableInfo {

	private static final Logger logger = LoggerFactory.getLogger(SystemTableInfo.class);
	
	private static final String SYSTEM_INFO_TABLE="internalgcubesysteminfotable";
	private static final String TABLE_NAME_FIELD="tablename";
	private static final String FIELD_NAME_FIELD="fieldname";
	private static final String COLUMN_NAME_FIELD="coulumnname";
	
	private DBInterface dbInterface;
	
	private static SystemTableInfo systemTableInfo= null;
	
	private SystemTableInfo(DBInterface dbInterface) throws Exception{
		this.dbInterface = dbInterface;
		Connection session= null;
		try{
			session = dbInterface.getConnection();
			this.create(session);
		}catch (TableAlreadyExistsException e) {
			logger.trace("the system table already exists");
		}finally{
			if (session!=null) dbInterface.release(session);
		}
	}
	
	protected static SystemTableInfo getSystemInfo(DBInterface dbInterface) throws Exception{
		if (systemTableInfo == null) systemTableInfo =  new SystemTableInfo(dbInterface);
		return systemTableInfo;
	}
	
	private void create(Connection session) throws TableAlreadyExistsException,Exception{
		Shortcuts sc = dbInterface.queries();
		CreateTable creator= dbInterface.builders().createTable().name(SYSTEM_INFO_TABLE).columns(
				sc.column(TABLE_NAME_FIELD, string(100), Specification.NOT_NULL), sc.column(FIELD_NAME_FIELD, string(100), Specification.NOT_NULL),
				sc.column(COLUMN_NAME_FIELD, string(100), Specification.NOT_NULL)
				).build();

		//TODO: create indexes !!!
		try{
			session.executeUpdate(creator);	
		}catch (SQLException e) {
			throw new TableAlreadyExistsException(e.getMessage());
		}	
	}
	
	protected TreeMap<String, String> retrieveInfo(String tableName) throws Exception{
		//the map to return has the class field name as key and the field name in the table as value
		Select select= dbInterface.builders().select().values(attribute(FIELD_NAME_FIELD), attribute(COLUMN_NAME_FIELD))
				.from(new Table(SYSTEM_INFO_TABLE)).where(eq(attribute(TABLE_NAME_FIELD), tableName)).build();
		logger.trace("info query is {}",select.getQuery());
		TreeMap<String, String> toReturn= new TreeMap<String, String>();
		Connection session= dbInterface.getConnection();
		try{
			
			ResultSet res= session.executeQuery(select).asDefault();
			while (res.next())
				toReturn.put(res.getString(FIELD_NAME_FIELD).toLowerCase(), res.getString(COLUMN_NAME_FIELD).toLowerCase());
		}catch(SQLException e){
			logger.error("error retrieving info",e);
			throw e;		
		}finally{
			dbInterface.release(session);
		}
		return toReturn;
	}
	
	protected String retrieveFieldName(String tableName, String columnName) throws Exception{
		//the map to return has the class field name as key and the field name in the table as value
		Select select= dbInterface.builders().select().values(attribute(FIELD_NAME_FIELD), attribute(COLUMN_NAME_FIELD))
				.from(new Table(SYSTEM_INFO_TABLE)).where(and(eq(attribute(TABLE_NAME_FIELD), tableName), eq(attribute(COLUMN_NAME_FIELD), columnName))).build();
		
		Connection session= dbInterface.getConnection();
		try{
			//logger.trace("retrieve info query is "+select.getExpression());
			ResultSet res= session.executeQuery(select).asDefault();
			if (!res.next()) throw new Exception("error retrieving the field corresponding to column name "+columnName);
			return res.getString(1);
		}finally{
			dbInterface.release(session);
		}
		
	}
	
	protected void addInfo(Map<String, String> fieldMapping, final String tableName) throws Exception{
		
		Connection conn = dbInterface.getConnection();
		
		PreparedInsert preparedInsert = dbInterface.builders().preparedInsert().table(SYSTEM_INFO_TABLE).elements(3).build();
		
		final Iterator<Entry<String, String>> mappingIterator =  fieldMapping.entrySet().iterator();
		
		Iterator<Object[]> it = new Iterator<Object[]>() {
			public boolean hasNext() {
				return mappingIterator.hasNext();
			}

			public Object[] next() {
				Entry<String, String> entry = mappingIterator.next();
				logger.trace("adding "+tableName+" - "+entry.getKey()+ "- "+entry.getValue());
				return new Object[]{tableName,entry.getKey(), entry.getValue() };
			}

			public void remove() {
				mappingIterator.remove();				
			}
		};
		
		BatchUpdater updater = dbInterface.builders().batchInsert().stopOnError().use(preparedInsert)
				.elements(it).build();
		
		logger.trace("prepared statement: {}",updater.getPreparedObject().getPreparedStatement());
		
		try{
			conn.executeBatchUpdate(updater);
		}finally{
			dbInterface.release(conn);
		}
		
	}
	
	protected void deleteInfo(String tableName) throws Exception{
		Delete deleteInfo= dbInterface.builders().delete().table(SYSTEM_INFO_TABLE).where(eq(attribute(TABLE_NAME_FIELD), tableName)).build();
		
		Connection session= dbInterface.getConnection();
		try{
			session.executeUpdate(deleteInfo);
		}finally{
			if (session!=null) dbInterface.release(session);
		}
	}
	
}
