/*
 * Decompiled with CFR 0.152.
 */
package com.netflix.astyanax.thrift;

import com.google.common.collect.Maps;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import com.netflix.astyanax.AstyanaxConfiguration;
import com.netflix.astyanax.AuthenticationCredentials;
import com.netflix.astyanax.CassandraOperationTracer;
import com.netflix.astyanax.CassandraOperationType;
import com.netflix.astyanax.KeyspaceTracerFactory;
import com.netflix.astyanax.connectionpool.Connection;
import com.netflix.astyanax.connectionpool.ConnectionFactory;
import com.netflix.astyanax.connectionpool.ConnectionPoolConfiguration;
import com.netflix.astyanax.connectionpool.ConnectionPoolMonitor;
import com.netflix.astyanax.connectionpool.Host;
import com.netflix.astyanax.connectionpool.HostConnectionPool;
import com.netflix.astyanax.connectionpool.Operation;
import com.netflix.astyanax.connectionpool.OperationResult;
import com.netflix.astyanax.connectionpool.RateLimiter;
import com.netflix.astyanax.connectionpool.SSLConnectionContext;
import com.netflix.astyanax.connectionpool.exceptions.ConnectionException;
import com.netflix.astyanax.connectionpool.exceptions.IsTimeoutException;
import com.netflix.astyanax.connectionpool.exceptions.ThrottledException;
import com.netflix.astyanax.connectionpool.impl.OperationResultImpl;
import com.netflix.astyanax.connectionpool.impl.SimpleRateLimiterImpl;
import com.netflix.astyanax.thrift.ThriftConverter;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import org.apache.cassandra.thrift.AuthenticationRequest;
import org.apache.cassandra.thrift.Cassandra;
import org.apache.cassandra.thrift.TBinaryProtocol;
import org.apache.thrift.protocol.TProtocol;
import org.apache.thrift.transport.TFramedTransport;
import org.apache.thrift.transport.TSSLTransportFactory;
import org.apache.thrift.transport.TSocket;
import org.apache.thrift.transport.TTransport;
import org.apache.thrift.transport.TTransportException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ThriftSyncConnectionFactoryImpl
implements ConnectionFactory<Cassandra.Client> {
    private static final String NAME_FORMAT = "ThriftConnection<%s-%d>";
    private static final Logger LOG = LoggerFactory.getLogger(ThriftSyncConnectionFactoryImpl.class);
    private static final ExecutorService executor = Executors.newCachedThreadPool(new ThreadFactoryBuilder().setDaemon(true).build());
    private final AtomicLong idCounter = new AtomicLong(0L);
    private final RateLimiter limiter;
    private final ConnectionPoolConfiguration cpConfig;
    private final KeyspaceTracerFactory tracerFactory;
    private final ConnectionPoolMonitor monitor;
    private final AstyanaxConfiguration asConfig;

    public ThriftSyncConnectionFactoryImpl(AstyanaxConfiguration asConfig, ConnectionPoolConfiguration cpConfig, KeyspaceTracerFactory tracerFactory, ConnectionPoolMonitor monitor) {
        this.cpConfig = cpConfig;
        this.asConfig = asConfig;
        this.limiter = new SimpleRateLimiterImpl(cpConfig);
        this.tracerFactory = tracerFactory;
        this.monitor = monitor;
    }

    @Override
    public Connection<Cassandra.Client> createConnection(HostConnectionPool<Cassandra.Client> pool) throws ThrottledException {
        if (!this.limiter.check()) {
            throw new ThrottledException("Too many connection attempts");
        }
        return new ThriftConnection(pool);
    }

    public class ThriftConnection
    implements Connection<Cassandra.Client> {
        private final long id;
        private Cassandra.Client cassandraClient;
        private TFramedTransport transport;
        private TSocket socket;
        private int timeout;
        private AtomicLong operationCounter;
        private AtomicBoolean closed;
        private volatile ConnectionException lastException;
        private volatile String keyspaceName;
        private final HostConnectionPool<Cassandra.Client> pool;
        private Map<String, Object> metadata;

        public ThriftConnection(HostConnectionPool<Cassandra.Client> pool) {
            this.id = ThriftSyncConnectionFactoryImpl.this.idCounter.incrementAndGet();
            this.timeout = 0;
            this.operationCounter = new AtomicLong();
            this.closed = new AtomicBoolean(false);
            this.lastException = null;
            this.metadata = Maps.newHashMap();
            this.pool = pool;
        }

        @Override
        public <R> OperationResult<R> execute(Operation<Cassandra.Client, R> op) throws ConnectionException {
            long now;
            long startTime = System.nanoTime();
            long latency = 0L;
            this.setTimeout(ThriftSyncConnectionFactoryImpl.this.cpConfig.getSocketTimeout());
            this.operationCounter.incrementAndGet();
            this.lastException = null;
            if (!(op.getKeyspace() == null || this.keyspaceName != null && op.getKeyspace().equals(this.keyspaceName))) {
                CassandraOperationTracer tracer = ThriftSyncConnectionFactoryImpl.this.tracerFactory.newTracer(CassandraOperationType.SET_KEYSPACE).start();
                try {
                    this.cassandraClient.set_keyspace(op.getKeyspace());
                    if (ThriftSyncConnectionFactoryImpl.this.asConfig.getCqlVersion() != null) {
                        this.cassandraClient.set_cql_version(ThriftSyncConnectionFactoryImpl.this.asConfig.getCqlVersion());
                    }
                    this.keyspaceName = op.getKeyspace();
                    now = System.nanoTime();
                    latency = now - startTime;
                    this.pool.addLatencySample(latency, now);
                    tracer.success();
                }
                catch (Exception e) {
                    long now2 = System.nanoTime();
                    latency = now2 - startTime;
                    this.lastException = ThriftConverter.ToConnectionPoolException(e).setLatency(latency);
                    if (e instanceof IsTimeoutException) {
                        this.pool.addLatencySample(TimeUnit.NANOSECONDS.convert(ThriftSyncConnectionFactoryImpl.this.cpConfig.getSocketTimeout(), TimeUnit.MILLISECONDS), now2);
                    }
                    tracer.failure(this.lastException);
                    throw this.lastException;
                }
                startTime = System.nanoTime();
            }
            try {
                R result = op.execute(this.cassandraClient, this);
                now = System.nanoTime();
                latency = now - startTime;
                this.pool.addLatencySample(latency, now);
                return new OperationResultImpl<R>(this.getHost(), result, latency);
            }
            catch (Exception e) {
                now = System.nanoTime();
                latency = now - startTime;
                this.lastException = ThriftConverter.ToConnectionPoolException(e).setLatency(latency);
                if (e instanceof IsTimeoutException) {
                    this.pool.addLatencySample(TimeUnit.NANOSECONDS.convert(ThriftSyncConnectionFactoryImpl.this.cpConfig.getSocketTimeout(), TimeUnit.MILLISECONDS), now);
                }
                throw this.lastException;
            }
        }

        @Override
        public void open() throws ConnectionException {
            if (this.cassandraClient != null) {
                throw new IllegalStateException("Open called on already open connection");
            }
            long startTime = System.currentTimeMillis();
            try {
                SSLConnectionContext sslCxt = ThriftSyncConnectionFactoryImpl.this.cpConfig.getSSLConnectionContext();
                if (sslCxt != null) {
                    TSSLTransportFactory.TSSLTransportParameters params = new TSSLTransportFactory.TSSLTransportParameters(sslCxt.getSslProtocol(), sslCxt.getSslCipherSuites().toArray(new String[0]));
                    params.setTrustStore(sslCxt.getSslTruststore(), sslCxt.getSslTruststorePassword());
                    this.socket = TSSLTransportFactory.getClientSocket((String)this.getHost().getIpAddress(), (int)this.getHost().getPort(), (int)ThriftSyncConnectionFactoryImpl.this.cpConfig.getSocketTimeout(), (TSSLTransportFactory.TSSLTransportParameters)params);
                } else {
                    this.socket = new TSocket(this.getHost().getIpAddress(), this.getHost().getPort(), ThriftSyncConnectionFactoryImpl.this.cpConfig.getConnectTimeout());
                }
                this.socket.getSocket().setTcpNoDelay(true);
                this.socket.getSocket().setKeepAlive(true);
                this.socket.getSocket().setSoLinger(false, 0);
                this.setTimeout(ThriftSyncConnectionFactoryImpl.this.cpConfig.getSocketTimeout());
                this.transport = new TFramedTransport((TTransport)this.socket);
                if (!this.transport.isOpen()) {
                    this.transport.open();
                }
                this.cassandraClient = new Cassandra.Client((TProtocol)new TBinaryProtocol((TTransport)this.transport));
                ThriftSyncConnectionFactoryImpl.this.monitor.incConnectionCreated(this.getHost());
                AuthenticationCredentials credentials = ThriftSyncConnectionFactoryImpl.this.cpConfig.getAuthenticationCredentials();
                if (credentials != null) {
                    HashMap thriftCredentials = Maps.newHashMapWithExpectedSize((int)2);
                    thriftCredentials.put("username", credentials.getUsername());
                    thriftCredentials.put("password", credentials.getPassword());
                    this.cassandraClient.login(new AuthenticationRequest((Map)thriftCredentials));
                }
            }
            catch (Exception e) {
                this.pool.addLatencySample(TimeUnit.NANOSECONDS.convert(ThriftSyncConnectionFactoryImpl.this.cpConfig.getSocketTimeout(), TimeUnit.MILLISECONDS), System.nanoTime());
                this.closeClient();
                ConnectionException ce = ThriftConverter.ToConnectionPoolException(e).setHost(this.getHost()).setLatency(System.currentTimeMillis() - startTime);
                ThriftSyncConnectionFactoryImpl.this.monitor.incConnectionCreateFailed(this.getHost(), ce);
                throw ce;
            }
            catch (Throwable t) {
                LOG.error("Error creating connection", t);
                this.pool.addLatencySample(TimeUnit.NANOSECONDS.convert(ThriftSyncConnectionFactoryImpl.this.cpConfig.getSocketTimeout(), TimeUnit.MILLISECONDS), System.nanoTime());
                this.closeClient();
                ConnectionException ce = ThriftConverter.ToConnectionPoolException(new RuntimeException("Error openning connection", t)).setHost(this.getHost()).setLatency(System.currentTimeMillis() - startTime);
                ThriftSyncConnectionFactoryImpl.this.monitor.incConnectionCreateFailed(this.getHost(), ce);
                throw ce;
            }
        }

        @Override
        public void openAsync(final Connection.AsyncOpenCallback<Cassandra.Client> callback) {
            final ThriftConnection This = this;
            executor.submit(new Runnable(){

                @Override
                public void run() {
                    try {
                        ThriftConnection.this.open();
                        callback.success(This);
                    }
                    catch (Exception e) {
                        ThriftConnection.this.pool.addLatencySample(TimeUnit.NANOSECONDS.convert(ThriftSyncConnectionFactoryImpl.this.cpConfig.getSocketTimeout(), TimeUnit.MILLISECONDS), System.nanoTime());
                        callback.failure(This, ThriftConverter.ToConnectionPoolException(e));
                    }
                }
            });
        }

        @Override
        public void close() {
            if (this.closed.compareAndSet(false, true)) {
                ThriftSyncConnectionFactoryImpl.this.monitor.incConnectionClosed(this.getHost(), this.lastException);
                executor.submit(new Runnable(){

                    @Override
                    public void run() {
                        try {
                            ThriftConnection.this.closeClient();
                        }
                        catch (Exception exception) {
                            // empty catch block
                        }
                    }
                });
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private void closeClient() {
            if (this.transport != null) {
                try {
                    this.transport.flush();
                }
                catch (TTransportException e) {
                    try {
                        this.transport.close();
                    }
                    catch (Exception e2) {
                    }
                    finally {
                        this.transport = null;
                    }
                }
                finally {
                    try {
                        this.transport.close();
                    }
                    catch (Exception e) {
                    }
                    finally {
                        this.transport = null;
                    }
                }
            }
            if (this.socket != null) {
                try {
                    this.socket.close();
                }
                catch (Exception exception) {
                }
                finally {
                    this.socket = null;
                }
            }
        }

        @Override
        public HostConnectionPool<Cassandra.Client> getHostConnectionPool() {
            return this.pool;
        }

        @Override
        public ConnectionException getLastException() {
            return this.lastException;
        }

        public String toString() {
            return String.format(ThriftSyncConnectionFactoryImpl.NAME_FORMAT, this.getHost().getHostName(), this.id);
        }

        public boolean equals(Object obj) {
            return this.toString().equals(obj.toString());
        }

        @Override
        public long getOperationCount() {
            return this.operationCounter.get();
        }

        @Override
        public Host getHost() {
            return this.pool.getHost();
        }

        public void setTimeout(int timeout) {
            if (this.timeout != timeout) {
                this.socket.setTimeout(timeout);
                this.timeout = timeout;
            }
        }

        @Override
        public void setMetadata(String key, Object data) {
            this.metadata.put(key, data);
        }

        @Override
        public Object getMetadata(String key) {
            return this.metadata.get(key);
        }

        @Override
        public boolean hasMetadata(String key) {
            return this.metadata.containsKey(key);
        }
    }
}

