package org.gcube.common.clients;

import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.InputStream;
import java.util.Enumeration;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * The runtime of gCube clients. 
 * 
 * @author Fabio Simeoni
 *
 */
public class ClientRuntime {

	public static final String location_property = "GLOBUS_LOCATION";

	private static Logger log = LoggerFactory.getLogger(ClientRuntime.class);
	
	private static final String installation_name = "ghn-client-distro";
	private static final String archive_name=installation_name+".zip"; 
	
	/**
	 * Installs an embedded client distribution on a temporary location on the file system, and then starts it. 
	 */
	public static synchronized void start() {
		
		String location = System.getProperty(location_property);
		
		if (location==null || !new File(location).exists())
			try {
				install();
				
				Runtime.getRuntime().addShutdownHook(new Thread() {
					public void run() {
						try {
							ClientRuntime.stop();
						}
						catch(Throwable t) {
							log.warn("could not stop the client container and its services",t);
						}
					};
				});
				
			}
			catch(Throwable t) {
				throw new RuntimeException("could not start client runtime",t);
			}
		else 
			log.info("client runtime is already started");
		
			
	}

	/**
	 * Deletes the client distribution. 
	 */
	public static synchronized void stop() {
		
		try {
			new File(tempDir(),archive_name).delete();
			new File(tempDir(),installation_name).delete();
			log.info("stopped client runtime");
		}
		catch(Throwable t) {
			throw new RuntimeException("could not stop client runtime",t);
		}
	}
	
	private static File tempDir() throws Exception {
		
		File random = File.createTempFile("temp", null);
		File dir = random.getParentFile();
		random.delete();
		return dir;
	}
	
	private static void copy(InputStream in, File file) throws Exception {
	
		byte data[] = new byte[2048];

		BufferedOutputStream out = new BufferedOutputStream(new FileOutputStream(file));
		
		int read = 0;
		while ((read=in.read(data))!=-1)
			out.write(data,0,read);
		
		out.flush();
		out.close();
		in.close();
	}
	
	private static void install() throws Exception {

		
		InputStream embedded = ClientRuntime.class.getResourceAsStream("/"+archive_name);
		File archive = new File(tempDir(),archive_name);
		copy(embedded,archive);
		
		ZipFile zipFile = new ZipFile(archive);
		
		// Create an enumeration of the entries in the zip file
		Enumeration<? extends ZipEntry> zipFileEntries = zipFile.entries();

		// Process each entry
		while (zipFileEntries.hasMoreElements()) {

			ZipEntry entry = zipFileEntries.nextElement();

			File destFile = new File(tempDir(), entry.getName());

			if (entry.isDirectory())
				destFile.mkdirs();
			else
				copy(zipFile.getInputStream(entry),destFile);
		
//			if (!destFile.exists()) {
//			}

		}
		zipFile.close();

		String installation = new File(tempDir(),installation_name).getAbsolutePath();
		System.setProperty(location_property, installation);
//		addToEnvironment(installation);
		
		log.info("started client container in {}",installation);

	}
	
//	@SuppressWarnings({ "rawtypes", "unchecked" })
//	// helper
//	// this is a serious hack to deal with legacy code, we are reflectively
//	// changing the
//	// unmodifiable map that contains the env vars in Java
//	private static void addToEnvironment(String installation) throws Exception {
//		Class<?>[] classes = Collections.class.getDeclaredClasses();
//		Map<String, String> env = System.getenv();
//		for (Class<?> cl : classes) {
//			if ("java.util.Collections$UnmodifiableMap".equals(cl.getName())) {
//				Field field = cl.getDeclaredField("m");
//				field.setAccessible(true);
//				Map<String, String> map = (Map) field.get(env);
//				log.debug("setting GLOBUS_LOCATION to {}",installation);
//				map.put("GLOBUS_LOCATION", installation);
//			}
//		}
//	}
}
