package org.gcube.security.soa3.connector.integration.server;

import java.rmi.Remote;
import java.util.Collections;
import java.util.Map;
import java.util.WeakHashMap;

import org.apache.axis.MessageContext;
import org.apache.axis.client.Stub;
import org.gcube.common.core.contexts.GCUBEServiceContext;
import org.gcube.common.core.security.SecurityCredentials;
import org.gcube.common.core.security.context.SecurityContextFactory;
import org.gcube.common.core.utils.logging.GCUBELog;
import org.gcube.security.soa3.configuration.ConfigurationManager;
import org.gcube.security.soa3.connector.integration.TokenCredentials;
import org.gcube.security.soa3.connector.integration.utils.Utils;
import org.gcube.soa3.connector.common.security.CredentialManager;
import org.gcube.soa3.connector.common.security.Credentials;
import org.globus.axis.gsi.GSIConstants;
import org.globus.wsrf.impl.security.authentication.Constants;
import org.ietf.jgss.GSSCredential;
import org.w3c.dom.Element;





public class SOA3IntegrationServiceCredentialInserter extends GCUBEServiceSecurityConfigurationManager
{
	private GCUBELog log;
	private boolean isSecurityEnabled;
	private boolean propagateCallerCredentials;
	private GCUBEServiceContext context;
	/** Cache of credentials indexed by thread id. */
	protected Map<Thread,Element>  callCredentials = Collections.synchronizedMap(new WeakHashMap<Thread,Element>());
	
	public SOA3IntegrationServiceCredentialInserter() 
	{
		this.log = new GCUBELog(this);
		this.propagateCallerCredentials = true;
		
	}

	@Override
	public boolean isSecurityEnabled() 
	{
		return isSecurityEnabled;
	}

	@Override
	public void useCredentials(GSSCredential credentials) throws Exception 
	{
		// TODO Auto-generated method stub
		
	}

	@Override
	public void useCredentials(SecurityCredentials credentials) throws Exception 
	{
		this.useCredentials(Thread.currentThread(),credentials);		
	}

	@Override
	public void useCredentials(Thread thread, SecurityCredentials... credentials) throws Exception 
	{
		if (!isSecurityEnabled()) return;	
		
		
		if (credentials.length==0) {// if no credentials are provided, use the current ones.
			credentials=new SecurityCredentials[]{this.getCredentials()};
		}
		//debug some information
		try 
		{
			Element tokenCredentials = (Element) (credentials[0].getCredentialsAsObject());
			log.debug("Using credentials of the token ("+credentials[0].getCredentialsAsString()+") in thread "+thread.getName()+"("+thread.getId()+")");
					
			callCredentials.put(thread,tokenCredentials);
	
		} catch (ClassCastException e)
		{
			log.error("Invalid credentials: expected Token Credentials and found "+ (credentials[0].getCredentialsAsObject ()).getClass());
		}

		
	}

	@Override
	public SecurityCredentials getCredentials() 
	{
		Element internalCred = this.callCredentials.get(Thread.currentThread());
		
		if (internalCred != null)
		{
			log.debug("Credentials found for the current thread");
			return new TokenCredentials(internalCred);
		}
		else if (propagateCallerCredentials)
		{
			log.debug("Credentials not set, using caller credentials");
			try 
			{
				return getCallerCredentials();
			} catch (Exception e)
			{
				log.error("Unable to find caller credentials, probably the credentials are not available, disable the credential propagation in the service configuration", e);
				log.error("Trying to use service credentials");
			}
		}

		// As last resort, service credentials will be used
		log.debug("Using service credentials");
		try 
		{
			return getServiceCredentials();
		} catch (Exception e)
		{
			log.error("Unable to find service credentials", e);
			log.error("No authorization credentials will be used");
			return null;
		}
	}

	@Override
	public void setSecurity(Remote s, AuthMode e, DelegationMode d) throws Exception 
	{
		this.log.debug("setting security parameters for service "+this.context.getService().getServiceName());
		
		if (isSecurityEnabled())
		{
			Stub stub = (Stub) s;
			// set the host certificate
			stub._setProperty(GSIConstants.GSI_CREDENTIALS, SecurityContextFactory.getInstance().getSecurityContext().getDefaultCredentials());
			//sets credentials (caller or service)
			TokenCredentials credentials = (TokenCredentials) getCredentials();
			//sets authentication
			if (credentials != null && credentials.getType() != null && credentials.getValue() != null)
			{
				this.log.debug("setting Security Token");
				stub.setHeader(Utils.generateSoapHeaderBinaryTokenElement(credentials.getType(), credentials.getValue()));
				this.log.debug("SAML assertion set");
			}
			else
			{
				this.log.error("credentials or token not found");
			}
			

			
			switch(e) {
				case INTEGRITY:stub._setProperty(Constants.GSI_TRANSPORT, Constants.SIGNATURE);break;
				case PRIVACY:stub._setProperty(Constants.GSI_TRANSPORT, Constants.ENCRYPTION);break;
				case BOTH:stub._setProperty(Constants.GSI_TRANSPORT, Constants.SIGNATURE);stub._setProperty(Constants.GSI_TRANSPORT, Constants.ENCRYPTION);
			}
			log.debug("Setting authentication GSI sec transport= "+e.name()+" on "+stub.getClass().getSimpleName());	
			
						
		}
		else 
		{
			log.debug("Security not enabled, nothing to do");
		}		
	}



	@Override
	public void initialise(GCUBEServiceContext ctxt) throws Exception 
	{
		this.context = ctxt;
		String serviceName = ctxt.getName();
		log.debug("Service name "+serviceName);
		if (!ConfigurationManager.getInstance().servicePropertiesSet(serviceName)) Utils.setServiceProperties(ctxt,serviceName);	
		this.isSecurityEnabled = ConfigurationManager.getInstance().isSecurityEnabled(ctxt.getName());
		this.propagateCallerCredentials(ConfigurationManager.getInstance().getCredentialPropagationPolicy(serviceName));
		log.debug("Init completed");
		
	}



	@Override
	public SecurityCredentials getServiceCredentials() throws Exception 
	{
		Credentials credentials = CredentialManager.instance.get();
		return new TokenCredentials(credentials.getAuthenticationType(),credentials.getHeaderString());
	}

	@Override
	public SecurityCredentials getCallerCredentials() throws Exception 
	{
		return getCallerToken();
	}



	@Override
	public boolean needServiceCredentials() {
		return true;
	}

	@Override
	public void propagateCallerCredentials(boolean propagateCallerCredentials) 
	{
		this.propagateCallerCredentials = propagateCallerCredentials;
		
	}

    /**
     * Retrieves the SAML Assertion from the {@link MessageContext}
     * @return the caller SAML Assertion
     */
    private TokenCredentials getCallerToken() 
    {
    	this.log.debug("getting caller Security Token...");
        MessageContext msgCtx = MessageContext.getCurrentContext();
        Element token = (Element) msgCtx.getProperty(Utils.SECURITY_TOKEN);
        TokenCredentials response = null;
        
        if (token == null) {
            log.warn("The Token is null!");
        }
        else
        {	response = new TokenCredentials(token);
	        try
	        {
	        	this.log.debug("Token found = "+ response.getCredentialsAsString());
	        }
	        catch (Exception e)
	        {
	        	log.error("Invalid token ",e);
	        }
        	
        }
        
        return response;
    } 

    
}
