package gr.uoa.di.madgik.environment.ftp;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.net.URL;
import java.net.URLConnection;
import java.util.UUID;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.apache.commons.net.ftp.FTP;
import org.apache.commons.net.ftp.FTPClient;
import org.apache.commons.net.ftp.FTPReply;

import gr.uoa.di.madgik.environment.exception.EnvironmentStorageSystemException;
import gr.uoa.di.madgik.environment.hint.EnvHintCollection;
import gr.uoa.di.madgik.environment.ss.IStorageSystemProvider;

public class FTPStorageSystemProvider implements IStorageSystemProvider
{
	public static final String FTPURLHintName="StorageSystemFTPURL";
	public static final String DeleteOnExitHintName="StorageSystemDeleteOnExit";
	public static final String LocalFileSystemBufferPathHintName="StorageSystemLocalFileSystemBufferPath";
	public static final String UserNameHintName="UserName";
	public static final String PasswordHintName="Password";
	public static final String PortHintName="Port";
	private static final String DefaultLocalFileSystemBufferPath="/tmp/";
	private static final String DefaultFTPURL="ftp://pe2ng:pe2ng2011@donald.di.uoa.gr/indexStorage/";
	private static final String DefaultUserName="pe2ng";
	private static final String DefaultPassword="pe2ng2011";
	
	private String host = null;
	private String path = null;
	private String userName = null;
	private String password = null;
	private Integer port = null;
	
	private static Logger logger = Logger.getLogger(FTPStorageSystemProvider.class.getName());
	
	private FTPClient ftpClient = new FTPClient();
	
	public void SessionInit(EnvHintCollection Hints) throws EnvironmentStorageSystemException
	{
		String userName = GetUserName(Hints);
		String password = GetPassword(Hints);
		Integer port = GetPort(Hints);
		
		String storeUrlLocation = null;
		if(Hints==null || !Hints.HintExists(FTPStorageSystemProvider.FTPURLHintName)) storeUrlLocation = FTPStorageSystemProvider.DefaultFTPURL;
		else storeUrlLocation=Hints.GetHint(FTPStorageSystemProvider.FTPURLHintName).Hint.Payload;
		if(storeUrlLocation == null) storeUrlLocation = FTPStorageSystemProvider.DefaultFTPURL;
		URL url = null;
		try { url = new URL(storeUrlLocation); } catch(Exception e){ throw new EnvironmentStorageSystemException("Malformed FTP host"); }
		if(url.getPort() > 0 ) 
		{ 
			if(this.port <= 0) this.port = url.getPort();
			else 
			{ 
				if(url.getPort() != this.port) throw new EnvironmentStorageSystemException("URL/Configuration port mismatch");
			}
		}
		
		String afterProto = storeUrlLocation.substring(storeUrlLocation.indexOf("//")+2);
		String afterHost = afterProto.substring(afterProto.indexOf("/")+1);
		String embeddedAuthority = afterProto;
		if(afterProto.indexOf("@") != -1)
		{
			if(storeUrlLocation.indexOf("//")+2+afterProto.indexOf("@") < afterProto.indexOf("/")+1)
			{
				embeddedAuthority = afterProto.substring(0, afterProto.indexOf("@"));
				String[] userPass = embeddedAuthority.split(":");
				if(userPass.length!=2) throw new EnvironmentStorageSystemException("Malformed user/pass pair in URL");
				if(userName != null)
				{
					if(!userName.equals(userPass[0])) throw new EnvironmentStorageSystemException("URL/Configuration user name mismatch");
				}
				else userName = userPass[0];
				
				if(password != null)
				{
					if(!password.equals(userPass[1])) throw new EnvironmentStorageSystemException("URL/Configuration password mismatch");
				}
				else password = userPass[1];
			}
		}
		if(userName == null)
		{
			if(password != null) throw new EnvironmentStorageSystemException("Unspecified user name");
		}
		if(password == null)
		{
			if(userName != null) throw new EnvironmentStorageSystemException("Unspecified password");
		}
		
		host = url.getHost();
		path = afterHost;
		if(userName != null) this.userName = userName;
		else this.userName = DefaultUserName;
		if(password != null) this.password = password;
		else this.password = DefaultPassword;
		if(port != null) this.port = port;
	}
	
	public File Retrieve(String ID, EnvHintCollection Hints) throws EnvironmentStorageSystemException
	{
		try
		{
			String newID=UUID.randomUUID().toString();
			File newtmp=FTPStorageSystemProvider.GetLocalFileSystemBufferFile(newID, Hints);
			if(FTPStorageSystemProvider.ShouldDeleteOnExit(Hints)) newtmp.deleteOnExit();
			
			SessionInit(Hints);
			try
			{
				Connect(host);
				ChangeDirectory(path);
				RemoteCopyFile(ID, newtmp);
			}
			finally
			{
				if(ftpClient.isConnected())
				{
					ftpClient.logout();
					ftpClient.disconnect();
				}
			}
			return newtmp;
		}catch(Exception ex)
		{
			throw new EnvironmentStorageSystemException("Could not retrieve requested content", ex);
		}
	}
	
	public void Delete(String ID, EnvHintCollection Hints) throws EnvironmentStorageSystemException
	{
		try
		{
			SessionInit(Hints);
			try
			{
				Connect(host);
				ChangeDirectory(path);
				ftpClient.deleteFile(getRemoteFileName(ID));
				int reply = ftpClient.getReplyCode();
				logger.log(Level.FINER, "Delete: Server response:" + ftpClient.getReplyString());
				if(!FTPReply.isPositiveCompletion(reply)) throw new EnvironmentStorageSystemException("Could not delete file from FTP server: " + reply + ": " + ftpClient.getReplyString());
			}
			finally
			{
				if(ftpClient.isConnected())
				{
					ftpClient.logout();
					ftpClient.disconnect();
				}
			}
		}catch(Exception ex)
		{
			throw new EnvironmentStorageSystemException("Could not retrieve requested content", ex);
		}
	}
	
	public String Store(File file, EnvHintCollection Hints) throws EnvironmentStorageSystemException
	{
		try
		{
			String ID=UUID.randomUUID().toString();
			SessionInit(Hints);
			boolean loggedIn = false;
			try
			{
				Connect(host);
				ChangeDirectory(path);
				RemoteCopyFile(file, ID);
			}
			finally
			{
				if(loggedIn) ftpClient.logout();
				if(ftpClient.isConnected()) ftpClient.disconnect();
			}
			return ID;
		}catch(Exception ex)
		{
			throw new EnvironmentStorageSystemException("Could not store provided content", ex);
		}
	}

	public String Store(URL location, EnvHintCollection Hints) throws EnvironmentStorageSystemException
	{
		try
		{
			String newID=UUID.randomUUID().toString();
			File newtmp=FTPStorageSystemProvider.GetLocalFileSystemBufferFile(newID, Hints);
			newtmp.deleteOnExit();
			String ID=UUID.randomUUID().toString();
			SessionInit(Hints);
			boolean loggedIn = false;
			try
			{
				Connect(host);
				ChangeDirectory(path);
				RemoteCopyFile(location, newtmp);
				RemoteCopyFile(newtmp, ID);
			}
			finally
			{
				if(loggedIn) ftpClient.logout();
				if(ftpClient.isConnected()) ftpClient.disconnect();
			}
	
			newtmp.delete();
			return ID;
		}catch(Exception ex)
		{
			throw new EnvironmentStorageSystemException("Could not store provided content", ex);
		}
	}

	public File GetLocalFSBufferLocation(EnvHintCollection Hints) throws EnvironmentStorageSystemException
	{
		if(Hints==null || !Hints.HintExists(FTPStorageSystemProvider.LocalFileSystemBufferPathHintName))
			return new File(FTPStorageSystemProvider.DefaultLocalFileSystemBufferPath);
		String loc=Hints.GetHint(FTPStorageSystemProvider.LocalFileSystemBufferPathHintName).Hint.Payload;
		if(loc==null || loc.trim().length()==0) return new File(FTPStorageSystemProvider.DefaultLocalFileSystemBufferPath);
		return new File(loc);
	}
	
	private void Connect(String host) throws EnvironmentStorageSystemException
	{
		try
		{
			ftpClient.connect(host, this.port != null ? this.port : ftpClient.getDefaultPort());
			int reply = ftpClient.getReplyCode();
			logger.log(Level.FINER, "Connect: Server response: " + ftpClient.getReplyString());
			if(!FTPReply.isPositiveCompletion(reply)) throw new EnvironmentStorageSystemException("Could not connect to FTP server: " + reply + ": " + ftpClient.getReplyString());
			ftpClient.login(this.userName, this.password);
			reply = ftpClient.getReplyCode();
			logger.log(Level.FINER, "Login: Server response:" + ftpClient.getReplyString());
			if(!FTPReply.isPositiveCompletion(reply)) throw new EnvironmentStorageSystemException("Could not login to FTP server: " + reply + ": " + ftpClient.getReplyString());
			ftpClient.setFileType(FTP.BINARY_FILE_TYPE);
		}catch(Exception e)
		{
			if(ftpClient.isConnected()) try { ftpClient.disconnect();} catch(Exception ee) { throw new EnvironmentStorageSystemException("Could not disconnect from server", ee); }
			throw new EnvironmentStorageSystemException("Could not initiate FTP session", e);
		}
	}
	
	private void ChangeDirectory(String path) throws EnvironmentStorageSystemException
	{
		try
		{
			ftpClient.changeWorkingDirectory(path);
			int reply = ftpClient.getReplyCode();
			logger.log(Level.FINER, "Chdir: Server response: " + ftpClient.getReplyString());
			if(!FTPReply.isPositiveCompletion(reply)) throw new EnvironmentStorageSystemException("Could not change working directory: " + reply + ": " + ftpClient.getReplyString());
		}catch(Exception e)
		{
			throw new EnvironmentStorageSystemException("Could not change working directory",e);
		}
	}
	
	private static void RemoteCopyFile(File src, URL dest) throws IOException
	{
		URLConnection con = dest.openConnection();
		BufferedOutputStream bout=new BufferedOutputStream(con.getOutputStream());
		BufferedInputStream din=new BufferedInputStream(new FileInputStream(src));
		while(true)
		{
			byte[] b=new byte[1024*4];
			int read=din.read(b);
			if(read<0) break;
			bout.write(b, 0, read);
		}
		din.close();
		bout.flush();
		bout.close();
	}
	
	private static void RemoteCopyFile(URL src, File dest) throws IOException
	{
		URLConnection con = src.openConnection();
		BufferedOutputStream bout=new BufferedOutputStream(new FileOutputStream(dest));
		BufferedInputStream din=new BufferedInputStream(con.getInputStream());
		while(true)
		{
			byte[] b=new byte[1024*4];
			int read=din.read(b);
			if(read<0) break;
			bout.write(b, 0, read);
		}
		din.close();
		bout.flush();
		bout.close();
	}
	
	private void RemoteCopyFile(String ID, File dest) throws Exception
	{
		BufferedOutputStream bout=new BufferedOutputStream(new FileOutputStream(dest));
		ftpClient.retrieveFile(getRemoteFileName(ID), bout);
		int reply = ftpClient.getReplyCode();
		logger.log(Level.FINER, "Retrieve: Server response: " + ftpClient.getReplyString());
		if(!FTPReply.isPositiveCompletion(reply)) throw new EnvironmentStorageSystemException("Could not retrieve file. " + reply + ": " + ftpClient.getReplyString());
		bout.flush();
		bout.close();
	}
	
	private void RemoteCopyFile(File src, String ID) throws Exception
	{
		
		BufferedInputStream din=new BufferedInputStream(new FileInputStream(src));
		ftpClient.storeFile(getRemoteFileName(ID), din);
		int reply = ftpClient.getReplyCode();
		logger.log(Level.FINER, "Retrieve: Server response: " + ftpClient.getReplyString());
		if(!FTPReply.isPositiveCompletion(reply)) throw new EnvironmentStorageSystemException("Could not store file. " + reply + ": " + ftpClient.getReplyString());
	}
	
	private static File GetLocalFileSystemBufferFile(String ID, EnvHintCollection Hints)
	{
		if(Hints==null || !Hints.HintExists(FTPStorageSystemProvider.LocalFileSystemBufferPathHintName))
			return new File(FTPStorageSystemProvider.DefaultLocalFileSystemBufferPath,ID+".ss.tmp");
		String loc=Hints.GetHint(FTPStorageSystemProvider.LocalFileSystemBufferPathHintName).Hint.Payload;
		if(loc==null || loc.trim().length()==0) return new File(FTPStorageSystemProvider.DefaultLocalFileSystemBufferPath,ID+".ss.tmp");
		return new File(loc,ID+".ss.tmp");
	}
	
	private String getRemoteFileName(String ID)
	{
		return ID + ".ss.tmp";
	}
	
	private static String GetStringHint(EnvHintCollection Hints, String hintName)
	{
		if(Hints==null || !Hints.HintExists(hintName))
			return null;
		String payload=Hints.GetHint(hintName).Hint.Payload;
		if(payload==null || payload.trim().length()==0) return null;
		return payload;
	}
	
	private static String GetUserName(EnvHintCollection Hints)
	{
		return GetStringHint(Hints, FTPStorageSystemProvider.UserNameHintName);
	}
	
	private static String GetPassword(EnvHintCollection Hints)
	{
		return GetStringHint(Hints, FTPStorageSystemProvider.PasswordHintName);
	}
	
	private static Integer GetPort(EnvHintCollection Hints)
	{
		String p = GetStringHint(Hints, FTPStorageSystemProvider.PortHintName);
		Integer value = -1;
		if(p != null) value = Integer.parseInt(p);
		if(value > 0) return value;
		return null;
	}
	
	private static boolean ShouldDeleteOnExit(EnvHintCollection Hints)
	{
		if(Hints==null) return true;
		if(!Hints.HintExists(FTPStorageSystemProvider.DeleteOnExitHintName)) return true;
		return Boolean.parseBoolean(Hints.GetHint(FTPStorageSystemProvider.DeleteOnExitHintName).Hint.Payload);
	}
}
