package gr.uoa.di.madgik.grs.bridge;

import java.net.URI;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.gcube.common.searchservice.searchlibrary.resultset.elements.ResultElementBase;
import org.gcube.common.searchservice.searchlibrary.rsclient.elements.RSLocator;

import gr.uoa.di.madgik.grs.bridge.exceptions.GCubeBridgeException;
import gr.uoa.di.madgik.grs.buffer.IBuffer;

/**
 * Base abstract class that implements the upgrade procedure from an "old style" RewsultSet to a new
 * gRS2 {@link IBuffer}. The class defines hooks that need to be implemented by extenders that will 
 * handle with the specific type of ResultSet type
 * 
 * @author gpapanikos
 *
 */
public abstract class BridgeUpgrade extends Thread
{
	/**
	 * The locator of the ResultSet input that needs to be upgraded
	 */
	protected RSLocator locator=null;
	/**
	 * The type of records that are moved with the ResultSet input
	 */
	protected Class<ResultElementBase> recordInputClass=null;
	/**
	 * The type of records to be produced in the output
	 */
	protected Class<GCubeRecord> recordOutputClass=null;
	
	/**
	 * Perform the upgrade. After all the configuration parameters are set, this is the only method the client
	 * needs to invoke. The operation takes pace in the background
	 * 
	 * @return The locator URI of the created gRS2 {@link IBuffer}
	 * @throws GCubeBridgeException The upgrade could not be completed
	 */
	public URI upgrade() throws GCubeBridgeException
	{
		URI locator=null;
		this.initialize();
		locator=this.getLocator();
		this.setName("gRS2 bridge upgrade");
		this.setDaemon(true);
		this.start();
		return locator;
	}
	
	/**
	 * Sets the RSLocator pointing to the input ResultSet
	 * 
	 * @param locator the locator
	 */
	public void setLocator(RSLocator locator)
	{
		this.locator=locator;
	}
	
	/**
	 * Sets the type of records that are moved with the ResultSet input
	 * 
	 * @param recordClass the type of record input
	 */
	public void setRecordInputClass(Class<ResultElementBase> recordClass)
	{
		this.recordInputClass=recordClass;
	}
	
	/**
	 * Sets the type of records to be produced in the output
	 * 
	 * @param recordClass the type of record output
	 */
	public void setRecordOutputClass(Class<GCubeRecord> recordClass)
	{
		this.recordOutputClass=recordClass;
	}
	
	/**
	 * Instantiates the appropriate output record and populates it through the provided record
	 * 
	 * @param record the record to upgrade
	 * @return the new upgraded record
	 * @throws GCubeBridgeException The record upgrade was not possible 
	 */
	protected GCubeRecord getRecord(ResultElementBase record) throws GCubeBridgeException
	{
		GCubeRecord rec;
		try
		{
			rec = this.recordOutputClass.newInstance();
			rec.upgrade(record.RS_toXML());
			return rec;
		} catch (Exception e)
		{
			throw new GCubeBridgeException("Could not retrieve the downgraded version of the record", e);
		}
	}
	
	/**
	 * Iterate over the input, produce the output and store it in the output writer
	 * 
	 * @throws GCubeBridgeException the upgrade could not be completed
	 */
	public abstract void doUpgrade() throws GCubeBridgeException;
	
	/**
	 * Perform any initialization needed before the upgrade begins
	 * 
	 * @throws GCubeBridgeException the initialization could not be completed
	 */
	public abstract void initialize() throws GCubeBridgeException;
	
	/**
	 * Retrieves the locator of the create {@link IBuffer}
	 * 
	 * @return the locator URI
	 * @throws GCubeBridgeException the locator could not be retrieved
	 */
	public abstract URI getLocator() throws GCubeBridgeException;
	
	/**
	 * The logger defined by the extending class to use the same logger
	 * 
	 * @return the logger
	 */
	public abstract Logger getLogger();
	
	public void run()
	{
		try
		{
			this.doUpgrade();
		}catch(Exception ex)
		{
			if(this.getLogger().isLoggable(Level.WARNING)) this.getLogger().log(Level.WARNING,"Could not complete the upgrade procedure",ex);
		}
	}

}
