package org.gcube.elasticsearch.helpers;

import static org.elasticsearch.common.xcontent.XContentFactory.jsonBuilder;
import static org.elasticsearch.index.query.QueryBuilders.termQuery;

import java.io.IOException;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;

import org.elasticsearch.ElasticSearchException;
import org.elasticsearch.action.admin.cluster.state.ClusterStateRequest;
import org.elasticsearch.action.admin.indices.delete.DeleteIndexRequest;
import org.elasticsearch.action.admin.indices.delete.DeleteIndexResponse;
import org.elasticsearch.action.admin.indices.flush.FlushResponse;
import org.elasticsearch.action.admin.indices.refresh.RefreshResponse;
import org.elasticsearch.action.bulk.BulkRequestBuilder;
import org.elasticsearch.action.count.CountRequestBuilder;
import org.elasticsearch.action.count.CountResponse;
import org.elasticsearch.action.deletebyquery.DeleteByQueryRequestBuilder;
import org.elasticsearch.action.deletebyquery.DeleteByQueryResponse;
import org.elasticsearch.action.index.IndexRequestBuilder;
import org.elasticsearch.action.index.IndexResponse;
import org.elasticsearch.action.search.SearchRequestBuilder;
import org.elasticsearch.action.search.SearchResponse;
import org.elasticsearch.action.search.SearchType;
import org.elasticsearch.client.Client;
import org.elasticsearch.cluster.metadata.IndexMetaData;
import org.elasticsearch.common.collect.ImmutableMap;
import org.elasticsearch.common.unit.TimeValue;
import org.elasticsearch.common.xcontent.XContentBuilder;
import org.elasticsearch.index.query.QueryBuilders;
import org.elasticsearch.search.SearchHit;
import org.gcube.indexmanagement.common.FullTextIndexType;
import org.gcube.indexmanagement.common.IndexField;
import org.gcube.indexmanagement.common.IndexType;
import org.gcube.indexmanagement.common.XMLProfileParser;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;


public class ElasticSearchHelper {
	private static final Logger logger = LoggerFactory.getLogger(ElasticSearchHelper.class);

	private static long SCROLL_TIMEOUT = 100000;
	private static int MAX_RESULTS = 10000;
	private static final List<String> NOT_HIGHLIGHTED_FIELDS = Arrays.asList(IndexType.COLLECTION_FIELD,
			IndexType.DOCID_FIELD, IndexType.LANGUAGE_FIELD);
	
	private static final String HIGHLIGHT_PRE_TAG = "&lt;b&gt;";
	private static final String HIGHLIGHT_POST_TAG = "&lt;/b&gt;";
	
	
//	@SuppressWarnings("unused")
//	private static Analyzer NoStopAnalyser = new StandardAnalyzer(Version.LUCENE_CURRENT);

	public static void commit(Client client, String indexName) {
		try
		{
			logger.info("flush request: " + indexName);
			long before = System.currentTimeMillis();
			FlushResponse flushResponse = client.admin().indices().prepareFlush(indexName).execute().actionGet();
			long after = System.currentTimeMillis();
			logger.info("Time for the flush request : " + (after - before) / 1000.0 + " secs");
			logger.info("flush response  failed shards: " + flushResponse.getFailedShards());
			logger.info("refresh request : " + indexName);
			before = System.currentTimeMillis();
			RefreshResponse refreshResponse = client.admin().indices().prepareRefresh(indexName).execute().actionGet();
			after = System.currentTimeMillis();
			logger.info("Time for the flush request : " + (after - before) / 1000.0 + " secs");
			logger.info("refresh response failed shards : " + refreshResponse.getFailedShards());
		}
		catch(Exception e)
		{
			logger.error("Exception while commiting:",e);
		}
	}

	

	public static SearchHit[] queryElasticSearch(Client client, String indexName, String queryString, int maxHits) {
		return queryElasticSearch(client, indexName, queryString, maxHits, null, 0, 0);
	}

	public static SearchHit[] queryElasticSearch(Client client, String indexName, String queryString, int maxHits,
			List<String> highlightedFields, int maxFragmentSize, int maxFragmentCnt) {

		queryString = "{\"query_string\" : { \"query\" : \"" + queryString + "\" }}";

		SearchRequestBuilder srb = client.prepareSearch(indexName)
				.setSearchType(SearchType.DFS_QUERY_THEN_FETCH)
				.setSize(MAX_RESULTS)
				.setQuery(queryString);

		if (highlightedFields != null) {
			for (String hilightedField : highlightedFields)
				if (!NOT_HIGHLIGHTED_FIELDS.contains(hilightedField))
					srb.addHighlightedField(hilightedField, maxFragmentSize, maxFragmentCnt);
			srb.setHighlighterOrder("score");
			srb.setHighlighterPreTags(HIGHLIGHT_PRE_TAG);
			srb.setHighlighterPostTags(HIGHLIGHT_POST_TAG);
		}

		if (maxHits > 0)
			srb.setSize(maxHits);

		logger.info("query request : " + srb.toString());
		SearchResponse response = srb.execute().actionGet();
		logger.info("query response : " + response);

		return response.getHits().getHits();
	}

	public static SearchResponse queryElasticSearchScroll(Client client, String indexName, String queryString,
			int maxHits) {
		return queryElasticSearchScroll(client, indexName, queryString, maxHits, null, 0, 0);
	}

	/* TODO: fix score problem */
	public static SearchResponse queryElasticSearchScroll(Client client, String indexName, String queryString,
			int maxHits, List<String> highlightedFields, int maxFragmentSize, int maxFragmentCnt) {

		queryString = "{\"query_string\" : { \"query\" : \"" + queryString + "\" }}";

		SearchRequestBuilder srb = client.prepareSearch(indexName).setTrackScores(true)
				// .setExplain(true)
				.setSearchType(SearchType.DFS_QUERY_THEN_FETCH).setScroll(TimeValue.timeValueMillis(SCROLL_TIMEOUT))
				.setQuery(queryString);

		if (maxHits > 0)
			srb.setSize(maxHits);

		if (highlightedFields != null) {
			for (String hilightedField : highlightedFields)
				if (!NOT_HIGHLIGHTED_FIELDS.contains(hilightedField))
					srb.addHighlightedField(hilightedField, maxFragmentSize, maxFragmentCnt);
			srb.setHighlighterOrder("score");
			
			srb.setHighlighterPreTags(HIGHLIGHT_PRE_TAG);
			srb.setHighlighterPostTags(HIGHLIGHT_POST_TAG);
		}

		logger.info("query request : " + srb.toString());
		SearchResponse scrollResponse = srb.execute().actionGet();
		logger.info("query response : " + scrollResponse);

		return scrollResponse;
	}

	public static long queryCountElasticSearch(Client client, String indexName, String queryString) {

		queryString = "{\"query_string\" : { \"query\" : \"" + queryString + "\" }}";

		CountRequestBuilder countRequest = client.prepareCount(indexName).setQuery(queryString.getBytes());

		logger.info("query request : " + countRequest.request());
		CountResponse countResponse = countRequest.execute().actionGet();
		logger.info("query response : " + countResponse);

		return countResponse.getCount();
	}

	public static SearchResponse getNextSearchResponse(Client client, SearchResponse scrollResponse) {
		scrollResponse = client.prepareSearchScroll(scrollResponse.getScrollId())
				.setScroll(TimeValue.timeValueMillis(SCROLL_TIMEOUT)).execute().actionGet();

		logger.info("scroll response : " + scrollResponse);

		return scrollResponse;
	}
	
	public static boolean delete(Client client, String indexName) {
		logger.info("deleting index : " + indexName);
		
		ImmutableMap<String, IndexMetaData> map = client.admin().cluster().state(new ClusterStateRequest()).actionGet().getState().getMetaData().getIndices();
		boolean exists = map.containsKey(indexName);
		
		if (!exists) {
			logger.info("index : " + indexName + " does not exist");
			return true;
		} else {
		 
			DeleteIndexResponse delete = client.admin().indices().delete(new DeleteIndexRequest(indexName)).actionGet();
			if (!delete.acknowledged()) {
				logger.error("Index wasn't deleted");
			}
			
			logger.info("deleted index : " + indexName + " ? " + delete.acknowledged());
			
			return delete.acknowledged();
		}
	}

	public static void deleteDocuments(Client client, String indexName, List<String> docIDs) {
		logger.info("docIds to be deleted : " + docIDs);

		for (String docID : docIDs) {
			logger.info("deleting document with " + IndexType.COLLECTION_FIELD + " : " + docID);

			DeleteByQueryRequestBuilder requestBuilder = client.prepareDeleteByQuery(indexName).setQuery(
					termQuery(IndexType.DOCID_FIELD, docID));
			logger.info("delete request : " + requestBuilder.request());
			requestBuilder.execute().actionGet();
		}
		commit(client, indexName);
	}
	
	
	public static Boolean deleteCollection(Client client, String indexName, String collID) {
		logger.info("collId to be deleted : " + collID);

		logger.info("deleting document with " + IndexType.COLLECTION_FIELD + " : " + collID);

		DeleteByQueryRequestBuilder requestBuilder = client.prepareDeleteByQuery(indexName).setQuery(
				termQuery(IndexType.COLLECTION_FIELD, collID));
		logger.info("delete request : " + requestBuilder.request());
		DeleteByQueryResponse response = requestBuilder.execute().actionGet();
		
		commit(client, indexName);
		
		return true;
	}

	public static void clearIndex(Client client, String indexName, String indexTypeName) {
		logger.info("index to be deleted : ");
		DeleteByQueryResponse dr = client.prepareDeleteByQuery(indexName).setQuery(QueryBuilders.matchAllQuery())
				.setTypes(indexTypeName).execute().actionGet();
		logger.info("Delete response : " + dr.toString());

		commit(client, indexName);
	}

	public static void insertSimple(String jsonDoc, Client indexClient, String indexName, String indexType,
			Set<String> allowedIndexTypes) throws ElasticSearchException, IOException {

		if (!allowedIndexTypes.contains(indexType))
			throw new ElasticSearchException("index type : " + indexType + " is not in registered index types : "
					+ allowedIndexTypes);

		IndexResponse response = indexClient.prepareIndex(indexName, indexType).setSource(jsonDoc).execute()
				.actionGet();
		logger.info("indexResponse : " + response);
	}

	public static void insertBulk(BulkRequestBuilder bulkRequest, String jsonDoc, Client indexClient, String indexName,
			String indexType, Set<String> allowedIndexTypes) throws ElasticSearchException, IOException {

		if (!allowedIndexTypes.contains(indexType))
			throw new ElasticSearchException("index type : " + indexType + " is not in registered index types : "
					+ allowedIndexTypes);

		IndexRequestBuilder indexRequest = indexClient.prepareIndex(indexName, indexType).setSource(jsonDoc);
		bulkRequest.add(indexRequest);
		logger.info("indexRequest : " + indexRequest);
	}

	public static int insertRowSet(BulkRequestBuilder bulkRequest, Client client, String indexName,
			FullTextIndexType idxType, Set<String> allowedIndexTypes, String rowsetXML) {
		logger.info("indexName : " + indexName);
		logger.info("idxType : " + idxType.getIndexTypeName());
		logger.info("allowedIndexTypes : " + allowedIndexTypes);
		logger.info("rowsetXML : " + rowsetXML);

		int docCount = 0;
		try {
			XMLProfileParser XMLparser = new XMLProfileParser();
			XMLparser.readString(rowsetXML, null);
			XMLparser.setRootNode("ROWSET");
			String[][] fieldData;
			while (XMLparser.setNextField()) {
				docCount++;
				fieldData = XMLparser.getSubFields();
				if (fieldData != null) {
					Map<String, Object> document = new HashMap<String, Object>();

					// String docContents = "";

					for (IndexField idxTypeField : idxType.getFields()) {
						StringBuilder fieldContentSum = new StringBuilder();

						for (int ii = 0; ii < fieldData[0].length; ii++) {
							String baseName = fieldData[0][ii];
							String fieldContents = fieldData[1][ii];

							if ((idxTypeField.name).equals(baseName) && fieldContents != null) {

								// for the payload field we won't add its
								// contents in the docContents field
								// if
								// (!baseName.equals(IndexType.PAYLOAD_FIELD))
								// docContents += " " + fieldContents;

								fieldContentSum.append(fieldContents);
								fieldContentSum.append(" ");

							} else if (QueryParser.isDescendant(baseName, idxTypeField) && fieldContents != null) {
								fieldContentSum.append(fieldContents);
								fieldContentSum.append(" ");
							}
						}

						if (fieldContentSum.length() > 0) {
							// TODO: store & index

							document.put(idxTypeField.name, fieldContentSum.toString().trim());
							logger.info("added field in document: " + idxTypeField.name + ", " + fieldContentSum);
						}
					}

					// TODO: store & index & tokenize
					// document.put("_contents", docContents);
					//
					// logger.info("added field in document: _contents, "
					// + docContents);
					// TokenStream tokens = NoStopAnalyser.tokenStream(
					// "_contents", new StringReader(docContents));
					//
					// int wordCount = 0;
					// while (tokens.incrementToken())
					// wordCount++;
					//
					// document.put("_wordcount", wordCount);
					//
					// logger.info("added field in document: _wordcount, "
					// + wordCount);

					for (int ii = 0; ii < fieldData[0].length; ii++) {
						if (fieldData[0][ii].equalsIgnoreCase(IndexType.DOCID_FIELD) && fieldData[1][ii] != null) {
							String data = fieldData[1][ii];

							// TODO:store & index
							// Store is done in FullTextNode addIndexType
//							Map<String, String> field = new HashMap<String, String>();
//							field.put("value", data);
//							field.put("store", "yes");
//							field.put("index", "analyzed");

							document.put(fieldData[0][ii].toLowerCase(), data.trim());

							logger.info("added field in document: " + fieldData[0][ii] + ", " + data);
						}
					}
					logger.info("************* INSERT");
					insertBulk(bulkRequest, createJSONObject(document).string(), client, indexName,
							idxType.getIndexTypeName(), allowedIndexTypes);
				}
			}
		} catch (Exception e) {
			logger.warn("Exception while inserting documents", e);
		}

		return docCount;
	}

	public static XContentBuilder createJSONObject(Map<String, ? extends Object> keyValues) throws IOException {
		XContentBuilder xcb = jsonBuilder();

		xcb = xcb.startObject();
		for (Entry<String, ? extends Object> keyvalue : keyValues.entrySet())
			xcb = xcb.field(keyvalue.getKey(), keyvalue.getValue());

		return xcb.endObject();
	}
	
	public static String createKnownHostsString(Map<String,Integer> hosts) {
		boolean first = true;

		StringBuffer strBuf = new StringBuffer();		
		for (Map.Entry<String, Integer> entry : hosts.entrySet()) {
			if (!first) {
				strBuf.append(", ");
				first = false;
			}
			strBuf.append(entry.getKey());
//			strBuf.append(":");
//			strBuf.append(entry.getValue());
		}
		
		return strBuf.toString();
	}
}
