package org.gcube.elasticsearch.helpers;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.gcube.elasticsearch.FTNodeCache;
import org.gcube.indexmanagement.common.FullTextIndexType;
import org.gcube.indexmanagement.common.IndexField;
import org.gcube.indexmanagement.common.IndexType;
import org.gcube.indexmanagement.common.XMLTokenReplacer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import gr.uoa.di.madgik.grs.buffer.GRS2BufferException;
import gr.uoa.di.madgik.grs.record.GRS2RecordDefinitionException;
import gr.uoa.di.madgik.grs.record.Record;
import gr.uoa.di.madgik.grs.record.field.StringField;

public class RowsetParser {
	private static final Logger logger = LoggerFactory.getLogger(RowsetParser.class);

	private static String regexIdxTypeNameRowset = "<ROWSET[^>]*idxType=\"([^\"]*?)\"";
	private static Pattern patternIdxTypeNameRowset = Pattern.compile(regexIdxTypeNameRowset);

	public static String getIdxTypeNameRowset(String indexName, String rowset) {
		return getMatchRegex(patternIdxTypeNameRowset, rowset, indexName);
	}

	private static String regexLangRowset = "<ROWSET[^>]*lang=\"([^\"]*?)\"";
	private static Pattern patternLangRowset = Pattern.compile(regexLangRowset);

	public static String getLangRowset(String indexName, String rowset) {
		return getMatchRegex(patternLangRowset, rowset, indexName);
	}

	private static String regexColIDRowset = "<ROWSET[^>]*colID=\"([^\"]*?)\"";
	private static Pattern patternColIDRowset = Pattern.compile(regexColIDRowset);

	public static String getColIDRowset(String indexName, String rowset) {
		return getMatchRegex(patternColIDRowset, rowset, indexName);
	}

	private static String getMatchRegex(Pattern pattern, String rowset, String indexName) {
		Matcher m = pattern.matcher(rowset);
		String match = null;
		try {
			if (m.find()) {
				match = m.group(1).trim();
				return !match.equals("") ? match : null;
			}
		} catch (Exception e) {
			logger.error(indexName + " exception while getting idxType", e);
		}
		return match;
	}

	private static String regexField = "<FIELD\\s+(?:lang=\"([^\"]*?)\"\\s+)*(?:[^>]*?\\s+)??name=\"([^\"]*?)\"(?:\\s+lang=\"([^\"]*?)\"\\s*)*\\s*>\\s*(.*?)\\s*</";
	private static Pattern patternField = Pattern.compile(regexField);

	/* adds a language aware fields to the ROWSET */
	public static String preprocessRowset(String rowset, String lang, String colID, String indexName, String IdxTypeID,
			String scope, FTNodeCache cache) {

		// xml unresolving for the fullpayload field
		int payloadIndex = rowset.indexOf(IndexType.PAYLOAD_FIELD);
		if (payloadIndex > -1) {
			logger.trace("Found payload field");
			String payload = rowset.substring(payloadIndex + IndexType.PAYLOAD_FIELD.length()).trim();
			if (payload.charAt(0) != '"') {
				logger.warn("please check the rowsets to be fed in the index. While there is a "
						+ IndexType.PAYLOAD_FIELD + " keyword, it doesn't seem to be a FIELD");

			} else {
				int payloadStart = rowset.indexOf('>', payloadIndex);
				int testInsideElement = rowset.indexOf('<', payloadIndex);
				if (payloadStart == -1) {
					logger.warn("please check the rowsets to be fed in the index. While there is a "
							+ IndexType.PAYLOAD_FIELD + " keyword, it seems to be in a weird spot");

				} else if (testInsideElement != -1 && payloadStart > testInsideElement) {
					logger.warn("please check the rowsets to be fed in the index. While there is a "
							+ IndexType.PAYLOAD_FIELD + " keyword, it doesn't seem to be INSIDE a FIELD");

				} else {
					int payloadEnd = rowset.indexOf("</FIELD>", payloadStart);
					if (payloadEnd == -1) {
						logger.warn("please check the rowsets to be fed in the index. The" + IndexType.PAYLOAD_FIELD
								+ " keyword, must be out of FIELD");
					}
					payload = rowset.substring(payloadStart + 1, payloadEnd);
					payload = XMLTokenReplacer.XMLUnresolve(payload);
					rowset = rowset.substring(0, payloadStart + 1) + payload + rowset.substring(payloadEnd);
				}
			}
		}

		StringBuilder alteredRowSet = new StringBuilder("<ROWSET>\n");
		boolean first = true;
		for (String row : rowset.split("<ROW>")) {

			// the first time we will have the payload before the first ROW.
			// continue
			if (first) {
				first = false;
				continue;
			}

			// in this regex the attribute lang is also grouped, but it isn't
			// used anymore

			Matcher m = patternField.matcher(row);

			HashMap<String, String> fields = new HashMap<String, String>();
			try {
				while (m.find()) {
					String fieldName = m.group(2).trim();
					String fieldContent = m.group(4).trim();

					// put in language neutral field
					String sumContent = fields.get(fieldName);
					if (sumContent == null) {
						sumContent = fieldContent;
					} else {
						sumContent += (" " + fieldContent);
					}
					fields.put(fieldName, sumContent);
					logger.trace("putting in fieldName: " + fieldName + ", new content: " + fieldContent);

				}
			} catch (Exception e) {
				logger.error(indexName + " exception while adding language fields", e);
			}

			logger.trace("Fields keySet: " + fields.keySet().size());
			
			
			//ArrayList<String> toBeAdded = new ArrayList<String>();
			//TODO: add all fields for this collection
			
			
			//put a field for collection ID
			String content = null;
			if((content = fields.get(IndexType.COLLECTION_FIELD)) != null){
				logger.error("-- rowset has already a value: " + content + "for the reserved field " + IndexType.COLLECTION_FIELD + ", that will be dropped.");
			}
			fields.put(IndexType.COLLECTION_FIELD, colID);
			
			//TODO: add collection fields for this collection

			//put a field for language
			content = null;
			if((content = fields.get(IndexType.LANGUAGE_FIELD)) != null){
				logger.error("-- rowset has already a value: " + content + "for the reserved field " + IndexType.LANGUAGE_FIELD + ", that will be dropped.");
			}
			fields.put(IndexType.LANGUAGE_FIELD, lang);
			//TODO: add language field for this collection

			if (!fields.isEmpty()) {
				alteredRowSet.append("   <ROW>\n");
				for (String field : fields.keySet()) {
					alteredRowSet.append("      <FIELD name=\"");
					alteredRowSet.append(field);
					alteredRowSet.append("\">");
					alteredRowSet.append(fields.get(field).trim());
					alteredRowSet.append("</FIELD>\n");
				}
				alteredRowSet.append("   </ROW>\n");
			}
		}
		alteredRowSet.append("</ROWSET>\n");
		return alteredRowSet.toString();
	}

	public static boolean addToFieldInfo(ArrayList<String> toBeAdded, String field, String colIDandLang,
			FullTextIndexType idxType) {
		boolean found = false;
		boolean searchable = false;
		boolean presentable = false;
		// search all the fields defined in the indexType

		// TODO: uncomment when IS communication is ready

		// TODO: change it to get idxType at the beggining of the feeding
		// FullTextIndexType idxType = retrieveIndexType(indexType, scope,
		// cache);
		for (IndexField idxField : idxType.getFields()) {
			// if we find the field in the indexType
			if (idxField.name.equals(field)) {
				// tag the searchable and presentable info
				found = true;
				searchable = idxField.index;
				presentable = idxField.returned;
				break;
			}
		}

		if (field.equalsIgnoreCase(IndexType.DOCID_FIELD)) {
			searchable = true;
			// it is not presentable because it is a default attribute in each
			// result
			presentable = false;
		} else if (!found) {
			logger.error("The field: " + field + ", is not declared in the indexType.");
			return false;
		}
		if (searchable)
			toBeAdded.add(colIDandLang + IndexType.SEPERATOR_FIELD_INFO + IndexType.SEARCHABLE_TAG
					+ IndexType.SEPERATOR_FIELD_INFO + field);
		if (presentable)
			toBeAdded.add(colIDandLang + IndexType.SEPERATOR_FIELD_INFO + IndexType.PRESENTABLE_TAG
					+ IndexType.SEPERATOR_FIELD_INFO + field);
		return true;
	}

	private static final String ROWSETFIELD = "Rowset";

	public static String getRowsetFromResult(Record record) throws GRS2RecordDefinitionException, GRS2BufferException {
		return ((StringField) record.getField(ROWSETFIELD)).getPayload();
	}

}
