package org.gcube.rest.index.service.resources;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import org.gcube.elasticsearch.FullTextNode;
import org.gcube.rest.commons.resourceawareservice.resources.ResourceFactory;
import org.gcube.rest.commons.resourceawareservice.resources.exceptions.StatefulResourceException;
import org.gcube.rest.index.common.resources.IndexResource;
import org.gcube.rest.index.service.IndexClientWrapper;
import org.gcube.rest.resourceawareservice.exceptions.ResourceNotFoundException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.gson.Gson;
import com.google.inject.Inject;
import com.google.inject.Provider;


public class IndexResourceFactory extends ResourceFactory<IndexResource> {

	private static final Logger logger = LoggerFactory.getLogger(IndexResourceFactory.class);
	
	private Map<String, IndexClientWrapper> indexClientWrappers = new ConcurrentHashMap<String, IndexClientWrapper>();
	
	private Provider<IndexClientWrapper> ftnClientProvider;
	
	@Inject
	public IndexResourceFactory(Provider<IndexClientWrapper> ftnClientProvider) {
		this.ftnClientProvider = ftnClientProvider;
	}
	
	
	public FullTextNode getIndexNode(IndexResource resource) throws ResourceNotFoundException {
		if (!this.indexClientWrappers.containsKey(resource.getResourceID()))
			throw new ResourceNotFoundException("resource with id : " + resource.getResourceID() + " not in map factory : " + this.indexClientWrappers.keySet());
		return this.indexClientWrappers.get(resource.getResourceID()).getFullTextNode();
	}
	
	@Override
	public IndexResource createResource(String resourceID, String params) throws StatefulResourceException {
		logger.info("IndexResource createResource");
		
		IndexResource resource = new Gson().fromJson(params, IndexResource.class);
		
		if (resource.getScope() != null && resource.getScope().equalsIgnoreCase(this.getScope()) == false){
			logger.error("scope set to : " + resource.getScope() + " but different to : " + this.getScope());
			throw new StatefulResourceException("scope set to : " + resource.getScope() + " but different to : " + this.getScope());
		}
		
		resource.setResourceID(resourceID);
		
		logger.info("IndexClientWrapper initializing");

		IndexClientWrapper clientWrapper = this.ftnClientProvider.get();
		initICWToResource(clientWrapper,resource);
		this.indexClientWrappers.put(resourceID, clientWrapper);
		
		
		resource.setHostname(clientWrapper.getFullTextNode().getHostname());
		logger.info("hostname set to resource : " + resource.getHostname());
		
		return resource;
	}
	
	@Override 
	public void loadResource(IndexResource resource) throws StatefulResourceException{
		logger.info("IndexResource loadResource");
		
		if (resource.getScope() != null && resource.getScope().equalsIgnoreCase(this.getScope()) == false){
			logger.error("scope set to : " + resource.getScope() + " but different to : " + this.getScope());
			throw new StatefulResourceException("scope set to : " + resource.getScope() + " but different to : " + this.getScope());
		}
		
		logger.info("IndexClientWrapper loading");
		IndexClientWrapper clientWrapper = this.ftnClientProvider.get();
		loadICWToResource(clientWrapper,resource);
		
		resource.setHostname(clientWrapper.getFullTextNode().getHostname());
		logger.info("hostname set to resource : " + resource.getHostname());
		
		resource.onLoad();
		this.indexClientWrappers.put(resource.getResourceID(), clientWrapper);
	}
	
	@Override
	public void closeResource(IndexResource resource)
			throws StatefulResourceException {
		IndexClientWrapper icw = this.indexClientWrappers.get(resource.getResourceID());
		logger.info("Closing index...");
		icw.getFullTextNode().close();
		logger.info("Closing index...OK");
		this.indexClientWrappers.remove(resource.getResourceID());
	}
	
	@Override
	public void destroyResource(IndexResource resource)
			throws StatefulResourceException {
		IndexClientWrapper icw = this.indexClientWrappers.get(resource.getResourceID());
		
		try {
			logger.info("Deleting index");
			icw.getFullTextNode().deleteIndex();
		} catch (Exception e) {
			logger.error("Error while deleting the index. Maybe it does not exist", e);
			throw new StatefulResourceException("Error while deleting the index. Maybe it does not exist", e);
		} finally {
			this.closeResource(resource);
		}
	}
	
	
	
	private static void initICWToResource(IndexClientWrapper indexClientWrapper, IndexResource resource) throws StatefulResourceException {
    	if(resource.getIndexID()==null || resource.getIndexID().trim().length()==0) {
    		logger.info("No indexID given, assigning a new one: " + resource.getResourceID());
    		resource.setIndexID(resource.getResourceID());
    	}
    	
    	if(!resource.getUseClusterID())
    		resource.setClusterID(resource.getScope());
    	logger.info("clusterID 1 : " + resource.getClusterID());
    	
    	// if no clusterID given, assign indexID as clusterID
		if(resource.getClusterID()==null)
			resource.setClusterID(resource.getIndexID());
    	
		logger.info("clusterID 2 : " + resource.getClusterID());
		
		logger.info("key : " + resource.getClusterID());
		logger.info("resource key : " + resource.getResourceID());
		
		try {
			logger.info("initializing FullTextNodeClient : ");
			indexClientWrapper.setClusterID(resource.getClusterID());
			indexClientWrapper.initialize();
		} catch (Exception e) {
			throw new StatefulResourceException("error while initializing the fulltext index client" , e);
		}
		
		String transportAddress = indexClientWrapper.getFullTextNode().getESTransportAddress();
		resource.setEsTransportAddress(transportAddress);
		
		logger.info("getting meta index values for existing index");
		List<String> collectionsOfMetaIndex = indexClientWrapper.getFullTextNode().getCollectionOfIndex();
		List<String> fieldsOfMetaIndex = indexClientWrapper.getFullTextNode().getFieldsOfIndex();
		
		
		if (collectionsOfMetaIndex != null && fieldsOfMetaIndex != null) {
			resource.setCollections(collectionsOfMetaIndex);
			resource.setFields(fieldsOfMetaIndex);
		} else {
			resource.setCollections(new ArrayList<String>());
			resource.setFields(new ArrayList<String>());
		}

		resource.setSupportedRelations(IndexResource.getSupportedRelationsSet());
		
    }
	

	private static void loadICWToResource(IndexClientWrapper indexClientWrapper, IndexResource resource) throws StatefulResourceException {

		if(!resource.getUseClusterID())
			resource.setClusterID(resource.getScope());
		logger.info("clusterID 1 : " + resource.getClusterID());
    	
    	// if no clusterID given, assign indexID as clusterID
		if(resource.getClusterID()==null)
			resource.setClusterID(resource.getIndexID());
    	
		logger.info("clusterID 2 : " + resource.getClusterID());
		
		logger.info("resource key : " + resource.getResourceID());
		
		try {
			System.out.println("initializing FullTextNodeClient : ");
			indexClientWrapper.setClusterID(resource.getClusterID());
			indexClientWrapper.initialize();
		} catch (Exception e) {
			throw new StatefulResourceException("error while initializing the fulltext index client" , e);
		}
		
		
		String transportAddress = indexClientWrapper.getFullTextNode().getESTransportAddress();
		resource.setEsTransportAddress(transportAddress);
		
		logger.info("getting meta index values for existing index");
		List<String> collectionsOfMetaIndex = indexClientWrapper.getFullTextNode().getCollectionOfIndex();
		List<String> fieldsOfMetaIndex = indexClientWrapper.getFullTextNode().getFieldsOfIndex();
		
		
		if (collectionsOfMetaIndex != null && fieldsOfMetaIndex != null) {
			resource.setCollections(collectionsOfMetaIndex);
			resource.setFields(fieldsOfMetaIndex);
		} else {
			resource.setCollections(new ArrayList<String>());
			resource.setFields(new ArrayList<String>());
		}

		resource.setSupportedRelations(IndexResource.getSupportedRelationsSet());
    }
	
}
